﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nd/detail/nd_DataConverter.h>
#include <nn/nd/detail/nd_Log.h>

namespace nn { namespace nd { namespace detail {

void MakeSendDataDescription(
    ndd::SendDataDescription* pOut,
    const ndd::DataId& dataId,
    const settings::LanguageCode& languageCode,
    const LocalUserId& localUserId,
    const util::optional<NetworkUserId>& networkUserId,
    const account::Nickname& nickName,
    int recentlyPlayedApplicationCount,
    const ncm::ApplicationId recentlyPlayedApplication[],
    size_t systemDataSize,
    const Bit8 systemData[]
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(pOut->data) % std::alignment_of<std::max_align_t>::value == 0);

    pOut->dataId = dataId;

    auto pPayload = reinterpret_cast<Payload*>(pOut->data);
    pPayload->version = PayloadVersion;
    pPayload->languageCode = languageCode;
    pPayload->localUserId = localUserId;
    if( networkUserId )
    {
        pPayload->hasNetworkUserId = true;
        pPayload->networkUserId = *networkUserId;
    }
    else
    {
        pPayload->hasNetworkUserId = false;
    }
    pPayload->nickName = nickName;
    pPayload->recentlyPlayedApplicationCount = std::min(recentlyPlayedApplicationCount, static_cast<int>(NN_ARRAY_SIZE(pPayload->recentlyPlayedApplication)));
    std::memcpy(pPayload->recentlyPlayedApplication, recentlyPlayedApplication, sizeof(pPayload->recentlyPlayedApplication[0]) * pPayload->recentlyPlayedApplicationCount);
    pPayload->systemDataSize = std::min(systemDataSize, sizeof(pPayload->systemData));
    std::memcpy(pPayload->systemData, systemData, systemDataSize);
    pOut->dataSize = sizeof(Payload) - SystemDataSizeMax + pPayload->systemDataSize;
}

bool ParseReceiveDataDescription(
    nd::NeighborInfoForSystem* pOut,
    const ndd::ReceiveDataDescription& data
) NN_NOEXCEPT
{
    Payload p;
    std::memcpy(&p, data.data, data.dataSize);
    if( p.version > PayloadVersion )
    {
        NN_DETAIL_ND_TRACE("[nd] ParseReceiveDataDescription : Input data's version(%u) is larger than this firmware's PayloadVersion(%u). Ignore.\n", p.version, PayloadVersion);
        return false;
    }

    pOut->receivedTimePoint = data.receiveTimePoint;
    pOut->systemLanguageCode = p.languageCode;
    pOut->localUserId = p.localUserId;
    pOut->networkUserId = p.hasNetworkUserId ? p.networkUserId : util::optional<NetworkUserId>();
    pOut->nickName = p.nickName;
    pOut->recentlyPlayedApplicationCount = std::min(p.recentlyPlayedApplicationCount, static_cast<int>(NN_ARRAY_SIZE(pOut->recentlyPlayedApplication)));
    std::memcpy(pOut->recentlyPlayedApplication, p.recentlyPlayedApplication, sizeof(pOut->recentlyPlayedApplication[0]) * pOut->recentlyPlayedApplicationCount);
    pOut->systemDataSize = std::min(p.systemDataSize, sizeof(pOut->systemData));
    std::memcpy(pOut->systemData, p.systemData, pOut->systemDataSize);
    return true;
}

void MakeReceiveDataDescription(
    ndd::ReceiveDataDescription* pOut,
    const nd::NeighborInfoForSystem& info
) NN_NOEXCEPT
{
    Payload p;
    p.version = PayloadVersion;
    p.languageCode = info.systemLanguageCode;
    p.localUserId = info.localUserId;
    if( info.networkUserId )
    {
        p.hasNetworkUserId = true;
        p.networkUserId = *info.networkUserId;
    }
    else
    {
        p.hasNetworkUserId = false;
    }
    p.nickName = info.nickName;
    p.recentlyPlayedApplicationCount = std::min(info.recentlyPlayedApplicationCount, static_cast<int>(NN_ARRAY_SIZE(p.recentlyPlayedApplication)));
    std::memcpy(p.recentlyPlayedApplication, info.recentlyPlayedApplication, sizeof(p.recentlyPlayedApplication[0]) * p.recentlyPlayedApplicationCount);
    p.systemDataSize = std::min(info.systemDataSize, sizeof(p.systemData));
    std::memcpy(p.systemData, info.systemData, p.systemDataSize);

    std::memcpy(pOut->data, &p, sizeof(p));
    pOut->dataSize = sizeof(p);
    pOut->receiveTimePoint = info.receivedTimePoint;
}

}}} // ~namespace nn::nd::detail
