﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_Cancellable.h>
#include <nn/migration/detail/migration_Log.h>
#include <nn/migration/idc/migration_SocketConnection.h>
#include <nn/migration/idc/migration_Result.h>
#include <nn/migration/idc/detail/migration_HandleSocketError.h>
#include <nn/migration/idc/detail/migration_Util.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nn/socket/socket_Errno.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace migration { namespace idc {

void swap(SocketConnection& o0, SocketConnection& o1) NN_NOEXCEPT
{
    std::swap(o0.m_SocketDescriptor, o1.m_SocketDescriptor);
}

SocketConnection::SocketConnection() NN_NOEXCEPT
    : m_SocketDescriptor(-1)
{
}

SocketConnection::SocketConnection(int socketDescriptor) NN_NOEXCEPT
    : m_SocketDescriptor(socketDescriptor)
{
    NN_SDK_REQUIRES_GREATER_EQUAL(m_SocketDescriptor, 0);
    NN_ABORT_UNLESS_EQUAL(socket::Fcntl(m_SocketDescriptor, nn::socket::FcntlCommand::F_SetFl, socket::Fcntl(m_SocketDescriptor, nn::socket::FcntlCommand::F_GetFl, 0) | static_cast<int>(nn::socket::FcntlFlag::O_NonBlock)), 0);
}

SocketConnection::SocketConnection(SocketConnection&& rhs) NN_NOEXCEPT
    : m_SocketDescriptor(rhs.m_SocketDescriptor)
{
    rhs.m_SocketDescriptor = -1;
}

SocketConnection::~SocketConnection() NN_NOEXCEPT
{
    if( m_SocketDescriptor >= 0 )
    {
        Close();
    }
}

SocketConnection& SocketConnection::operator =(SocketConnection&& rhs) NN_NOEXCEPT
{
    SocketConnection tmp(std::move(rhs));
    swap(*this, tmp);
    return *this;
}

Result SocketConnection::Send(
    size_t* pOutSentSize, const void* stream, size_t size, int timeoutSeconds,
    const migration::detail::Cancellable* pCancellable,
    detail::TransferSpeedMonitor* pSpeedMonitor) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(m_SocketDescriptor, 0);
    NN_SDK_REQUIRES_NOT_NULL(pOutSentSize);
    NN_SDK_REQUIRES_NOT_NULL(stream);
    NN_SDK_REQUIRES_GREATER(size, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(timeoutSeconds, 0);

    NN_RESULT_THROW_UNLESS(!migration::detail::IsCanceled(pCancellable), idc::ResultCanceled());

    const int PollIntervalMs = 10;

    os::TimerEvent timeoutTimer(os::EventClearMode_AutoClear);
    if( timeoutSeconds > 0 )
    {
        timeoutTimer.StartOneShot(nn::TimeSpan::FromSeconds(timeoutSeconds));
    }

    if( pSpeedMonitor != nullptr )
    {
        pSpeedMonitor->Resume();
    }
    NN_UTIL_SCOPE_EXIT
    {
        if( pSpeedMonitor != nullptr )
        {
            pSpeedMonitor->Pause();
        }
    };

    auto pollFd = socket::PollFd{ m_SocketDescriptor, nn::socket::PollEvent::PollOut, nn::socket::PollEvent::PollNone };

    int pollResult = 0;
    while( !((pollResult = socket::Poll(&pollFd, 1, PollIntervalMs)) >= 1) )
    {
        NN_RESULT_THROW_UNLESS(!migration::detail::IsCanceled(pCancellable), idc::ResultCanceled());
        NN_RESULT_THROW_UNLESS(!timeoutTimer.TryWait(), idc::ResultTimeout());
        NN_RESULT_THROW_UNLESS(pollResult >= 0, detail::HandleSocketError(socket::GetLastError()));
        if( pSpeedMonitor != nullptr )
        {
            pSpeedMonitor->Update(0u);
            NN_RESULT_THROW_UNLESS(!pSpeedMonitor->IsLowLimitDetected(), idc::ResultTimeout());
        }
    }
    NN_SDK_ASSERT_EQUAL(pollResult, 1);

    auto sentSize = socket::Send(m_SocketDescriptor, stream, size, nn::socket::MsgFlag::Msg_None);
    if( sentSize < 0 )
    {
        NN_SDK_ASSERT(sentSize == -1);
        auto socketError = socket::GetLastError();
#if defined (NN_BUILD_CONFIG_OS_WIN32)
        // ノンブロッキングの場合、相手がクローズしていると Send は返り値 -1 でエラー無し。
        if( socketError == nn::socket::Errno::ESuccess )
        {
            NN_RESULT_THROW(idc::ResultPeerClosed());
        }
        NN_RESULT_THROW(detail::HandleSocketError(socketError));
#else
        NN_RESULT_THROW(detail::HandleSocketError(socketError));
#endif
    }
    if( pSpeedMonitor != nullptr )
    {
        pSpeedMonitor->Update(sentSize);
        NN_RESULT_THROW_UNLESS(!pSpeedMonitor->IsLowLimitDetected(), idc::ResultTimeout());
    }
    *pOutSentSize = static_cast<size_t>(sentSize);
    NN_RESULT_SUCCESS;
}

Result SocketConnection::Receive(
    size_t *pOutReceivedSize, void* outStream, size_t outStreamSize, int timeoutSeconds,
    const migration::detail::Cancellable* pCancellable,
    detail::TransferSpeedMonitor* pSpeedMonitor) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(m_SocketDescriptor, 0);
    NN_SDK_REQUIRES_NOT_NULL(pOutReceivedSize);
    NN_SDK_REQUIRES_NOT_NULL(outStream);
    NN_SDK_REQUIRES_GREATER(outStreamSize, 0u);
    NN_SDK_REQUIRES_GREATER_EQUAL(timeoutSeconds, 0);

    NN_RESULT_THROW_UNLESS(!migration::detail::IsCanceled(pCancellable), idc::ResultCanceled());

    const int PollIntervalMs = 10;

    os::TimerEvent timeoutTimer(os::EventClearMode_AutoClear);
    if( timeoutSeconds > 0 )
    {
        timeoutTimer.StartOneShot(nn::TimeSpan::FromSeconds(timeoutSeconds));
    }

    if( pSpeedMonitor != nullptr )
    {
        pSpeedMonitor->Resume();
    }
    NN_UTIL_SCOPE_EXIT
    {
        if( pSpeedMonitor != nullptr )
        {
            pSpeedMonitor->Pause();
        }
    };

    auto pollFd = socket::PollFd{ m_SocketDescriptor, socket::PollEvent::PollIn, nn::socket::PollEvent::PollNone };

    int pollResult = 0;
    while( !((pollResult = socket::Poll(&pollFd, 1, PollIntervalMs)) >= 1) )
    {
        NN_RESULT_THROW_UNLESS(!migration::detail::IsCanceled(pCancellable), idc::ResultCanceled());
        NN_RESULT_THROW_UNLESS(!timeoutTimer.TryWait(), idc::ResultTimeout());
        if( pSpeedMonitor != nullptr )
        {
            pSpeedMonitor->Update(0);
            NN_RESULT_THROW_UNLESS(!pSpeedMonitor->IsLowLimitDetected(), idc::ResultTimeout());
        }
        NN_RESULT_THROW_UNLESS(pollResult >= 0, detail::HandleSocketError(socket::GetLastError()));
    }
    NN_SDK_ASSERT_EQUAL(pollResult, 1);

    auto recvCount = socket::Recv(m_SocketDescriptor, outStream, outStreamSize, nn::socket::MsgFlag::Msg_None);
    if( recvCount <= 0 )
    {
        if( recvCount == 0 )
        {
            NN_RESULT_THROW(idc::ResultPeerClosed());
        }
        auto socketError = socket::GetLastError();
        NN_RESULT_THROW(detail::HandleSocketError(socketError));
    }
    if( pSpeedMonitor != nullptr )
    {
        pSpeedMonitor->Update(recvCount);
        NN_RESULT_THROW_UNLESS(!pSpeedMonitor->IsLowLimitDetected(), idc::ResultTimeout());
    }
    *pOutReceivedSize = static_cast<size_t>(recvCount);
    NN_RESULT_SUCCESS;
}

void SocketConnection::Close() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(m_SocketDescriptor, 0);
    if( socket::Close(m_SocketDescriptor) != 0 )
    {
        NN_DETAIL_MIGRATION_TRACE("SocketConnection::Close : socket::Close failed (%d)\n", socket::GetLastError());
    }
    m_SocketDescriptor = -1;
}

}}};
