﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_ThreadResourceAllocator.h>

#include <algorithm>

namespace nn { namespace migration { namespace detail {

const size_t ThreadResourceAllocator::ResourceCountMax;

ThreadResourceAllocator::ThreadResourceAllocator(int priority, const char* name) NN_NOEXCEPT
    : m_Lock(false)
    , m_Priority(priority)
    , m_Name(name)
    , m_Count(0)
{
}
void ThreadResourceAllocator::InitializeImpl(void* buffer, size_t unit, size_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(unit, 0u);
    NN_SDK_REQUIRES_GREATER(count, 0u);
    NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(buffer) % os::MemoryPageSize, 0u);
    NN_SDK_REQUIRES_EQUAL(unit % os::MemoryPageSize, 0u);

    m_Count = std::min(count, ResourceCountMax);
    for (size_t i = 0u; i < m_Count; ++i)
    {
        ThreadResourceType r = {
            reinterpret_cast<char*>(buffer) + i * unit,
            unit,
            m_Priority,
            m_Name,
        };
        m_Resources[i] = r;
    }
}

ThreadResource ThreadResourceAllocator::Allocate() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Lock);
    for (size_t i = 0u; i < m_Count; ++i)
    {
        if (m_Counters[i].Get() == 0)
        {
            return ThreadResource(m_Resources[i], m_Counters[i]);
        }
    }
    NN_ABORT("ThreadResourceAllocator::Allocate()");
}

}}} // ~namespace nn::migration::detail

