﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_SubsystemInitializer.h>

#include <mutex>

#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/pdm/pdm_NotifyEventApi.h>
#include <nn/socket/socket_Api.h>

namespace nn { namespace migration { namespace detail {

// ---------------------------------------------------------------------------------------------
// SubsystemInitializer

Counter SubsystemInitializer::s_Counter;
os::Mutex SubsystemInitializer::s_Lock(false);
bool SubsystemInitializer::s_IsAttached = false;
SubsystemInitializer::SocketBuffer* SubsystemInitializer::s_Buffer = nullptr;

void SubsystemInitializer::AttachSocketBuffer(SocketBuffer& buffer) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_EQUAL(reinterpret_cast<uintptr_t>(&buffer) % std::alignment_of<SocketBuffer>::value, 0u);

    std::lock_guard<os::Mutex> lock(s_Lock);
    NN_ABORT_UNLESS(!s_IsAttached);
    s_IsAttached = true;
    s_Buffer = &buffer;
}
UniqueResource<SubsystemInitializer::SocketBuffer*> SubsystemInitializer::AcquireSocketConfigBuffer() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(s_Lock);
    NN_ABORT_UNLESS(s_IsAttached);
    NN_ABORT_UNLESS(s_Counter.Get() == 0);
    return UniqueResource<SocketBuffer*>(s_Buffer, s_Counter);
}

SubsystemInitializer::SubsystemInitializer() NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(account::ListAllUsers(&m_UidCount, m_Uids, std::extent<decltype(m_Uids)>::value));

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(npns::Suspend());

    for (int i = 0; i < m_UidCount; ++i)
    {
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(pdm::SuspendUserAccountEventService(m_Uids[i]));
    }
#endif
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(friends::SuspendDaemon(&m_FriendsSuspension));
}
SubsystemInitializer::~SubsystemInitializer() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    for (int i = 0; i < m_UidCount; ++i)
    {
        auto r = pdm::ResumeUserAccountEventService(m_Uids[i]);
        if (!r.IsSuccess())
        {
            NN_MIGRATION_DETAIL_WARN("<!> pdm::ResumeUserAccountEventService failed with %03d-%04d (%08lx)\n", r.GetModule(), r.GetDescription(), r.GetInnerValueForDebug());
        }
    }

    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(npns::Resume());
#endif
}

// ---------------------------------------------------------------------------------------------
// ClientSubsystemInitializer

ClientSubsystemInitializer::ClientSubsystemInitializer() NN_NOEXCEPT
    : m_Buffer(SubsystemInitializer::AcquireSocketConfigBuffer())
{
    m_pConfig = new(m_Buffer.Get()) SocketConfigForReceiver(2);
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(socket::Initialize(*m_pConfig));
}
ClientSubsystemInitializer::~ClientSubsystemInitializer() NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(socket::Finalize());
    m_pConfig->~SocketConfigForReceiver();
}

// ---------------------------------------------------------------------------------------------
// ServerSubsystemInitializer

ServerSubsystemInitializer::ServerSubsystemInitializer() NN_NOEXCEPT
    : m_Buffer(SubsystemInitializer::AcquireSocketConfigBuffer())
{
    m_pConfig = new(m_Buffer.Get()) SocketConfigForSender(2);
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(socket::Initialize(*m_pConfig));
}
ServerSubsystemInitializer::~ServerSubsystemInitializer() NN_NOEXCEPT
{
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(socket::Finalize());
    m_pConfig->~SocketConfigForSender();
}

}}} // ~namespace nn::migration::detail
