﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_StoragePolicy.h>

#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/migration/detail/migration_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

namespace nn { namespace migration { namespace detail {

namespace {
const size_t FullPathLengthMax = 64;
typedef char FullPath[FullPathLengthMax];

#define SCOPE_CLOSER(file) \
    NN_UTIL_SCOPE_EXIT { \
        fs::CloseFile(file); \
    }

#define SCOPE_FOPEN(file, path, mode) \
    NN_RESULT_DO(fs::OpenFile(&file, path, mode)); \
    SCOPE_CLOSER(file)

} // ~namespace nn::migration::detail::<anonymous>

SaveDataStorage::SaveDataStorage(const char* volumeName, fs::SystemSaveDataId saveId, int64_t dataSize, int64_t journalSize, uint32_t flags) NN_NOEXCEPT
    : m_SaveId(saveId)
    , m_DataSize(dataSize)
    , m_JournalSize(journalSize)
    , m_Flags(flags)
{
    std::strncpy(m_VolumeName, volumeName, sizeof(m_VolumeName));
    NN_ABORT_UNLESS_LESS(strnlen(m_VolumeName, sizeof(m_VolumeName)), sizeof(m_VolumeName));
    Mount();
}
SaveDataStorage::~SaveDataStorage() NN_NOEXCEPT
{
    Unmount();
}

void SaveDataStorage::Mount() const NN_NOEXCEPT
{
    auto r = fs::CreateSystemSaveData(m_SaveId, m_DataSize, m_JournalSize, m_Flags);
    if (!(r.IsSuccess() || fs::ResultPathAlreadyExists::Includes(r)))
    {
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
    NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(fs::MountSystemSaveData(m_VolumeName, m_SaveId));
}
void SaveDataStorage::Unmount() const NN_NOEXCEPT
{
    fs::Unmount(m_VolumeName);
}

void SaveDataStorage::CleanupImpl() const NN_NOEXCEPT
{
    Unmount();
    NN_UTIL_SCOPE_EXIT
    {
        Mount();
    };
    auto r = fs::DeleteSaveData(m_SaveId);
    if (!(r.IsSuccess() || fs::ResultTargetNotFound::Includes(r)))
    {
        NN_MIGRATION_DETAIL_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
}

template <size_t N>
void SaveDataStorage::GetFullPath(char (&fullPath)[N], const char* path) const NN_NOEXCEPT
{
    auto l = util::SNPrintf(fullPath, sizeof(fullPath), "%s:/%s", m_VolumeName, path);
    NN_ABORT_UNLESS(l < N);
}

Result SaveDataStorage::CreateImpl(const char* path, size_t sizeToReserve) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS_EQUAL(sizeToReserve, static_cast<size_t>(std::numeric_limits<int64_t>::max()));
    FullPath fullPath;
    GetFullPath(fullPath, path);

    return fs::CreateFile(fullPath, static_cast<int64_t>(sizeToReserve));
}
Result SaveDataStorage::DeleteImpl(const char* path) const NN_NOEXCEPT
{
    FullPath fullPath;
    GetFullPath(fullPath, path);

    return fs::DeleteFile(fullPath);
}
Result SaveDataStorage::MoveImpl(const char* to, const char* from) const NN_NOEXCEPT
{
    FullPath fullPathFrom;
    GetFullPath(fullPathFrom, from);
    FullPath fullPathTo;
    GetFullPath(fullPathTo, to);

    return fs::RenameFile(fullPathFrom, fullPathTo);
}
Result SaveDataStorage::GetSizeImpl(size_t* pOutSize, const char* path) const NN_NOEXCEPT
{
    FullPath fullPath;
    GetFullPath(fullPath, path);

    fs::FileHandle file;
    SCOPE_FOPEN(file, fullPath, fs::OpenMode_Read);

    int64_t fileSize;
    NN_RESULT_DO(fs::GetFileSize(&fileSize, file));
    NN_SDK_ASSERT(fileSize >= 0);

    *pOutSize = static_cast<size_t>(fileSize);
    NN_RESULT_SUCCESS;
}

Result SaveDataStorage::CommitImpl() const NN_NOEXCEPT
{
    return fs::CommitSaveData(m_VolumeName);
}
Result SaveDataStorage::WriteImpl(const char* path, const void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS_EQUAL(bufferSize, static_cast<size_t>(std::numeric_limits<int64_t>::max()));
    FullPath fullPath;
    GetFullPath(fullPath, path);

    fs::FileHandle file;
    SCOPE_FOPEN(file, fullPath, fs::OpenMode_Write | fs::OpenMode_AllowAppend);
    NN_RESULT_DO(fs::SetFileSize(file, static_cast<int64_t>(bufferSize)));
    NN_RESULT_DO(fs::WriteFile(file, 0, buffer, static_cast<int64_t>(bufferSize), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
    NN_RESULT_SUCCESS;
}
Result SaveDataStorage::AppendImpl(const char* path, size_t offset, const void* buffer, size_t bufferSize) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS_EQUAL(offset, static_cast<size_t>(std::numeric_limits<int64_t>::max()));
    NN_SDK_ASSERT_LESS_EQUAL(bufferSize, static_cast<size_t>(std::numeric_limits<int64_t>::max()));
    FullPath fullPath;
    GetFullPath(fullPath, path);

    fs::FileHandle file;
    SCOPE_FOPEN(file, fullPath, fs::OpenMode_Write | fs::OpenMode_AllowAppend);
    NN_RESULT_DO(fs::WriteFile(file, static_cast<int64_t>(offset), buffer, static_cast<int64_t>(bufferSize), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));
    NN_RESULT_SUCCESS;
}
Result SaveDataStorage::ReadImpl(size_t* pOutActualSize, void* buffer, size_t bufferSize, const char* path, size_t offset) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_LESS_EQUAL(offset, static_cast<size_t>(std::numeric_limits<int64_t>::max()));
    NN_SDK_ASSERT_LESS_EQUAL(bufferSize, static_cast<size_t>(std::numeric_limits<int64_t>::max()));
    FullPath fullPath;
    GetFullPath(fullPath, path);

    fs::FileHandle file;
    SCOPE_FOPEN(file, fullPath, fs::OpenMode_Read);
    size_t readSize;
    NN_RESULT_DO(fs::ReadFile(&readSize, file, static_cast<int64_t>(offset), buffer, static_cast<int64_t>(bufferSize), fs::ReadOption::MakeValue(0)));
    *pOutActualSize = readSize;
    NN_RESULT_SUCCESS;
}

}}} // ~namespace nn::migration::detail
