﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_Diagnosis.h>

#include <nn/migration/migration_Result.h>
#include <nn/migration/detail/migration_Log.h>
#include <nn/migration/detail/migration_Result.h>

#include <nn/erpt/erpt_Context.h>
#include <nn/err/err_SystemApi.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace migration { namespace detail {

namespace {
inline Result ReportErrorImpl(Result result, bool isAborting) NN_NOEXCEPT
{
    // エラーコード
    auto errorCode = err::ConvertResultToErrorCode(result);
    char errorString[nn::err::ErrorCode::StringLengthMax];
    err::GetErrorCodeString(errorString, sizeof(errorString), errorCode);
    auto errorStringLength = static_cast<uint32_t>(strnlen(errorString, sizeof(errorString)));

    erpt::Context context(erpt::ErrorInfo);
    NN_RESULT_DO(context.Add(erpt::FieldId::AbortFlag, isAborting));
    NN_RESULT_DO(context.Add(erpt::FieldId::ErrorCode, errorString, errorStringLength));
    NN_RESULT_DO(context.AddCurrentThreadName());
    NN_RESULT_DO(context.CreateReport(erpt::ReportType_Invisible));
    NN_RESULT_SUCCESS;
}
int GetTimeStamp(char* str, size_t size) NN_NOEXCEPT
{
    time::PosixTime posix;
    auto r = time::StandardUserSystemClock::GetCurrentTime(&posix);
    if (!r.IsSuccess())
    {
        auto span = os::GetSystemTick().ToTimeSpan();
        auto l = util::SNPrintf(
            str, size,
            "[migration] Uptime=%lldh%02lldm%02llds | ",
            span.GetHours(),
            span.GetMinutes() % 60,
            span.GetSeconds() % 60);
        NN_SDK_ASSERT(l > 0 && static_cast<size_t>(l) < size);
        return l;
    }

    auto calendar = time::ToCalendarTimeInUtc(posix);
    auto l = util::SNPrintf(
        str, size,
        "[migration] %04hd-%02hhd-%02hhd %02hhd:%02hhd:%02hhd | ",
        calendar.year, calendar.month, calendar.day,
        calendar.hour, calendar.minute, calendar.second);
    NN_SDK_ASSERT(l > 0 && static_cast<size_t>(l) < size);
    return l;
}
} // ~namespace nn::migration::detail::<anonymous>

void ReportError(Result result, bool isAborting) NN_NOEXCEPT
{
    if (ResultInterrupted::Includes(result))
    {
        // 割り込み系の Result はレポートしない
        return;
    }

    auto r = ReportErrorImpl(result, isAborting);
    if (!r.IsSuccess())
    {
        PrintWithTimeStamp(DiagnosisKind_Error, "<!> ReportError() failed with %03d-%04d (%08lx)\n", r.GetModule(), r.GetDescription(), r.GetInnerValueForDebug());
    }
}

void PrintWithTimeStamp(DiagnosisKind kind, const char* pFormat, ...) NN_NOEXCEPT
{
    char buffer[256 + 64];
    auto l = GetTimeStamp(buffer, sizeof(buffer));

    std::va_list vaList;
    va_start(vaList, pFormat);
    auto ret = util::VSNPrintf(buffer + l, sizeof(buffer) - l, pFormat, vaList);
    va_end(vaList);
    NN_UNUSED(ret);

    switch (kind)
    {
    case DiagnosisKind_Info:
        NN_DETAIL_MIGRATION_INFO(buffer);
        break;
    case DiagnosisKind_Trace:
        NN_DETAIL_MIGRATION_TRACE(buffer);
        break;
    case DiagnosisKind_Warn:
        NN_DETAIL_MIGRATION_WARN(buffer);
        break;
    case DiagnosisKind_Error:
        NN_DETAIL_MIGRATION_ERROR(buffer);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}}} // ~nn::migration::detail
