﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/migration/detail/migration_ConnectionUtil.h>

#include <nn/ldn/ldn_Api.h>
#include <nn/migration/migration_Result.h>
#include <nn/migration/idc/migration_SocketConnection.h>
#include <nn/migration/idc/detail/migration_HandleSocketError.h>
#include <nn/migration/detail/migration_Diagnosis.h>
#include <nn/socket/socket_Api.h>

namespace nn { namespace migration { namespace detail {

Result SetupConnectionPort(int acceptSocket, socket::InAddr inAddr, uint16_t port) NN_NOEXCEPT
{
    socket::SockAddrIn server = {};
    server.sin_addr = inAddr;
    server.sin_port = socket::InetHtons(port);
    server.sin_family = socket::Family::Af_Inet;

    NN_MIGRATION_DETAIL_TRACE(
        "[SetupConnectionPort] Initialize socket for accepting connection to %u.%u.%u.%u\n",
        reinterpret_cast<char*>(&inAddr)[0], reinterpret_cast<char*>(&inAddr)[1], reinterpret_cast<char*>(&inAddr)[2], reinterpret_cast<char*>(&inAddr)[3]);

    auto r = socket::Bind(acceptSocket, reinterpret_cast<socket::SockAddr*>(&server), sizeof(server));
    NN_RESULT_THROW_UNLESS(r >= 0, idc::detail::HandleSocketError(socket::GetLastError()));
    r = socket::Listen(acceptSocket, 1);
    NN_RESULT_THROW_UNLESS(r >= 0, idc::detail::HandleSocketError(socket::GetLastError()));
    NN_RESULT_SUCCESS;
}

Result SetupConnectionPort(int acceptSocket, const char* inAddr, uint16_t port) NN_NOEXCEPT
{
    socket::InAddr addr;
    NN_ABORT_UNLESS(socket::InetAton(inAddr, &addr));
    return SetupConnectionPort(acceptSocket, addr, port);
}

Result SetupConnectionPort(int acceptSocket, const ldn::Ipv4Address& inAddr, uint16_t port) NN_NOEXCEPT
{
    socket::InAddr addr;
    addr.S_addr = socket::InetHtonl(inAddr.raw);
    return SetupConnectionPort(acceptSocket, addr, port);
}

Result WaitConnection(idc::SocketConnection* pOut, int acceptSocket, int timeoutSeconds, const Cancellable* pCancellable) NN_NOEXCEPT
{
    socket::PollFd fds[1] = {};
    fds[0].fd = acceptSocket;
    fds[0].events = nn::socket::PollEvent::PollIn;

    int count = 0;
    os::TimerEvent timer(os::EventClearMode_ManualClear);
    timer.StartOneShot(TimeSpan::FromSeconds(timeoutSeconds));
    while (!(count > 0) && !timer.TryWait())
    {
        count = socket::Poll(fds, std::extent<decltype(fds)>::value, 1000);
        NN_RESULT_THROW_UNLESS(count >= 0, idc::detail::HandleSocketError(socket::GetLastError()));
        NN_RESULT_THROW_UNLESS(!IsCanceled(pCancellable), ResultCanceled());
    }
    NN_RESULT_THROW_UNLESS(count > 0, ResultConnectionWaitingTimeout());
    NN_SDK_ASSERT_EQUAL(count, 1);

    socket::SockAddrIn client = {};
    auto clientAddrSize = static_cast<nn::socket::SockLenT>(sizeof(client));
    auto serverSocket = socket::Accept(acceptSocket, reinterpret_cast<nn::socket::SockAddr*>(&client), &clientAddrSize);
    NN_RESULT_THROW_UNLESS(serverSocket >= 0, idc::detail::HandleSocketError(socket::GetLastError()));

    *pOut = idc::SocketConnection(serverSocket);
    NN_RESULT_SUCCESS;
}

Result ConnectToPort(int clientSocket, socket::InAddr inAddr, uint16_t port) NN_NOEXCEPT
{
    socket::SockAddrIn server = {};
    server.sin_addr = inAddr;
    server.sin_port = socket::InetHtons(port);
    server.sin_family = socket::Family::Af_Inet;

    NN_MIGRATION_DETAIL_TRACE(
        "[ConnectToPort] Connecting to %u.%u.%u.%u\n",
        reinterpret_cast<char*>(&inAddr)[0], reinterpret_cast<char*>(&inAddr)[1], reinterpret_cast<char*>(&inAddr)[2], reinterpret_cast<char*>(&inAddr)[3]);

    auto r = socket::Connect(clientSocket, reinterpret_cast<socket::SockAddr*>(&server), sizeof(server));
    NN_RESULT_THROW_UNLESS(r >= 0, idc::detail::HandleSocketError(socket::GetLastError()));
    NN_RESULT_SUCCESS;
}

Result ConnectToPort(int clientSocket, const char* inAddr, uint16_t port) NN_NOEXCEPT
{
    socket::InAddr addr;
    NN_ABORT_UNLESS(socket::InetAton(inAddr, &addr));
    return ConnectToPort(clientSocket, addr, port);
}

Result ConnectToPort(int clientSocket, const ldn::Ipv4Address& inAddr, uint16_t port) NN_NOEXCEPT
{
    socket::InAddr addr;
    addr.S_addr = socket::InetHtonl(inAddr.raw);
    return ConnectToPort(clientSocket, addr, port);
}

}}} // ~namespace nn::migration::detail
