﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <string>

#include <nn/manu/server/manu_Log.h>
#include <nn/manu/manu_Result.public.h>

#include "manu_Request.h"
#include "manu_Types.h"
#include "manu_Usb.h"

namespace nn { namespace manu { namespace detail {

namespace
{
    UsbTransferPipe usbTransferPipe;
    const size_t UsbBufferSize=1024 * 128;
    NN_USB_DMA_ALIGN nn::Bit8 usbWorkBuffer[UsbBufferSize]={0};

    nn::Result UsbReceive(size_t* pOutSuccessXferSize, void* pOutBuffer, size_t receiveSize)
    {
        *pOutSuccessXferSize = 0;
        for(size_t remainSize=receiveSize; remainSize>0;)
        {
            auto xferSize = ::std::min(remainSize, UsbBufferSize);
            uint32_t actualXferSize = 0;
            NN_RESULT_DO(usbTransferPipe.Receive(&actualXferSize, usbWorkBuffer, static_cast<uint32_t>(xferSize)));
            ::std::memcpy(reinterpret_cast<nn::Bit8*>(pOutBuffer) + *pOutSuccessXferSize, usbWorkBuffer, actualXferSize);
            remainSize -= actualXferSize;
            *pOutSuccessXferSize += actualXferSize;
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result UsbSend(size_t* pOutSuccessXferSize, const void* pBuffer, size_t sendSize)
    {
        *pOutSuccessXferSize = 0;
        for(size_t remainSize=sendSize; remainSize>0;)
        {
            auto xferSize = ::std::min(remainSize, UsbBufferSize);
            uint32_t actualXferSize = 0;
            ::std::memcpy(usbWorkBuffer, reinterpret_cast<const nn::Bit8*>(pBuffer) + *pOutSuccessXferSize, xferSize);
            NN_RESULT_DO(usbTransferPipe.Send(&actualXferSize, usbWorkBuffer, static_cast<uint32_t>(xferSize)));
            remainSize -= actualXferSize;
            *pOutSuccessXferSize += actualXferSize;
        }
        NN_RESULT_SUCCESS;
    }
}

void InitializeForUsbTransfer(int portNumber)
{
    usbTransferPipe.Initialize(static_cast<uint8_t>(portNumber));
}

nn::Result UsbRead(size_t* pOutSuccessReadSize, void* pOutBuffer, size_t bufferSize, size_t readSize)
{
    NN_RESULT_DO(usbTransferPipe.Cancel());
    NN_RESULT_DO(UsbReceive(pOutSuccessReadSize, pOutBuffer, readSize));
    NN_RESULT_SUCCESS;
}

nn::Result UsbWrite(size_t* pOutSuccessWriteSize, const void* pBuffer, size_t bufferSize, size_t writeSize)
{
    NN_RESULT_DO(usbTransferPipe.Cancel());
    NN_RESULT_DO(UsbSend(pOutSuccessWriteSize, pBuffer, writeSize));
    NN_RESULT_SUCCESS;
}

}}}
