﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/ldn/ldn_PrivateResult.h>
#include <nn/ldn/ldn_Result.h>
#include <nn/ldn/detail/ldn_SessionManager.h>
#include <nn/ldn/detail/Debug/ldn_Assert.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/Utility/ldn_Crypto.h>
#include <nn/ldn/detail/Utility/ldn_Stringize.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace ldn { namespace detail { namespace
{
    // 鍵を生成するために使用するパラメータです。
    const Bit8 KekSourceForDataKey[] =
    {
        0x2B, 0x3F, 0xFD, 0x0A, 0x34, 0xAC, 0xE2, 0x77,
        0x6F, 0x08, 0x5F, 0x99, 0x87, 0xC2, 0xB9, 0x3D
    };

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    SessionManager::SessionManager(void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_Mutex(false),
          m_Buffer(static_cast<impl::SessionManagerBuffer*>(buffer)),
          m_TcpIpStackConfiguration(),
          m_AddressingServer(m_Buffer->addressing, sizeof(m_Buffer->addressing)),
          m_AddressingClient(),
          m_pAddressGetter(nullptr),
          m_IsInitialized(false)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, NN_ALIGNOF(impl::SessionContext));
        NN_SDK_ASSERT(RequiredBufferSize <= bufferSize);
        NN_UNUSED(bufferSize);
        auto& context = m_Buffer->context;
        std::memset(&context, 0, sizeof(impl::SessionContext));
    }

    SessionManager::~SessionManager() NN_NOEXCEPT
    {
        if (m_IsInitialized)
        {
            Finalize();
        }
    }

    void SessionManager::Initialize(
        const NetworkInterfaceProfile& profile,
        nn::os::SystemEventType* pStateChangeEvent) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(!m_IsInitialized);
        NN_SDK_ASSERT_NOT_NULL(pStateChangeEvent);

        // MAC アドレスを記憶しておきます。
        auto& context = m_Buffer->context;
        context.networkInterfaceProfile = profile;

        // 初期化処理に成功しました。
        m_IsInitialized = true;
        nn::os::ClearSystemEvent(pStateChangeEvent);
        m_pStateChangeEvent = pStateChangeEvent;
    }

    void SessionManager::Finalize() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_EQUAL(m_Buffer->context.role, SessionRole_None);

        // バッファに残った情報を全て破棄しておきます。
        auto& context = m_Buffer->context;
        std::memset(&context, 0, sizeof(impl::SessionContext));

        // 終了処理に成功しました。
        m_pStateChangeEvent = nullptr;
        m_IsInitialized = false;
    }

    Result SessionManager::GetNetworkInfo(
        NetworkId* pOutNetworkId, LdnNetworkInfo* pOutNetwork) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutNetworkId);
        NN_SDK_ASSERT_NOT_NULL(pOutNetwork);
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        // セッションが構築されていることを確認します。
        const auto& context = m_Buffer->context;
        if (!m_IsInitialized || context.role == SessionRole_None)
        {
            return ResultSessionInvalidState();
        }

        // 全体を 0 で初期化しておきます。
        auto& ldn = *pOutNetwork;
        std::memset(&ldn, 0, sizeof(ldn));

        // LdnNetworkInfo を構築します。
        std::memcpy(ldn.serverRandom, context.nodes[0].random, RandomSize);
        ldn.securityMode = context.security.securityMode;
        ldn.stationAcceptPolicy = context.stationAcceptPolicy;
        ldn.nodeCountMax = context.nodeCountMax;
        for (int i = 0; i < NodeCountMax; ++i)
        {
            auto& dst = ldn.nodes[i];
            const auto& src = context.nodes[i];
            dst.nodeId = static_cast<int8_t>(i);
            if (src.isJoined)
            {
                dst.ipv4Address = src.ipv4Address;
                dst.macAddress = src.macAddress;
                dst.isConnected = true;
                dst.localCommunicationVersion = src.localCommunicationVersion;
                std::memcpy(dst.userName, src.userName, UserNameBytesMax);
                ++ldn.nodeCount;
            }
        }
        ldn.advertiseDataSize = context.advertiseDataSize;
        if (0 < context.advertiseDataSize)
        {
            std::memcpy(ldn.advertiseData, context.advertiseData, context.advertiseDataSize);
        }

        // NetworkId を出力します。
        *pOutNetworkId = context.networkId;
        NN_RESULT_SUCCESS;
    }

    Result SessionManager::GetNodeLatestUpdate(
        NodeLatestUpdate* outUpdates, int bufferCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(outUpdates);
        NN_SDK_ASSERT(NodeCountMax <= bufferCount);
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        // セッションが構築されていることを確認します。
        auto& context = m_Buffer->context;
        if (!m_IsInitialized || context.role == SessionRole_None)
        {
            return ResultSessionInvalidState();
        }

        // 接続状態の変化を出力します。
        for (int i = 0; i < bufferCount && i < NodeCountMax; ++i)
        {
            auto& node = context.nodes[i];
            outUpdates[i].stateChange = node.stateChange;
            node.stateChange = NodeStateChange_None;
        }
        NN_RESULT_SUCCESS;
    }

    Result SessionManager::GetIpv4Address(
        Ipv4Address* pOutAddress, SubnetMask* pOutMask) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutAddress);
        NN_SDK_ASSERT_NOT_NULL(pOutMask);
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        // セッションが構築されていることを確認します。
        if (m_pAddressGetter == nullptr)
        {
            return ResultSessionInvalidState();
        }

        // IPv4 アドレスを出力します。
        *pOutAddress = m_pAddressGetter->GetIpv4Address();
        *pOutMask = m_pAddressGetter->GetSubnetMask();
        NN_RESULT_SUCCESS;
    }

    void SessionManager::SetUser(const UserConfig& user) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_EQUAL(m_Buffer->context.role, SessionRole_None);
        auto& context = m_Buffer->context;
        context.user = user;
    }

    void SessionManager::SetSecurity(
        const SecurityConfig& security, const Bit8 (&serverRandom)[RandomSize]) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_EQUAL(m_Buffer->context.role, SessionRole_None);
        NN_SDK_ASSERT_MINMAX(security.passphraseSize, PassphraseSizeMin, PassphraseSizeMax);
        NN_SDK_ASSERT_NOT_NULL(serverRandom);
        auto& context = m_Buffer->context;
        context.security = security;
        std::memcpy(context.nodes[0].random, serverRandom, RandomSize);
    }

    void SessionManager::SetLocalCommunicationVersion(int version) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_EQUAL(m_Buffer->context.role, SessionRole_None);
        NN_SDK_ASSERT_MINMAX(version, LocalCommunicationVersionMin, LocalCommunicationVersionMax);
        auto& context = m_Buffer->context;
        context.localCommunicationVersion = static_cast<int16_t>(version);
    }

    void SessionManager::CreateDataKey(
        void* buffer, size_t* pOutSize, size_t bufferSize) const NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_EQUAL(m_Buffer->context.role, SessionRole_None);
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_NOT_NULL(pOutSize);
        NN_SDK_ASSERT(16 <= bufferSize);

        // Debug/SystemDebug では鍵はありません。
        const auto& context = m_Buffer->context;
        const auto& security = context.security;
        if (security.securityMode != SecurityMode_Product)
        {
            *pOutSize = 0;
            return;
        }

        // ServerRandom/Passphrase を連結して入力メッセージとします。
        const auto& server = context.nodes[0];
        Bit8 message[RandomSize + PassphraseSizeMax] = { };
        std::memcpy(message + 0, server.random, RandomSize);
        std::memcpy(message + RandomSize, security.passphrase, security.passphraseSize);
        size_t messageSize = RandomSize + security.passphraseSize;

        // キーのソースをダンプします。
        NN_LDN_LOG_DEBUG("Data Src : ");
        for (size_t i = 0; i < messageSize; ++i)
        {
            NN_LDN_LOG_DEBUG_WITHOUT_PREFIX("%02X ", message[i]);
        }
        NN_LDN_LOG_DEBUG_WITHOUT_PREFIX("\n");

        // メッセージダイジェストをソースとしてセッション鍵を生成します。
        Bit8 key[16];
        GenerateAesKey(
            key, message, messageSize, KekSourceForDataKey, sizeof(KekSourceForDataKey));
        *pOutSize = 16 < bufferSize ? 16 : bufferSize;
        std::memcpy(buffer, key, *pOutSize);

        // 生成されたキーをダンプします。
        NN_LDN_LOG_DEBUG("Data Key : ");
        for (size_t i = 0; i < 16U; ++i)
        {
            NN_LDN_LOG_DEBUG_WITHOUT_PREFIX("%02X ", key[i]);
        }
        NN_LDN_LOG_DEBUG_WITHOUT_PREFIX("\n");
    }

    int SessionManager::Find(Ipv4Address ipv4Address) const NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_NOT_EQUAL(ipv4Address, ZeroIpv4Address);
        NN_SDK_ASSERT_NOT_EQUAL(ipv4Address, BroadcastIpv4Address);
        auto& context = m_Buffer->context;
        if (context.role != SessionRole_None)
        {
            for (int aid = 1; aid < NodeCountMax; ++aid)
            {
                const auto& node = context.nodes[aid];
                if (node.isJoined && node.ipv4Address == ipv4Address)
                {
                    return aid;
                }
            }
        }
        return -1;
    }

    void SessionManager::CreateSession(
        const NetworkId& networkId, int nodeCountMax,
        const AddressEntry* addressEntries, int addressEntryCount) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_EQUAL(m_Buffer->context.role, SessionRole_None);
        NN_SDK_ASSERT_MINMAX(nodeCountMax, 1, NodeCountMax);

        // セッション情報を記憶しておきます。
        auto& context = m_Buffer->context;
        context.networkId = networkId;
        context.nodeCountMax = static_cast<int8_t>(nodeCountMax);
        context.aid = 0;
        context.role = static_cast<Bit8>(SessionRole_Administrator);

        // AP の情報を記憶しておきます。
        auto& ap = context.nodes[0];
        ap.macAddress = context.networkInterfaceProfile.physicalAddress;;
        ap.localCommunicationVersion = static_cast<int16_t>(context.localCommunicationVersion);
        std::memcpy(ap.userName, context.user.userName, UserNameBytesMax);

        // 静的エントリを追加します。
        for (int i = 0; i < addressEntryCount; ++i)
        {
            NN_SDK_ASSERT_NOT_NULL(addressEntries);
            const auto& entry = addressEntries[i];
            m_AddressingServer.AddEntry(entry.ipv4Address, entry.macAddress);
        }

        // 自身の IP アドレスを割り当てます。
        m_AddressingServer.StartServer(context.networkInterfaceProfile.physicalAddress);
        ap.ipv4Address = m_AddressingServer.GetIpv4Address();
        StartUp(&m_AddressingServer);
        SetJoin(0, true);

        // セッションの構築に成功しました。
        nn::os::SignalSystemEvent(m_pStateChangeEvent);
    }

    void SessionManager::DestroySession() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_EQUAL(m_Buffer->context.role, SessionRole_Administrator);

        // IP アドレスを払い下げます。
        Down();
        m_AddressingServer.StopServer();
        m_AddressingServer.ClearEntries();

        // 引き継ぐ必要のないセッション情報を破棄しておきます。
        auto& context = m_Buffer->context;
        std::memset(&context.networkId, 0, sizeof(NetworkId));
        std::memset(&context.nodes, 0, sizeof(context.nodes));
        context.role = static_cast<Bit8>(SessionRole_None);

        // セッションの破棄に成功しました。
        nn::os::SignalSystemEvent(m_pStateChangeEvent);
    }

    void SessionManager::SetAdvertiseData(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT_MINMAX(dataSize, 0U, AdvertiseDataSizeMax);

        // Advertise で配信するデータをコピーしておきます。
        auto& context = m_Buffer->context;
        context.advertiseDataSize = static_cast<uint16_t>(dataSize);
        if (0 < dataSize)
        {
            std::memcpy(context.advertiseData, data, dataSize);
        }
    }

    void SessionManager::SetStationAcceptPolicy(AcceptPolicy policy) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_NOT_EQUAL(m_Buffer->context.role, SessionRole_User);
        auto& context = m_Buffer->context;
        context.stationAcceptPolicy = static_cast<Bit8>(policy);
    }

    Result SessionManager::IsAcceptable(MacAddress macAddress) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_NOT_EQUAL(macAddress, ZeroMacAddress);
        NN_SDK_ASSERT_NOT_EQUAL(macAddress, BroadcastMacAddress);
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        int i;

        // セッションが構築されていることを確認します。
        const auto& context = m_Buffer->context;
        if (context.role != SessionRole_Administrator)
        {
            return ResultSessionInvalidState();
        }

        // 接続数が最大に達していないことを確認します。
        if (context.nodeCountMax <= GetNodeCount())
        {
            return ResultConnectionRejected();
        }

        // 現在のポリシーに応じて接続の是非を判定します。
        auto policy = static_cast<AcceptPolicy>(context.stationAcceptPolicy);
        switch (policy)
        {
        case AcceptPolicy_AlwaysAccept:
            break;
        case AcceptPolicy_AlwaysReject:
            return ResultConnectionRejected();
        case AcceptPolicy_BlackList:
            for (i = 0; i < AcceptFilterEntryCountMax; ++i)
            {
                if (macAddress == context.filterEntries[i])
                {
                    return ResultConnectionRejected();
                }
            }
            break;
        case AcceptPolicy_WhiteList:
            for (i = 0; i < AcceptFilterEntryCountMax; ++i)
            {
                if (macAddress == context.filterEntries[i])
                {
                    break;
                }
            }
            if (AcceptFilterEntryCountMax <= i)
            {
                return ResultConnectionRejected();
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        NN_RESULT_SUCCESS;
    }

    Result SessionManager::AddUser(
        int aid,
        MacAddress macAddress,
        int version,
        const Bit8 (&clientRandom)[RandomSize],
        const UserConfig& user) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_RANGE(aid, 0, NodeCountMax);
        NN_SDK_ASSERT_NOT_EQUAL(macAddress, ZeroMacAddress);
        NN_SDK_ASSERT_NOT_EQUAL(macAddress, BroadcastMacAddress);
        NN_SDK_ASSERT_MINMAX(version, LocalCommunicationVersionMin, LocalCommunicationVersionMax);

        // セッションが構築されていることを確認します。
        auto& context = m_Buffer->context;
        if (context.role != SessionRole_Administrator)
        {
            return ResultSessionInvalidState();
        }

        // ノードリストの空きを確認します。
        if (aid <= 0 || NodeCountMax <= aid || context.nodes[aid].isJoined ||
            context.nodeCountMax <= GetNodeCount())
        {
            return ResultConnectionRejected();
        }

        // ノード情報を保存します。
        auto& node = context.nodes[aid];
        node.macAddress = macAddress;
        node.localCommunicationVersion = static_cast<int16_t>(version);
        std::memcpy(node.random, clientRandom, RandomSize);
        std::memcpy(node.userName, user.userName, UserNameBytesMax);

        // IPv4 アドレスを割り当てます。
        node.ipv4Address = m_AddressingServer.Assign(macAddress);
        NN_SDK_ASSERT_NOT_EQUAL(node.ipv4Address, ZeroIpv4Address);
        NN_SDK_ASSERT_NOT_EQUAL(node.ipv4Address, BroadcastIpv4Address);
        m_TcpIpStackConfiguration.AddEntry(node.ipv4Address, macAddress);
        SetJoin(aid, true);
        nn::os::SignalSystemEvent(m_pStateChangeEvent);

        // デバッグのため、セッションに参加したステーションの情報を出力します。
        NN_LDN_STRINGIZE_HELPER(ipHelper, StringizedIpv4AddressLengthMax);
        NN_LDN_STRINGIZE_HELPER(macHelper, StringizedMacAddressLength);
        NN_LDN_LOG_INFO("accepted node %d (IP=%s, MAC=%s)\n",
            aid, ipHelper.ToString(node.ipv4Address), macHelper.ToString(node.macAddress));
        NN_RESULT_SUCCESS;
    }

    Result SessionManager::RemoveUser(int aid) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_RANGE(aid, 0, NodeCountMax);

        // セッションが構築されていることを確認します。
        auto& context = m_Buffer->context;
        if (context.role != SessionRole_Administrator)
        {
            return ResultSessionInvalidState();
        }

        // ノードリストの空きを確認します。
        if (aid <= 0 || NodeCountMax <= aid || !context.nodes[aid].isJoined)
        {
            return ResultNodeNotFound();
        }

        // IPv4 アドレスの割り当てを解除します。
        auto& node = context.nodes[aid];
        m_TcpIpStackConfiguration.RemoveEntry(node.ipv4Address);
        m_AddressingServer.Free(node.ipv4Address);

        // デバッグのため、セッションから離脱したステーションの情報を出力します。
        NN_LDN_STRINGIZE_HELPER(ipHelper, StringizedIpv4AddressLengthMax);
        NN_LDN_STRINGIZE_HELPER(macHelper, StringizedMacAddressLength);
        NN_LDN_LOG_INFO("removed node %d (IP=%s, MAC=%s)\n",
            aid, ipHelper.ToString(node.ipv4Address), macHelper.ToString(node.macAddress));

        // ノード情報を削除します。
        SetJoin(aid, false);
        nn::os::SignalSystemEvent(m_pStateChangeEvent);
        NN_RESULT_SUCCESS;
    }

    Result SessionManager::AddAcceptFilterEntry(MacAddress macAddress) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_NOT_EQUAL(m_Buffer->context.role, SessionRole_User);
        NN_SDK_ASSERT_NOT_EQUAL(macAddress, ZeroMacAddress);
        NN_SDK_ASSERT_NOT_EQUAL(macAddress, BroadcastMacAddress);
        auto& context = m_Buffer->context;
        context.filterEntries[context.nextfilterEntryIndex] = macAddress;
        context.nextfilterEntryIndex = static_cast<int16_t>(
            (context.nextfilterEntryIndex + 1) % AcceptFilterEntryCountMax);
        NN_RESULT_SUCCESS;
    }

    Result SessionManager::ClearAcceptFilter() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_NOT_EQUAL(m_Buffer->context.role, SessionRole_User);
        auto& context = m_Buffer->context;
        std::memset(context.filterEntries, 0, sizeof(context.filterEntries));
        context.nextfilterEntryIndex = 0;
        NN_RESULT_SUCCESS;
    }

    Result SessionManager::JoinSession(
        const NetworkId& networkId, const LdnNetworkInfo& ldn, int aid) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_EQUAL(m_Buffer->context.role, SessionRole_None);
        NN_SDK_ASSERT_RANGE(aid, 1, NodeCountMax);

        // ローカル通信バージョンが異なる場合は接続に失敗します。
        auto& context = m_Buffer->context;
        if (ldn.nodes[0].localCommunicationVersion < context.localCommunicationVersion)
        {
            return ResultHigherVersion();
        }
        else if (context.localCommunicationVersion < ldn.nodes[0].localCommunicationVersion)
        {
            return ResultLowerVersion();
        }

        // Network Stack を Start Up します。
        m_AddressingClient.StartClient(ldn.nodes[0].ipv4Address, ldn.nodes[aid].ipv4Address);
        StartUp(&m_AddressingClient);

        // セッション情報を記憶しておきます。
        context.networkId = networkId;
        context.aid = static_cast<int8_t>(aid);
        context.role = SessionRole_User;
        UpdateNetworkInfoImpl(nullptr, ldn);
        SetJoin(context.aid, true);
        nn::os::SignalSystemEvent(m_pStateChangeEvent);
        NN_RESULT_SUCCESS;
    }

    void SessionManager::LeaveSession() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_EQUAL(m_Buffer->context.role, SessionRole_User);

        // Network Stack を Down します。
        Down();
        m_AddressingClient.StopClient();

        // 引き継ぐ必要のないセッション情報を破棄しておきます。
        auto& context = m_Buffer->context;
        std::memset(&context.networkId, 0, sizeof(NetworkId));
        std::memset(&context.nodes, 0, sizeof(context.nodes));
        context.role = static_cast<Bit8>(SessionRole_None);

        // セッションの破棄に成功しました。
        nn::os::SignalSystemEvent(m_pStateChangeEvent);
    }

    Result SessionManager::UpdateNetworkInfo(const LdnNetworkInfo& ldn) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        NN_SDK_ASSERT(m_IsInitialized);

        // セッションに参加していることを確認します。
        auto& context = m_Buffer->context;
        if (context.role != SessionRole_User)
        {
            return ResultSessionInvalidState();
        }

        // セッション情報を記録しておきます。
        bool isUpdated;
        NN_RESULT_DO(UpdateNetworkInfoImpl(&isUpdated, ldn));
        if (isUpdated)
        {
            nn::os::SignalSystemEvent(m_pStateChangeEvent);
        }
        NN_RESULT_SUCCESS;
    }

    void SessionManager::StartUp(IAddressGetter* pAddressGetter) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_pAddressGetter == nullptr);
        m_TcpIpStackConfiguration.Startup(
            pAddressGetter->GetIpv4Address(),
            pAddressGetter->GetGatewayAddress(),
            pAddressGetter->GetSubnetMask(),
            m_Buffer->context.networkInterfaceProfile);
        m_pAddressGetter = pAddressGetter;
    }

    void SessionManager::Down() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pAddressGetter);
        m_TcpIpStackConfiguration.Down();
        m_pAddressGetter = nullptr;
    }

    bool SessionManager::UpdateAddress(
        int aid, const impl::SessionNodeInfo& prev, const NodeInfo& current) NN_NOEXCEPT
    {
        if (current.isConnected)
        {
            if (!prev.isJoined)
            {
                // 新規接続です。
                m_TcpIpStackConfiguration.AddEntry(current.ipv4Address, current.macAddress);
                SetJoin(aid, true);
                return true;
            }
            else if (current.macAddress != prev.macAddress ||
                     current.ipv4Address != prev.ipv4Address)
            {
                // 切断と新規接続が同時に発生しました。
                SetJoin(aid, false);
                m_TcpIpStackConfiguration.RemoveEntry(prev.ipv4Address);
                m_TcpIpStackConfiguration.AddEntry(current.ipv4Address, current.macAddress);
                SetJoin(aid, true);
                return true;
            }
        }
        else
        {
            if (prev.isJoined)
            {
                // 切断です。
                SetJoin(aid, false);
                m_TcpIpStackConfiguration.RemoveEntry(prev.ipv4Address);
                return true;
            }
        }
        return false;
    }

    Result SessionManager::UpdateNetworkInfoImpl(
        bool *pOutIsUpdated, const LdnNetworkInfo& ldn) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsInitialized);
        auto& context = m_Buffer->context;

        // 自分自身が含まれることを確認します。
        const auto& node = ldn.nodes[context.aid];
        if (!node.isConnected || node.ipv4Address != m_pAddressGetter->GetIpv4Address())
        {
            return ResultSessionInvalidState();
        }

        // コンテキストを更新します。
        context.nodeCountMax = ldn.nodeCountMax;
        context.stationAcceptPolicy = ldn.stationAcceptPolicy;
        context.advertiseDataSize = ldn.advertiseDataSize;
        std::memcpy(context.advertiseData, ldn.advertiseData, ldn.advertiseDataSize);
        bool isUpdated = false;
        for (int i = 0; i < NodeCountMax; ++i)
        {
            const auto& src = ldn.nodes[i];
            auto& dst = context.nodes[i];
            if (i != context.aid)
            {
                isUpdated |= UpdateAddress(i, dst, src);
            }
            dst.ipv4Address = src.ipv4Address;
            dst.macAddress = src.macAddress;
            dst.localCommunicationVersion = src.localCommunicationVersion;
            std::memcpy(dst.userName, src.userName, UserNameBytesMax);
        }
        if (pOutIsUpdated != nullptr)
        {
            *pOutIsUpdated = isUpdated;
        }
        NN_RESULT_SUCCESS;
    }

    int SessionManager::GetNodeCount() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsInitialized);
        auto& context = m_Buffer->context;
        int nodeCount = 0;
        for (int i = 0; i < NodeCountMax; ++i)
        {
            if (context.nodes[i].isJoined)
            {
                ++nodeCount;
            }
        }
        return nodeCount;
    }

    void SessionManager::SetJoin(int aid, bool isJoined) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsInitialized);
        NN_SDK_ASSERT_RANGE(aid, 0, NodeCountMax);
        auto& node = m_Buffer->context.nodes[aid];
        NN_SDK_ASSERT_NOT_EQUAL(node.isJoined, isJoined);
        NodeStateChange stateChange;
        switch (node.stateChange)
        {
        case NodeStateChange_None:
            if (isJoined)
            {
                stateChange = NodeStateChange_Connect;
            }
            else
            {
                stateChange = NodeStateChange_Disconnect;
            }
            break;
        case NodeStateChange_Connect:
            stateChange = NodeStateChange_None;
            break;
        case NodeStateChange_Disconnect:
            stateChange = NodeStateChange_DisconnectAndConnect;
            break;
        case NodeStateChange_DisconnectAndConnect:
            stateChange = NodeStateChange_Disconnect;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        node.isJoined = isJoined;
        node.stateChange = static_cast<int8_t>(stateChange);
    }

}}} // namespace nn::ldn::detail
