﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>
#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/ldn/detail/ldn_Version.h>
#include <nn/ldn/detail/Authentication/ldn_ChallengeResponseAuthentication.h>
#include <nn/ldn/detail/Debug/ldn_Log.h>
#include <nn/ldn/detail/NetworkInterface/ldn_NintendoEthernet.h>
#include <nn/ldn/detail/Utility/ldn_Stringize.h>
#include <nn/os/os_MultipleWait.h>

namespace nn { namespace ldn { namespace detail { namespace
{
    // サーバの受信キューの容量です。
    const size_t ServerReceiveQueueCapacity = 4;

    // クライアントの受信キューのサイズです。
    const int ClientReceiveQueueCapacity = 4;

    // サーバ側の認証タイムアウト時間です。
    const nn::TimeSpan ServerTimeout = nn::TimeSpan::FromMilliSeconds(5000);

    // 認証要求・認証応答のフラグです。
    enum AuthenticationFlag
    {
        AuthenticationFlag_Response = 1 << 0
    };

    // 認証フレームのヘッダです。
    struct AuthenticationHeader
    {
        Version     version;
        uint8_t     dataSize;
        Bit8        result;
        Bit8        flag;
        NN_PADDING4;
        char        networkId[32];
        Bit8        serverRandom[RandomSize];
        Bit8        clientRandom[RandomSize];
    };

    // 認証フレームの最大ペイロードサイズです。
    const size_t AuthenticationFramePayloadSizeMax = 200;

    // 認証フレームです。
    struct AuthenticationFrame
    {
        EthernetHeader              ethernetHeader;
        OuiExtendedEthernetHeader   ouiExtendedEthernetHeader;
        NintendoEthernetHeader      nintendoEthernetHeader;
        AuthenticationHeader        authenticationHeader;
        Bit8                        payload[AuthenticationFramePayloadSizeMax];
    };
    NN_STATIC_ASSERT(sizeof(AuthenticationFrame) == 292);

    // 認証フレームの最小サイズです。
    const size_t AuthenticationFrameSizeMin =
        sizeof(AuthenticationFrame) - AuthenticationFramePayloadSizeMax;

}}}} // namespace nn::ldn::detail::<unnamed>

namespace nn { namespace ldn { namespace detail
{
    size_t ChallengeResponseAuthenticationServer::GetRequiredBufferSize() NN_NOEXCEPT
    {
        return sizeof(impl::ChallengeResponseAuthenticationServerBuffer);
    }

    ChallengeResponseAuthenticationServer::ChallengeResponseAuthenticationServer(
        void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_AuthEvent(nn::os::EventClearMode_AutoClear),
          m_GetAuthEvent(nn::os::EventClearMode_AutoClear),
          m_ReceivedEvent(nn::os::EventClearMode_AutoClear),
          m_ClientMutex(false),
          m_DataMutex(false),
          m_Buffer(static_cast<impl::ChallengeResponseAuthenticationServerBuffer*>(buffer)),
          m_pReceiver(nullptr),
          m_pSender(nullptr),
          m_Counter(0),
          m_IsRunning(false),
          m_HasHistory(false)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, nn::os::ThreadStackAlignment);
        NN_SDK_ASSERT(GetRequiredBufferSize() <= bufferSize);
        NN_UNUSED(bufferSize);
        m_ReceiveQueue.Initialize(
            m_Buffer->receive, sizeof(m_Buffer->receive),
            sizeof(AuthenticationFrame), ServerReceiveQueueCapacity);
    }

    ChallengeResponseAuthenticationServer::~ChallengeResponseAuthenticationServer() NN_NOEXCEPT
    {
        if (m_IsRunning)
        {
            EndServer();
        }
        m_ReceiveQueue.Finalize();
    }

    void ChallengeResponseAuthenticationServer::BeginServer(
        const NetworkId& networkId, const Bit8 (&serverRandom)[RandomSize],
        IFrameReceiver* pReceiver, IFrameSender* pSender) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsRunning);
        NN_SDK_ASSERT_NOT_NULL(serverRandom);
        NN_SDK_ASSERT_NOT_NULL(pReceiver);
        NN_SDK_ASSERT_NOT_NULL(pSender);

        // 認証対象のステーションを管理するリストとバッファを初期化しておきます。
        std::memset(m_Buffer, 0, sizeof(impl::ChallengeResponseAuthenticationServerBuffer));
        std::memset(m_Clients, 0, sizeof(m_Clients));

        // 引数で受け取ったパラメータを保存します。
        m_pReceiver = pReceiver;
        m_pSender = pSender;
        m_NetworkId = networkId;

        // 送信フレームのテンプレート（固定部分）を生成します。
        auto& response = *reinterpret_cast<AuthenticationFrame*>(m_Buffer->send);
        response.authenticationHeader.version = CurrentVersion;
        response.authenticationHeader.flag = static_cast<Bit8>(AuthenticationFlag_Response);
        std::memcpy(response.authenticationHeader.networkId, &networkId, sizeof(NetworkId));
        std::memcpy(response.authenticationHeader.serverRandom, serverRandom, RandomSize);

        // フレームの受信を開始します。
        m_ReceiveQueue.Clear();
        m_pReceiver->Register(
            AuthenticationProtocolId, m_ReceivedEvent.GetBase(), &m_ReceiveQueue);

        // 認証処理用のスレッドを生成します。
        m_IsRunning = true;
        m_HasHistory = false;
        m_AuthEvent.Clear();
        m_GetAuthEvent.Clear();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
            &m_Thread, AuthThread, this, m_Buffer->threadStack, sizeof(m_Buffer->threadStack),
            NN_SYSTEM_THREAD_PRIORITY(ldn, AuthenticationServer)));
        nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(ldn, AuthenticationServer));
        nn::os::StartThread(&m_Thread);
    }

    void ChallengeResponseAuthenticationServer::EndServer() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning);

        // 認証スレッドの終了を待機します。
        m_IsRunning = false;
        m_ReceivedEvent.Signal();
        nn::os::WaitThread(&m_Thread);
        nn::os::DestroyThread(&m_Thread);

        // フレームの受信を中止します。
        m_pReceiver->Unregister(AuthenticationProtocolId);

        // 管理領域をクリアします。
        m_pSender = nullptr;
        m_pReceiver = nullptr;
    }

    int ChallengeResponseAuthenticationServer::Register(
        MacAddress macAddress, bool accept) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_ClientMutex);
        NN_SDK_ASSERT(m_IsRunning);
        NN_SDK_ASSERT_NOT_EQUAL(macAddress, ZeroMacAddress);
        NN_SDK_ASSERT_NOT_EQUAL(macAddress, BroadcastMacAddress);

        // リストの空きを検索します。
        int i;
        for (i = 0;
             i < StationCountMax && m_Clients[i].status != impl::AuthenticationState_Unregistered;
             ++i)
        {
            if (m_Clients[i].macAddress == macAddress)
            {
                NN_SDK_ASSERT(false, "This client has already been registered");
                return -1;
            }
        }

        // 空きを見つけた場合には登録します。
        if (i < StationCountMax)
        {
            auto& client = m_Clients[i];
            client.id = m_Counter;
            client.status = impl::AuthenticationState_Registered;
            client.accept = accept;
            client.macAddress = macAddress;
            client.registeredAt = nn::os::GetSystemTick().GetInt64Value();
            ++m_Counter;
            return client.id;
        }
        else
        {
            return -1;
        }
    }

    void ChallengeResponseAuthenticationServer::Unregister(int id) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_ClientMutex);
        NN_SDK_ASSERT(m_IsRunning);

        // リストから対象のクライアントを検索します。
        int i;
        for (i = 0; i < StationCountMax && m_Clients[i].id != id; ++i)
        {
        }

        // 対象のクライアントを発見した場合は削除します。
        if (i < StationCountMax)
        {
            auto& client = m_Clients[i];
            std::memset(&client, 0, sizeof(client));
        }
    }

    void ChallengeResponseAuthenticationServer::SetData(
        const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(m_pReceiver);
        NN_SDK_ASSERT_NOT_NULL(m_pSender);
        NN_SDK_ASSERT_NOT_NULL(data);
        NN_SDK_ASSERT(dataSize < AuthenticationFramePayloadSizeMax);
        std::lock_guard<nn::os::Mutex> lock(m_DataMutex);
        auto& response = *reinterpret_cast<AuthenticationFrame*>(m_Buffer->send);
        size_t copySize = std::min(dataSize, AuthenticationFramePayloadSizeMax);
        response.authenticationHeader.dataSize = static_cast<uint8_t>(copySize);
        if (0 < copySize)
        {
            std::memcpy(response.payload, data, copySize);
        }
    }

    nn::os::Event& ChallengeResponseAuthenticationServer::GetAuthenticationEvent() NN_NOEXCEPT
    {
        return m_AuthEvent;
    }

    bool ChallengeResponseAuthenticationServer::GetAuthenticationHistory(
        AuthenticationHistroy* pOutHistory) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning);
        NN_SDK_ASSERT_NOT_NULL(pOutHistory);
        if (m_HasHistory)
        {
            *pOutHistory = m_History;
            m_HasHistory = false;
            m_GetAuthEvent.Signal();
            return true;
        }
        else
        {
            return false;
        }
    }

    void ChallengeResponseAuthenticationServer::Expire() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> lock(m_ClientMutex);

        // 期限切れのクライアントを検索して削除します。
        auto now = nn::os::GetSystemTick();
        for (int i = 0; i < StationCountMax && !m_HasHistory; ++i)
        {
            auto& client = m_Clients[i];
            if (client.status == impl::AuthenticationState_Registered)
            {
                auto diff = (now - nn::os::Tick(client.registeredAt)).ToTimeSpan();
                if (ServerTimeout <= diff)
                {
                    NN_LDN_LOG_WARN("Authentication Failed: Expired\n");
                    std::memset(&m_History, 0, sizeof(AuthenticationHistroy));
                    m_History.id = client.id;
                    m_History.authenticationResult = AuthenticationResult_Timeout;
                    std::memset(&client, 0, sizeof(client));
                    m_HasHistory = true;
                    m_AuthEvent.Signal();
                }
            }
        }
    }

    AuthenticationResult ChallengeResponseAuthenticationServer::Judge(
        const void* data, size_t dataSize) const NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(data);

        // 受信データを取得します。
        const auto& request = *static_cast<const AuthenticationFrame*>(data);
        const auto& response = *reinterpret_cast<const AuthenticationFrame*>(m_Buffer->send);
        const size_t calculatedPayloadSize = dataSize - AuthenticationFrameSizeMin;

        // 認証リクエストの正当性を評価します。
        if (dataSize < AuthenticationFrameSizeMin)
        {
            NN_LDN_LOG_WARN("authentication failed: bad request (too small)\n");
            return AuthenticationResult_BadRequest;
        }
        else if (!IsCompatibleVersion(request.authenticationHeader.version))
        {
            NN_LDN_LOG_WARN("authentication failed: incompatible version\n");
            return AuthenticationResult_IncompatibleVersion;
        }
        else if (request.authenticationHeader.dataSize != calculatedPayloadSize ||
                 AuthenticationFramePayloadSizeMax < request.authenticationHeader.dataSize)
        {
            NN_LDN_LOG_WARN("authentication failed: bad request (payload size)\n");
            return AuthenticationResult_BadRequest;
        }
        else if (std::memcmp(request.authenticationHeader.serverRandom,
                             response.authenticationHeader.serverRandom, RandomSize) != 0)
        {
            NN_LDN_LOG_WARN("authentication failed: bad request (server random)\n");
            return AuthenticationResult_BadRequest;
        }
        else if (std::memcmp(request.authenticationHeader.networkId,
                 response.authenticationHeader.networkId, sizeof(NetworkId)) != 0)
        {
            NN_LDN_LOG_WARN("authentication failed: bad request (network id)\n");
            return AuthenticationResult_BadRequest;
        }
        else if ((request.authenticationHeader.flag & AuthenticationFlag_Response) != 0)
        {
            NN_LDN_LOG_WARN("authentication failed: bad request (response flag)\n");
            return AuthenticationResult_BadRequest;
        }

        // クライアントの接続を許可できるか確認します。
        const auto& src = request.ethernetHeader.src;
        std::lock_guard<nn::os::Mutex> lock(m_ClientMutex);
        int i;
        for (i = 0; i < StationCountMax; ++i)
        {
            const auto& client = m_Clients[i];
            if (client.status != impl::AuthenticationState_Unregistered &&
                client.macAddress == src)
            {
                break;
            }
        }
        if (StationCountMax <= i)
        {
            NN_LDN_LOG_WARN("authentication failed: unknown station\n");
            return AuthenticationResult_UnknownStation;
        }

        // 認証の成否を判定します。
        if (m_Clients[i].accept)
        {
            NN_LDN_LOG_DEBUG("authentication succeeded\n");
            return AuthenticationResult_Success;
        }
        else
        {
            NN_LDN_LOG_WARN("authentication failed: forbidden\n");
            return AuthenticationResult_Forbidden;
        }
    }

    void ChallengeResponseAuthenticationServer::Reply(
        const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(data);

        // 受信データを取得します。
        const auto& request = *reinterpret_cast<const AuthenticationFrame*>(data);
        const auto& requestHeader = request.authenticationHeader;
        NN_LDN_STRINGIZE_HELPER(helper, StringizedMacAddressLength);
        NN_LDN_LOG_DEBUG("received authentication request from %s\n",
            helper.ToString(request.ethernetHeader.src));

        // 認証を受け入れるか判定します。
        auto result = Judge(data, dataSize);

        // 未登録のステーションからの要求は無視します。
        if (result == AuthenticationResult_UnknownStation)
        {
            return;
        }

        // 送信データを用意します。
        auto& response = *reinterpret_cast<AuthenticationFrame*>(m_Buffer->send);
        response.authenticationHeader.result = static_cast<Bit8>(result);
        std::memcpy(response.authenticationHeader.clientRandom,
                    requestHeader.clientRandom, RandomSize);

        // ステーションに認証の結果を通知します。
        const auto& src = request.ethernetHeader.src;
        size_t responseSize = response.authenticationHeader.dataSize + AuthenticationFrameSizeMin;
        m_pSender->Send(
            &response.authenticationHeader,
            responseSize - NintendoEthernetFrameSizeMin,
            src, AuthenticationProtocolId);

        // 対象のクライアントを発見します。
        std::lock_guard<nn::os::Mutex> lock(m_ClientMutex);
        int i;
        for (i = 0; i < StationCountMax; ++i)
        {
            const auto& client = m_Clients[i];
            if (client.status != impl::AuthenticationState_Unregistered &&
                client.macAddress == src)
            {
                break;
            }
        }
        if (StationCountMax <= i)
        {
            return;
        }
        auto& client = m_Clients[i];

        // 認証成功の再送でない場合、認証の結果を保存します。
        if (client.status != impl::AuthenticationState_Processed)
        {
            std::memset(&m_History, 0, sizeof(AuthenticationHistroy));
            m_History.id = client.id;
            m_History.authenticationResult = static_cast<Bit8>(result);
            if (result == AuthenticationResult_Success)
            {
                client.status = impl::AuthenticationState_Processed;
                std::memcpy(m_History.clientRandom, requestHeader.clientRandom, RandomSize);
                std::memcpy(m_History.receivedData, request.payload, requestHeader.dataSize);
                m_History.receivedSize = static_cast<uint8_t>(requestHeader.dataSize);
            }
            m_HasHistory = true;
            m_AuthEvent.Signal();
        }
    }

    void ChallengeResponseAuthenticationServer::AuthThread(void* arg) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(arg);
        NN_LDN_LOG_DEBUG("AuthThread: started\n");
        auto& server = *static_cast<ChallengeResponseAuthenticationServer*>(arg);

        // このスレッドでは複数のイベントを同時に待機します。
        nn::os::MultiWaitType multiWait;
        nn::os::MultiWaitHolderType receivedHolder;
        nn::os::InitializeMultiWait(&multiWait);
        nn::os::InitializeMultiWaitHolder(&receivedHolder, server.m_ReceivedEvent.GetBase());
        nn::os::LinkMultiWaitHolder(&multiWait, &receivedHolder);

        // EndServer() が実行されるまでスレッドは生存します。
        const auto timeout = nn::TimeSpan::FromMilliSeconds(1000);
        for (;;)
        {
            // イベントの発生か、一定時間経過まで待機します。
            nn::os::TimedWaitAny(&multiWait, timeout);
            server.m_ReceivedEvent.Clear();

            // EndServer を呼ばれた場合にはスレッドを終了します。
            if (!server.m_IsRunning)
            {
                break;
            }

            // 期限切れになったクライアントを除去します。
            server.Expire();

            // 認証フレームを処理します。
            auto& queue = server.m_ReceiveQueue;
            while (!queue.IsEmpty() && !server.m_HasHistory)
            {
                size_t dataSize;
                const void* data = queue.GetFront(&dataSize);
                server.Reply(data, dataSize);
                queue.Dequeue();
            }
        }
        NN_LDN_LOG_DEBUG("AuthThread: finished\n");
    }

    size_t ChallengeResponseAuthenticationClient::GetRequiredBufferSize() NN_NOEXCEPT
    {
        return sizeof(impl::ChallengeResponseAuthenticationClientBuffer);
    }

    ChallengeResponseAuthenticationClient::ChallengeResponseAuthenticationClient(
        void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_ReceivedEvent(nn::os::EventClearMode_AutoClear),
          m_Buffer(static_cast<impl::ChallengeResponseAuthenticationClientBuffer*>(buffer)),
          m_pReceiver(nullptr),
          m_pSender(nullptr),
          m_IsRunning(false),
          m_IsAuthenticated(false)
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_ALIGNED(buffer, nn::os::ThreadStackAlignment);
        NN_SDK_ASSERT(GetRequiredBufferSize() <= bufferSize);
        NN_UNUSED(bufferSize);
        m_ReceiveQueue.Initialize(
            m_Buffer->receive, sizeof(m_Buffer->receive),
            sizeof(AuthenticationFrame), ClientReceiveQueueCapacity);
    }

    ChallengeResponseAuthenticationClient::~ChallengeResponseAuthenticationClient() NN_NOEXCEPT
    {
        if (m_IsRunning)
        {
            EndClient();
        }
        m_ReceiveQueue.Finalize();
    }

    void ChallengeResponseAuthenticationClient::BeginClient(
        const Bit8 (&clientRandom)[RandomSize],
        IFrameReceiver* pReceiver, IFrameSender* pSender) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsRunning);
        NN_SDK_ASSERT_NOT_NULL(clientRandom);
        NN_SDK_ASSERT_NOT_NULL(pReceiver);
        NN_SDK_ASSERT_NOT_NULL(pSender);

        // バッファを初期化しておきます。
        std::memset(m_Buffer, 0, sizeof(impl::ChallengeResponseAuthenticationServerBuffer));

        // 引数で受け取ったパラメータを保存します。
        m_pReceiver = pReceiver;
        m_pSender = pSender;

        // 送信フレームのテンプレート (固定部分) を生成します。
        auto& request = *reinterpret_cast<AuthenticationFrame*>(m_Buffer->send);
        request.authenticationHeader.version = CurrentVersion;
        std::memcpy(request.authenticationHeader.clientRandom, clientRandom, RandomSize);

        // 状態を記録しておきます。
        m_IsRunning = true;
        m_IsAuthenticated = false;
    }

    void ChallengeResponseAuthenticationClient::EndClient() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning);
        m_IsRunning = false;
        m_pSender = nullptr;
        m_pReceiver = nullptr;
    }

    void ChallengeResponseAuthenticationClient::SetData(
        const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning);
        NN_SDK_ASSERT_NOT_NULL(data);
        NN_SDK_ASSERT(dataSize <= AuthenticationFramePayloadSizeMax);
        auto& request = *reinterpret_cast<AuthenticationFrame*>(m_Buffer->send);
        request.authenticationHeader.dataSize = static_cast<uint8_t>(
            std::min(AuthenticationFramePayloadSizeMax, dataSize));
        std::memcpy(request.payload, data, request.authenticationHeader.dataSize);
    }

    AuthenticationResult ChallengeResponseAuthenticationClient::Authenticate(
        void* buffer, size_t* pOutSize, size_t bufferSize,
        MacAddress serverAddress, const NetworkId& networkId,
        const Bit8 (&serverRandom)[RandomSize]) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(m_IsRunning);
        NN_SDK_ASSERT(!m_IsAuthenticated);

        // 認証応答を受信できる状態にします。
        m_ReceivedEvent.Clear();
        m_pReceiver->Register(AuthenticationProtocolId, m_ReceivedEvent.GetBase(), &m_ReceiveQueue);

        // 認証要求を生成します。
        auto& request = *reinterpret_cast<AuthenticationFrame*>(m_Buffer->send);
        auto& requestHeader = request.authenticationHeader;
        std::memcpy(requestHeader.networkId, &networkId, sizeof(NetworkId));
        std::memcpy(requestHeader.serverRandom, serverRandom, RandomSize);

        // 認証要求に成功するまで最大 3 回リトライします。
        const int retryCount = 3;
        AuthenticationResult result = AuthenticationResult_Timeout;
        for (int i = 0; i < retryCount && (result == AuthenticationResult_Timeout ||
             result == AuthenticationResult_Forbidden); ++i)
        {
            // 認証要求を送信します。送信処理に失敗した場合には認証を中止します。
            size_t requestSize = requestHeader.dataSize + AuthenticationFrameSizeMin;
            size_t payloadSize = requestSize - NintendoEthernetFrameSizeMin;
            if (m_pSender->Send(
                &requestHeader, payloadSize, serverAddress, AuthenticationProtocolId).IsFailure())
            {
                break;
            }
            NN_LDN_LOG_DEBUG("succeeded to send authentication request\n");

            // 認証応答の受信まで待機します。タイムアウトした場合はリトライします。
            const auto timeout = nn::TimeSpan::FromMilliSeconds(700);
            const auto start = nn::os::GetSystemTick();
            auto diff = TimeSpan();
            while (diff < timeout && m_ReceivedEvent.TimedWait(timeout - diff))
            {
                // レスポンスを取得します。
                size_t frameSize;
                const void* data= m_ReceiveQueue.GetFront(&frameSize);
                const auto& response = *static_cast<const AuthenticationFrame*>(data);
                const auto& responseHeader = response.authenticationHeader;

                // 不正なレスポンスは無視します。
                const size_t calculatedPayloadSize = frameSize - AuthenticationFrameSizeMin;
                if (frameSize < AuthenticationFrameSizeMin)
                {
                    NN_LDN_LOG_WARN("authentication failed: bad response (too small)\n");
                }
                else if (!IsCompatibleVersion(responseHeader.version))
                {
                    NN_LDN_LOG_WARN("authentication failed: incompatible version\n");
                }
                else if (calculatedPayloadSize != responseHeader.dataSize ||
                         AuthenticationFramePayloadSizeMax <= responseHeader.dataSize)
                {
                    NN_LDN_LOG_WARN("authentication failed: bad response (payload size)\n");
                }
                else if (std::memcmp(responseHeader.networkId, &networkId, sizeof(NetworkId)) != 0)
                {
                    NN_LDN_LOG_WARN("authentication failed: bad response (network id)\n");
                }
                else if (std::memcmp(responseHeader.clientRandom,
                         requestHeader.clientRandom, RandomSize) != 0)
                {
                    NN_LDN_LOG_WARN("authentication failed: bad response (client random)\n");
                }
                else if (std::memcmp(responseHeader.serverRandom, serverRandom, RandomSize) != 0)
                {
                    NN_LDN_LOG_WARN("authentication failed: bad response (server random)\n");
                }
                else if (MakeVersion(0, 2) <= responseHeader.version &&
                        (responseHeader.flag & AuthenticationFlag_Response) == 0)
                {
                    NN_LDN_LOG_WARN("authentication failed: bad response (response flag)\n");
                }

                // 正常なレスポンスとして処理します。
                else
                {
                    result = static_cast<AuthenticationResult>(responseHeader.result);
                    if (result == AuthenticationResult_Success)
                    {
                        // 認証に成功しました。
                        *pOutSize = std::min<size_t>(responseHeader.dataSize, bufferSize);
                        std::memcpy(buffer, response.payload, *pOutSize);
                        m_IsAuthenticated = true;
                    }
                    NN_LDN_LOG_DEBUG("received authentication response: %d\n", responseHeader.result);
                    break;
                }

                // 送信後の経過時間を計算します。
                const auto end = nn::os::GetSystemTick();
                diff = (end - start).ToTimeSpan();
                m_ReceiveQueue.Dequeue();
            }
        }

        // 認証の結果を表示します。
        if (result == AuthenticationResult_Success)
        {
            NN_LDN_LOG_DEBUG("authentication succeeded\n");
        }
        else
        {
            NN_LDN_LOG_DEBUG("authentication rejected: %d\n", result);
        }

        // 認証応答の受信を終了します。
        m_pReceiver->Unregister(AuthenticationProtocolId);
        m_ReceiveQueue.Clear();
        return result;
    }

}}} // namespace nn::ldn::detail
