﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/lcs/lcs_Result.h>
#include <nn/lcs/lcs_PrivateDebugApi.h>
#include <nn/lcs/detail/lcs_Definition.h>
#include <nn/lcs/detail/lcs_NetworkApi.h>
#include <nn/lcs/detail/Debug/lcs_Log.h>
#include <nn/ldn/ldn_Api.h>
#include <nn/ldn/ldn_PrivateApi.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/ldn/ldn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Endian.h>

namespace nn { namespace lcs { namespace detail
{

    bool MakeSessionInfo(SessionInfo *pOutInfo, const nn::ldn::NetworkInfo& info) NN_NOEXCEPT
    {
        // セッション情報の作成
        AdvertiseData advertiseData = {};
        if (info.ldn.advertiseDataSize < sizeof(AdvertiseData))
        {
            NN_LCS_LOG_WARN("Failed to make session info: small advertise data\n");
            return false;
        }
        ::std::memcpy(&advertiseData, info.ldn.advertiseData, sizeof(AdvertiseData));

        // ホスト名の末尾が必ず 0 終端されるようにします
        ::std::strncpy(pOutInfo->hostUserName, advertiseData.hostUserName, UserNameBytesMax);
        pOutInfo->hostUserName[UserNameBytesMax] = 0;

        // コンテンツ情報に不正な値が含まれないことを検証しつつコピーします。
        if (SharableContentsCountMax < advertiseData.contentCount)
        {
            NN_LCS_LOG_WARN("Failed to make session info: contents count = %u\n",
                advertiseData.contentCount);
            return false;
        }
        pOutInfo->contentsCount = advertiseData.contentCount;
        for (unsigned j = 0; j < advertiseData.contentCount; ++j)
        {
            pOutInfo->contents[j].applicationId.value =
                nn::util::LoadBigEndian(&advertiseData.contents[j].id);
            pOutInfo->contents[j].attributeFlag =
                nn::util::LoadBigEndian(&advertiseData.contents[j].attributeFlag);
            ::memcpy(pOutInfo->contents[j].displayVersion,
                advertiseData.contents[j].displayVersion, DisplayVersionSizeMax);

            pOutInfo->contents[j].contentsFlag.Reset();
            if ((advertiseData.contents[j].contentsFlag & (0x01 << ContentsType::System::Index)) > 0)
            {
                pOutInfo->contents[j].contentsFlag.Set<ContentsType::System>(true);
            }
            if ((advertiseData.contents[j].contentsFlag & (0x01 << ContentsType::Application::Index)) > 0)
            {
                pOutInfo->contents[j].contentsFlag.Set<ContentsType::Application>(true);
            }
            if ((advertiseData.contents[j].contentsFlag & (0x01 << ContentsType::Patch::Index)) > 0)
            {
                pOutInfo->contents[j].contentsFlag.Set<ContentsType::Patch>(true);
            }
        }

        // 現在接続中の端末数に不正がないことを確認しつつコピーします。
        if (advertiseData.nodeCountMax == 0U || NodeCountMax < advertiseData.nodeCountMax)
        {
            NN_LCS_LOG_WARN("Failed to make session info: maximum node count = %u\n",
                advertiseData.nodeCountMax);
            return false;
        }
        else if (advertiseData.nodeCount == 0 ||
                 advertiseData.nodeCountMax < advertiseData.nodeCount)
        {
            NN_LCS_LOG_WARN("Failed to make session info: node count = %u/%u\n",
                advertiseData.nodeCount, advertiseData.nodeCountMax);
            return false;
        }
        pOutInfo->nodeCount = advertiseData.nodeCount;
        pOutInfo->nodeCountMax = advertiseData.nodeCountMax;

        // リンクレベルをコピーします
        pOutInfo->linkLevel = info.common.linkLevel;

        // コンテンツの成功・失敗をコピーします
        pOutInfo->isContentShareSucceeded = advertiseData.isContentShareSucceeded;

        // 接続設定の作成
        nn::ldn::NetworkConfig networkConfig = {};
        nn::ldn::SecurityParameter securityParam = {};
        size_t ServerRandoSize = 16; //sizeof(nn::ldn::SecurityParameter::_serverRandom)

        networkConfig.intentId = info.networkId.intentId;
        networkConfig.channel = info.common.channel;
        networkConfig.nodeCountMax = info.ldn.nodeCountMax;
        networkConfig.localCommunicationVersion = info.ldn.nodes[0].localCommunicationVersion;
        ::std::memcpy(securityParam._serverRandom, info.ldn.serverRandom, ServerRandoSize);
        securityParam._sessionId = info.networkId.sessionId;

        NetworkSetting setting = {};
        ::std::memcpy(setting.networkConfig, &networkConfig, NetworkConfigSize);
        ::std::memcpy(setting.securityParam, &securityParam, SecurityParameterSize);

        ::std::memcpy(pOutInfo->_internal, &setting, sizeof(NetworkSetting));

        return true;
    }

    bool IsNetworkAvailable() NN_NOEXCEPT
    {
        nn::ldn::State ldnState = nn::ldn::GetState();
        if (ldnState == nn::ldn::State_AccessPointCreated || ldnState == nn::ldn::State_StationConnected)
        {
            return true;
        }
        return false;
    }

    bool IsAccessPointOpened() NN_NOEXCEPT
    {
        nn::ldn::State ldnState = nn::ldn::GetState();
        if (ldnState == nn::ldn::State_AccessPointCreated || ldnState == nn::ldn::State_AccessPoint)
        {
            return true;
        }
        return false;
    }

    Result OpenAccessPoint() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("OpenAccessPoint\n");
        NN_RESULT_DO(ConvertLdnResult(nn::ldn::SetOperationMode(nn::ldn::OperationMode_HighSpeed)));
        NN_RESULT_DO(ConvertLdnResult(nn::ldn::OpenAccessPoint()));
        NN_RESULT_SUCCESS;
    }

    Result CloseAccessPoint() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("CloseAccessPoint\n");
        NN_RESULT_DO(ConvertLdnResult(nn::ldn::CloseAccessPoint()));
        NN_RESULT_SUCCESS;
    }

    Result OpenStation() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("OpenStation\n");
        NN_RESULT_DO(ConvertLdnResult(nn::ldn::SetOperationMode(nn::ldn::OperationMode_HighSpeed)));
        NN_RESULT_DO(ConvertLdnResult(nn::ldn::OpenStation()));
        NN_RESULT_SUCCESS;
    }

    Result CloseStation() NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("CloseStation\n");
        NN_RESULT_DO(ConvertLdnResult(nn::ldn::CloseStation()));
        NN_RESULT_SUCCESS;
    }

    Result SetAdvertiseDataForNetwork(const void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(data);

        NN_RESULT_DO(ConvertLdnResult(nn::ldn::SetAdvertiseData(data, dataSize)));
        NN_RESULT_SUCCESS;
    }

    Result CreateNetwork(Bit64 networkId, int nodeCountMax, const Version& version) NN_NOEXCEPT
    {
        NN_LCS_LOG_DEBUG("CreateNetwork\n");

        nn::ldn::NetworkConfig  networkConfig = {};
        nn::ldn::SecurityConfig securityConfig = {};
        nn::ldn::UserConfig     userConfig = {};
        nn::ldn::IntentId intentId = nn::ldn::MakeIntentId(networkId, GetSceneId());

        networkConfig.intentId = intentId;
        networkConfig.channel = nn::ldn::AutoChannel;
        networkConfig.nodeCountMax = static_cast<uint8_t>(nodeCountMax);
        networkConfig.localCommunicationVersion = static_cast<int16_t>(GetMajorVersion(version));
        securityConfig.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
        securityConfig.passphraseSize = sizeof(Passphrase);
        std::memcpy(securityConfig.passphrase, Passphrase, sizeof(Passphrase));
        std::strncpy(userConfig.userName, HostUserName, nn::ldn::UserNameBytesMax);

        return ConvertLdnResult(nn::ldn::CreateNetwork(networkConfig, securityConfig, userConfig));
    }

    Result CreateNetworkPrivate(
        const NetworkSetting& setting, AddressEntry* entry, int entryCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(entry);

        NN_LCS_LOG_DEBUG("CreateNetworkPrivate\n");

        nn::ldn::NetworkConfig  networkConfig = {};
        nn::ldn::SecurityConfig securityConfig = {};
        nn::ldn::SecurityParameter securityParam = {};
        nn::ldn::UserConfig userConfig = {};
        nn::ldn::AddressEntry addressEntry[NodeCountMax] = {};

        securityConfig.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
        securityConfig.passphraseSize = sizeof(Passphrase);
        ::std::memcpy(securityConfig.passphrase, Passphrase, sizeof(Passphrase));
        ::std::strncpy(userConfig.userName, HostUserName, nn::ldn::UserNameBytesMax);
        ::std::memcpy(&networkConfig, &setting.networkConfig, NetworkConfigSize);
        ::std::memcpy(&securityParam, &setting.securityParam, SecurityParameterSize);

        for (int i = 0; i < entryCount; ++i)
        {
            addressEntry[i].ipv4Address.raw = entry[i].ipv4Address;
            for (int j = 0; j < 6; ++j)
            {
                addressEntry[i].macAddress.raw[j] = entry[i].macAddress.raw[j];
            }
        }

        return ConvertLdnResult(nn::ldn::CreateNetworkPrivate(
            networkConfig, securityConfig, securityParam, userConfig, entryCount, addressEntry));
    }

    Result Scan(SessionInfo* outBuffer, int* pOutCount, int bufferCount, Bit64 networkId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(outBuffer);
        NN_SDK_ASSERT_NOT_NULL(pOutCount);

        nn::ldn::NetworkInfo info[nn::ldn::ScanResultCountMax];
        int infoCount = 0;
        nn::ldn::ScanFilter filter = {};
        nn::ldn::IntentId intentId = nn::ldn::MakeIntentId(networkId, GetSceneId());

        filter.networkType = nn::ldn::NetworkType_Ldn;
        filter.networkId.intentId = intentId;
        filter.flag = nn::ldn::ScanFilterFlag_IntentId | nn::ldn::ScanFilterFlag_NetworkType;

        NN_RESULT_DO(ConvertLdnResult(nn::ldn::Scan(
            info, &infoCount, nn::ldn::ScanResultCountMax, filter, nn::ldn::AutoChannel)));

        int sessionCount = 0;
        for (int i = 0; i < infoCount; ++i)
        {
            AdvertiseData* advertiseData =
                reinterpret_cast<AdvertiseData*>(info[i].ldn.advertiseData);

            if (advertiseData->sessionAttribute == SessionAttribute_Open)
            {
                // ToDo : 子機対応するまで、子機は見えないようにしておく
                if (advertiseData->contents[0].contentsFlag == (0x01 << ContentsType::Application::Index) ||
                    advertiseData->contents[1].contentsFlag == (0x01 << ContentsType::Application::Index))
                {
                    continue;
                }
                // ToDo : ここまで子機未対応の対応

                // 不正なアドバータイズはスキャン結果に含まれないようにします。
                if (MakeSessionInfo(&outBuffer[sessionCount], info[i]))
                {
                    ++sessionCount;
                }
            }

            if (sessionCount == bufferCount)
            {
                break;
            }
        }
        *pOutCount = sessionCount;
        NN_RESULT_SUCCESS;
    }


    Result Connect(const NetworkSetting& setting, int16_t version) NN_NOEXCEPT
    {
        nn::ldn::NetworkConfig networkConfig = {};
        nn::ldn::SecurityParameter securityParam = {};
        nn::ldn::SecurityConfig securityConfig = {};
        nn::ldn::UserConfig userConfig = {};

        securityConfig.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
        securityConfig.passphraseSize = sizeof(Passphrase);
        ::std::memcpy(securityConfig.passphrase, Passphrase, sizeof(Passphrase));
        ::std::strncpy(userConfig.userName, ClientUserName, nn::ldn::UserNameBytesMax);
        ::std::memcpy(&networkConfig, &setting.networkConfig, NetworkConfigSize);
        ::std::memcpy(&securityParam, &setting.securityParam, SecurityParameterSize);

        NN_LCS_LOG_DEBUG("/** Connect Setting **/\n");
        NN_LCS_LOG_DEBUG("-- NetworkConfig --\n");
        NN_LCS_LOG_DEBUG("channel                   : %d\n", networkConfig.channel);
        NN_LCS_LOG_DEBUG("localCommunicationId      : %d\n", networkConfig.intentId.localCommunicationId);
        NN_LCS_LOG_DEBUG("sceneId                   : %d\n", networkConfig.intentId.sceneId);
        NN_LCS_LOG_DEBUG("localCommunicationVersion : %d\n", networkConfig.localCommunicationVersion);
        NN_LCS_LOG_DEBUG("nodeCountMax              : %d\n", networkConfig.nodeCountMax);
        NN_LCS_LOG_DEBUG("-- SecurityParameter --\n");
        NN_LCS_LOG_DEBUG("_serverRandom\n");
        for (int i = 0; i < 16; ++i)
        {
            NN_LCS_LOG_DEBUG(" %x\n", securityParam._serverRandom[i]);
        }
        NN_LCS_LOG_DEBUG("_sessionId.random\n");
        for (int i = 0; i < 16; ++i)
        {
            NN_LCS_LOG_DEBUG(" %x\n", securityParam._sessionId.random[i]);
        }

        return ConvertLdnResult(nn::ldn::ConnectPrivate(networkConfig, securityConfig,
            securityParam, userConfig, version, nn::ldn::ConnectOption_None));
    }

    Result Disconnect() NN_NOEXCEPT
    {
        NN_RESULT_DO(ConvertLdnResult(nn::ldn::Disconnect()));
        NN_RESULT_SUCCESS;
    }

    Result RejectOnNetwork(Ipv4Address address) NN_NOEXCEPT
    {
        nn::ldn::Ipv4Address ldnAddress;
        ldnAddress.raw = address;

        return ConvertLdnResult(nn::ldn::Reject(ldnAddress));
    }

    Result GetConnectedNodeIpAddress(int* pOutCount, Ipv4Address* ipAddress, int ipAddressCount) NN_NOEXCEPT
    {
        nn::ldn::NetworkInfo networkInfo = {};

        NN_RESULT_DO(
            ConvertLdnResult(nn::ldn::GetNetworkInfo(&networkInfo)));

        *pOutCount = networkInfo.ldn.nodeCount;
        for (int i = 0; i < ::std::min(*pOutCount, ipAddressCount); ++i)
        {
            ipAddress[i] = networkInfo.ldn.nodes[i].ipv4Address.raw;
        }

        NN_RESULT_SUCCESS;
    }

    Result GetSessionInfoFromNetwork(SessionInfo* pOutInfo) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutInfo);

        nn::ldn::NetworkInfo networkInfo = {};

        NN_RESULT_DO(
            ConvertLdnResult(nn::ldn::GetNetworkInfo(&networkInfo)));

        if (!MakeSessionInfo(pOutInfo, networkInfo))
        {
            // TODO: 適切な Result に置き換えた方が良いが、
            // とりあえず nn::ldn::ResultInvalidState と同様にしておく
            return nn::lcs::ResultLdnNotInitialized();
        }

        NN_RESULT_SUCCESS;
    }

    Result GetAdvertiseData(AdvertiseData* pOutData) NN_NOEXCEPT
    {
        nn::ldn::NetworkInfo networkInfo = {};

        NN_RESULT_DO(
            ConvertLdnResult(nn::ldn::GetNetworkInfo(&networkInfo)));

        ::std::memcpy(pOutData, networkInfo.ldn.advertiseData, sizeof(AdvertiseData));

        NN_RESULT_SUCCESS;
    }

    Result GetAccessPointIpv4Address(Ipv4Address* pOutAddress) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutAddress);

        nn::ldn::NetworkInfo networkInfo = {};

        NN_RESULT_DO(
            ConvertLdnResult(nn::ldn::GetNetworkInfo(&networkInfo)));

        *pOutAddress = networkInfo.ldn.nodes[0].ipv4Address.raw;

        NN_RESULT_SUCCESS;
    }

    Result GetMyIpv4Address(Ipv4Address* pOutAddress) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutAddress);

        nn::ldn::Ipv4Address address = {};
        nn::ldn::SubnetMask subnetMask = {};

        NN_RESULT_DO(
            ConvertLdnResult(nn::ldn::GetIpv4Address(&address, &subnetMask)));

        *pOutAddress = address.raw;

        NN_RESULT_SUCCESS;
    }

    Result GetMacAddress(MacAddress* pOutMacaddress, Ipv4Address address) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutMacaddress);

        nn::ldn::NetworkInfo networkInfo = {};

        NN_RESULT_DO(
            ConvertLdnResult(nn::ldn::GetNetworkInfo(&networkInfo)));

        for (int i = 0; i < networkInfo.ldn.nodeCount; ++i)
        {
            if (address == networkInfo.ldn.nodes[i].ipv4Address.raw)
            {
                for (int j = 0; j < 6; ++j)
                {
                    pOutMacaddress->raw[j] = networkInfo.ldn.nodes[i].macAddress.raw[j];
                }
                NN_RESULT_SUCCESS;
            }
        }
        return ResultNodeNotFound();
    }

    Result GetConnectedNodesCount(int *pOutCount) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(pOutCount);

        nn::ldn::NetworkInfo networkInfo = {};

        NN_RESULT_DO(
            ConvertLdnResult(nn::ldn::GetNetworkInfo(&networkInfo)));

        *pOutCount = networkInfo.ldn.nodeCount;

        NN_RESULT_SUCCESS;
    }

    Result ConvertLdnResult(Result result)  NN_NOEXCEPT
    {
        if (result.IsSuccess())
        {
            NN_RESULT_SUCCESS;
        }

        if (nn::ldn::ResultWifiOff::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultWifiOff\n");
            return ResultWifiOff();
        }
        else if (nn::ldn::ResultSleep::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultSleep\n");
            return ResultSleep();
        }
        else if (nn::ldn::ResultDeviceDisabled::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultDeviceDisabled\n");
            return ResultDeviceDisabled();
        }
        else if (nn::ldn::ResultDeviceOccupied::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultDeviceOccupied\n");
            return ResultDeviceOccupied();
        }
        else if (nn::ldn::ResultDeviceNotAvailable::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultDeviceNotAvailable\n");
            return ResultDeviceDisabled();
        }

        else if (nn::ldn::ResultInvalidState::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultInvalidState\n");
            return ResultLdnNotInitialized();
        }

        else if (nn::ldn::ResultNodeNotFound::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultNodeNotFound\n");
            return ResultNodeNotFound();
        }

        else if (nn::ldn::ResultNetworkNotFound::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultNetworkNotFound\n");
            return ResultSessionNotFound();
        }
        else if (nn::ldn::ResultConnectionTimeout::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultConnectionTimeout\n");
            return ResultCommunicationError();
        }
        else if (nn::ldn::ResultConnectionRejected::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultConnectionRejected\n");
            return ResultConnectionFailed();
        }
        else if (nn::ldn::ResultInvalidNetwork::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultInvalidNetwork\n");
            return ResultConnectionFailed();
        }
        else if (nn::ldn::ResultConnectionFailed::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultConnectionFailed\n");
            return ResultConnectionFailed();
        }

        else if (nn::ldn::ResultCancelled::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultCancelled\n");
            return ResultInvalidState();
        }

        else if (nn::ldn::ResultLowerVersion::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultLowerVersion\n");
            return ResultLowerVersion();
        }
        else if (nn::ldn::ResultHigherVersion::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultHigherVersion\n");
            return ResultHigherVersion();
        }

        else if (nn::ldn::ResultNodeCountLimitation::Includes(result))
        {
            //NN_LCS_LOG_DEBUG("ResultNodeCountLimitation\n");
            return ResultNodeCountLimitation();
        }

        return ResultInvalidState();
    }

    ContentsShareFailureReason GetContentsShareFailureReasonFromLdn() NN_NOEXCEPT
    {
        ContentsShareFailureReason retReason = ContentsShareFailureReason_None;
        nn::ldn::DisconnectReason ldnReason = nn::ldn::GetDisconnectReason();

        NN_LCS_LOG_DEBUG("GetDisconnectReason : %d\n", ldnReason);

        switch (ldnReason)
        {
        case nn::ldn::DisconnectReason_Unknown:
            retReason = ContentsShareFailureReason_NodeLeaved;
            break;
        case nn::ldn::DisconnectReason_None:
            break;
        case nn::ldn::DisconnectReason_DisconnectedByUser:
            retReason = ContentsShareFailureReason_CommunicationError;
            break;
        case nn::ldn::DisconnectReason_DisconnectedBySystem:
            retReason = ContentsShareFailureReason_CommunicationError;
            break;
        case nn::ldn::DisconnectReason_DestroyedByUser:
            retReason = ContentsShareFailureReason_NodeLeaved;
            break;
        case nn::ldn::DisconnectReason_DestroyedBySystem:
            retReason = ContentsShareFailureReason_NodeLeaved;
            break;
        case nn::ldn::DisconnectReason_Rejected:
            retReason = ContentsShareFailureReason_CommunicationError;
            break;
        case nn::ldn::DisconnectReason_SignalLost:
            retReason = ContentsShareFailureReason_CommunicationError;
            break;
        default:
            break;
        }

        return retReason;
    }

}}} // end of namespace nn::lcs
