﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/lcs/lcs_DebugApi.h>
#include <nn/lcs/detail/lcs_Definition.h>
#include <nn/lcs/detail/lcs_SystemShareApi.h>
#include <nn/lcs/detail/lcs_Version.h>
#include <nn/lcs/detail/Debug/lcs_Log.h>
#include <nn/lcs/detail/Packet/lcs_Packet.h>
#include <nn/lcs/detail/Packet/lcs_PacketSender.h>
#include <nn/lcs/detail/Packet/lcs_PacketUtilty.h>

namespace nn { namespace lcs { namespace detail
{

    void BuildHeader(void* packet, Type type, size_t length) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(packet);

        CommonHeader* pHeader = reinterpret_cast<CommonHeader*>(packet);

        pHeader->version = static_cast<uint8_t>((ProtocolMajorVersion << 4) | ProtocolMinorVersion);
        pHeader->type = static_cast<uint8_t>(type);
        uint32_t len = static_cast<uint32_t>(length - sizeof(CommonHeader));
        pHeader->length = nn::socket::InetHtonl(len);
    }

    ssize_t Send(int socket, void* buffer, size_t bufferSize, nn::socket::MsgFlag flag) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);
        NN_SDK_ASSERT_LESS_EQUAL(bufferSize, TcpDataSizeMax);

        int count = 0;
        ssize_t sendSize = static_cast<ssize_t>(bufferSize);
        while (count < SendRetryCountMax)
        {
            ssize_t retSize = nn::socket::Send(socket, buffer, bufferSize, flag);
            //NN_LCS_LOG_DEBUG("Send retSize[%d]: %d / %d\n", count, retSize, bufferSize);
            if (retSize == sendSize)
            {
                //NN_LCS_LOG_DEBUG("Send ok : %d\n", retSize);
                return retSize;
            }
            else
            {
                if (nn::socket::GetLastError() == nn::socket::Errno::EPipe || nn::socket::GetLastError() == nn::socket::Errno::EBadf)
                {
                    NN_LCS_LOG_DEBUG("Send Error EPIPE/EBADF\n");
                    return -1;
                }
                NN_LCS_LOG_DEBUG("Send Error : %d\n", nn::socket::GetLastError());
            }
            ++count;
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(SendTimeoutMicroSeconds));
        }
        //NN_LCS_LOG_DEBUG("Send Error\n");
        return -1;
    }

    ssize_t SendApplicationControlDataReq(
        int socket, const ncm::ApplicationId& id, bool isSizeOnly) NN_NOEXCEPT
    {
        AppControlDataReqPacket packet = {};
        size_t size = sizeof(AppControlDataReqPacket);
        Type type = Type_ApplicationControlDataReq;
        if (isSizeOnly)
        {
            type = Type_ApplicationControlDataSizeReq;
        }

        BuildHeader(&packet, type, size);
        packet.appUpperId = nn::socket::InetHtonl(GetUpper(id.value));
        packet.appLowerId = nn::socket::InetHtonl(GetLower(id.value));

        return Send(socket, &packet, size, nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendApplicationControlDataRes(
        int socket, const ncm::ApplicationId& id, size_t dataSize, bool isSizeOnly) NN_NOEXCEPT
    {
        AppControlDataResPacket packet = {};
        size_t size = sizeof(AppControlDataResPacket);
        if (isSizeOnly)
        {
            BuildHeader(&packet, Type_ApplicationControlDataSizeRes, size);
        }
        else
        {
            BuildHeader(&packet, Type_ApplicationControlDataRes, size + dataSize);
        }

        packet.appUpperId = nn::socket::InetHtonl(GetUpper(id.value));
        packet.appLowerId = nn::socket::InetHtonl(GetLower(id.value));
        packet.dataSize = nn::socket::InetHtonl(static_cast<uint32_t>(dataSize));

        return Send(socket, &packet, size, nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendApplicationControlData(int socket, void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(data);

        ssize_t retSize = 0;
        ssize_t completedSize = 0;
        size_t sendSize = 0;
        bool endFlag = false;

        while (!endFlag)
        {
            if ((dataSize - completedSize) > TcpDataSizeMax)
            {
                sendSize = TcpDataSizeMax;
            }
            else
            {
                sendSize = dataSize - completedSize;
                endFlag = true;
            }

            retSize = Send(socket, reinterpret_cast<uint8_t*>(data) + completedSize, sendSize, nn::socket::MsgFlag::Msg_None);
            if (retSize <= 0)
            {
                return -1;
            }
            completedSize += retSize;
        }
        return retSize;
    }

    ssize_t SendJoin(int socket, char* name,
        const Version& lcsVersion, int appCount, size_t appInfoSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(name);

        NodeDetailInfoPacket packet = {};

        packet.lcsVersion = lcsVersion._raw;
        packet.appCount = static_cast<uint8_t>(appCount);
        ::std::memcpy(&packet.userName, name, UserNameBytesMax + 1);
        SystemDeliveryInfo sysInfo;
        GetSystemDeliveryInfo(&sysInfo);
        ::std::memcpy(&packet.sysInfo, &sysInfo, sizeof(SystemDeliveryInfo));

        BuildHeader(&packet, Type_Join, sizeof(NodeDetailInfoPacket) + appInfoSize);

        return Send(socket, &packet, sizeof(NodeDetailInfoPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendUpdate(int socket, char* name,
        const Version& lcsVersion, const SystemDeliveryInfo& sysInfo,
        int appCount, int downloadedContentCount, size_t appInfoSize) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_NOT_NULL(name);

        NodeDetailInfoPacket packet = {};

        packet.lcsVersion = lcsVersion._raw;
        packet.appCount = static_cast<uint8_t>(appCount);
        packet.downloadedContentCount = static_cast<uint8_t>(downloadedContentCount);
        ::std::memcpy(&packet.userName, name, UserNameBytesMax + 1);
        ::std::memcpy(&packet.sysInfo, &sysInfo, sizeof(SystemDeliveryInfo));

        BuildHeader(&packet, Type_Update, sizeof(NodeDetailInfoPacket) + appInfoSize);

        return Send(socket, &packet, sizeof(NodeDetailInfoPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendLeave(int socket, LeaveReason reason) NN_NOEXCEPT
    {
        LeavePacket packet = {};

        packet.leaveReason = static_cast<uint8_t>(reason);
        BuildHeader(&packet, Type_Leave, sizeof(LeavePacket));

        return Send(socket, &packet, sizeof(LeavePacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendAccept(int socket, uint32_t index) NN_NOEXCEPT
    {
        AcceptPacket packet = {};

        packet.index = nn::socket::InetHtonl(index);
        BuildHeader(&packet, Type_Accept, sizeof(AcceptPacket));

        return Send(socket, &packet, sizeof(AcceptPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendReject(int socket, RejectReason reason) NN_NOEXCEPT
    {
        RejectPacket packet = {};

        packet.rejectReason = static_cast<uint8_t>(reason);
        BuildHeader(&packet, Type_Reject, sizeof(RejectPacket));

        return Send(socket, &packet, sizeof(RejectPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendSessionInfoHeader(
        int socket, int appCount, int nodeCount, size_t packetSize) NN_NOEXCEPT
    {
        SessionInfoPacket packet = {};

        packet.appCount = static_cast<uint8_t>(appCount);
        packet.nodeCount = static_cast<uint8_t>(nodeCount);
        BuildHeader(&packet, Type_SessionInfo, sizeof(SessionInfoPacket) + packetSize);

        return Send(socket, &packet, sizeof(SessionInfoPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendAppInfo(int socket, const ApplicationInfo& appInfo) NN_NOEXCEPT
    {
        AppInfoPacket packet = {};

        packet.appDeliveryInfoCount = static_cast<uint8_t>(appInfo.deliveryInfoCount);
        ::std::memcpy(packet.detailInfo, &appInfo.detailInfo, ApplicationDetailInfoSize);

        return Send(socket, &packet, sizeof(AppInfoPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendAppDeliveryInfo(int socket, const ApplicationDeliveryInfo& deliveryInfo) NN_NOEXCEPT
    {
        AppDeliveryInfoPacket packet = {};

        ::std::memcpy(packet.deliveryInfo, deliveryInfo.appDeliveryInfo, ApplicationDeliveryInfoSize);

        return Send(socket, &packet, sizeof(AppDeliveryInfoPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendNodeInfo(int socket, const NodeInfo& nodeInfo) NN_NOEXCEPT
    {
        NodeInfoPacket packet = {};

        packet.index = nn::socket::InetHtonl(nodeInfo.index);
        ::std::memcpy(packet.userName, nodeInfo.userName, UserNameBytesMax + 1);

        return Send(socket, &packet, sizeof(NodeInfoPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendStart(int socket) NN_NOEXCEPT
    {
        StartContentsSarePacket packet = {};

        BuildHeader(&packet, Type_StartContentsSare, sizeof(StartContentsSarePacket));

        return Send(socket, &packet, sizeof(StartContentsSarePacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendMigration(int socket, const MigrationInfo& info) NN_NOEXCEPT
    {
        MigrationPacket packet = {};

        ::std::memcpy(packet.migrationInfo, &info, MigrationInfoSize);

        BuildHeader(&packet, Type_Migration, sizeof(MigrationPacket));

        return Send(socket, &packet, sizeof(MigrationPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendSystemShareClearPacket(int socket, SystemUpdateReason reason) NN_NOEXCEPT
    {
        SystemShareClearPacket packet = {};

        packet.systemUpdateReason = static_cast<uint8_t>(reason);
        BuildHeader(&packet, Type_ShareSystemClear, sizeof(SystemShareClearPacket));

        return Send(socket, &packet, sizeof(SystemShareClearPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendSystemShareInfoPacket(int socket, Type type, TransferRole role,
        uint32_t index, Ipv4Address address, const SystemDeliveryInfo& sysInfo, int channel) NN_NOEXCEPT
    {
        SystemShareInfoPacket packet = {};

        packet.role = static_cast<uint8_t>(role);
        packet.connectTimeout = static_cast<uint8_t>(WaitReceiverTimeSeconds);
        packet.channel = static_cast<uint8_t>(channel);
        packet.nodeIndex = nn::socket::InetHtonl(index);
        packet.address = nn::socket::InetHtonl(address);

        ::std::memcpy(packet.sysInfo, sysInfo.sysDeliveryInfo, SystemDeliveryInfoSize);
        BuildHeader(&packet, type, sizeof(SystemShareInfoPacket));

        return Send(socket, &packet, sizeof(SystemShareInfoPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendSystemShareResPacket(int socket, Type type, Response response,
        const SystemDeliveryInfo& sysInfo) NN_NOEXCEPT
    {
        SystemShareResPacket packet = {};

        packet.response = static_cast<uint8_t>(response);
        ::std::memcpy(packet.sysInfo, sysInfo.sysDeliveryInfo, SystemDeliveryInfoSize);
        BuildHeader(&packet, type, sizeof(SystemShareResPacket));

        return Send(socket, &packet, sizeof(SystemShareResPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendContentShareClearPacket(int socket,
        nn::util::BitFlagSet<8, ContentsType> contentsFlag, Bit64 id) NN_NOEXCEPT
    {
        ContentShareClearPacket packet = {};

        packet.contentsFlag |= (0x01 << ContentsType::Patch::Index);
        packet.appUpperId = nn::socket::InetHtonl(GetUpper(id));
        packet.appLowerId = nn::socket::InetHtonl(GetLower(id));
        BuildHeader(&packet, Type_ShareContentClear, sizeof(ContentShareClearPacket));

        return Send(socket, &packet, sizeof(ContentShareClearPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendContentShareInfoPacket(int socket, Type type,
        TransferRole role, uint32_t index,
        Ipv4Address address, int contentMetaCount, Bit64 id, int channel, size_t packetSize) NN_NOEXCEPT
    {
        ContentShareInfoPacket packet = {};

        packet.role = static_cast<uint8_t>(role);
        packet.contentsFlag |= (0x01 << ContentsType::Patch::Index);
        packet.connectTimeout = static_cast<uint8_t>(WaitReceiverTimeSeconds);
        packet.channel = static_cast<uint8_t>(channel);
        packet.contentMetaKeyCount = static_cast<uint8_t>(contentMetaCount);
        packet.nodeIndex = nn::socket::InetHtonl(index);
        packet.address = nn::socket::InetHtonl(address);
        packet.appUpperId = nn::socket::InetHtonl(GetUpper(id));
        packet.appLowerId = nn::socket::InetHtonl(GetLower(id));
        BuildHeader(&packet, type, sizeof(ContentShareInfoPacket) + packetSize);

        return Send(socket, &packet, sizeof(ContentShareInfoPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendContentShareResPacket(int socket, Type type,
        Response response, int contentMetaCount, size_t packetSize) NN_NOEXCEPT
    {
        ContentShareResPacket packet = {};

        packet.response = static_cast<uint8_t>(response);
        packet.contentMetaKeyCount = static_cast<uint8_t>(contentMetaCount);
        BuildHeader(&packet, type, sizeof(ContentShareResPacket) + packetSize);

        return Send(socket, &packet, sizeof(ContentShareResPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendContentMetaKey(int socket, const ContentMetaKey& metaKey) NN_NOEXCEPT
    {
        ContentMetaKeyPacket packet = {};

        ::std::memcpy(packet.contentMetaKey, metaKey.key, ContentMetaKeySize);

        return Send(socket, &packet, sizeof(ContentMetaKeyPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendSuspend(int socket) NN_NOEXCEPT
    {
        SuspendPacket packet = {};

        BuildHeader(&packet, Type_Suspend, sizeof(SuspendPacket));

        return Send(socket, &packet, sizeof(SuspendPacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendResume(int socket) NN_NOEXCEPT
    {
        ResumePacket packet = {};

        BuildHeader(&packet, Type_Resume, sizeof(ResumePacket));

        return Send(socket, &packet, sizeof(ResumePacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendEnd(int socket) NN_NOEXCEPT
    {
        EndContentsSarePacket packet = {};

        BuildHeader(&packet, Type_EndContentShare, sizeof(EndContentsSarePacket));

        return Send(socket, &packet, sizeof(EndContentsSarePacket), nn::socket::MsgFlag::Msg_None);
    }

    ssize_t SendEndWithProgress(int socket, const AdvertiseData& advertiseData) NN_NOEXCEPT
    {
        EndContentsShareWithProgressPacket packet = {};

        ::std::memcpy(packet.progress, advertiseData.progress, sizeof(AdvertiseNodeProgress) * NodeCountMax);

        BuildHeader(&packet, Type_EndContentShare, sizeof(EndContentsShareWithProgressPacket));

        return Send(socket, &packet, sizeof(EndContentsShareWithProgressPacket), nn::socket::MsgFlag::Msg_None);
    }

}}} // end of namespace nn::lcs
