﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/irsensor/irsensor_ResultPrivate.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/hid/system/hid_Irsensor.h>

#include "irsensor_IrCameraHandle.h"
#include "irsensor_ResourceManager.h"
#include "irsensor_StaticObject.h"
#include "../../hid/detail/hid_NpadId.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include "../../hid/detail/hid_ResourceManager-os.win.h"
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "../../hid/detail/hid_ResourceManager-os.horizon.h"
#else
    #error   "未サポートの OS 種別が指定されています。"
#endif

namespace nn { namespace irsensor { namespace detail {

// IR センサーのRequestHandlerスレッドの優先度
const int ResourceManager::RequestHandlerTaskThreadPriority =
        NN_SYSTEM_THREAD_PRIORITY(irsensor, RequestHandler);

ResourceManager::ResourceManager() NN_NOEXCEPT
: m_DeviceCount(0)
, m_StatusManagerHolder()
, m_AppletResourceMutex(false)
, m_AppletResourceIdMutex(false)
, m_AppletResourceManager()
{
    for (auto i = 0; i < ::nn::hid::system::IrSensorSupportedNpadIdsCount; ++i)
    {
        m_RequestHandlerTasks[i].SetThreadStack(s_RequestHandlerTaskThreadStacks[i],
            sizeof(s_RequestHandlerTaskThreadStacks[i]));
        m_RequestHandlerTasks[i].SetThreadPriority(RequestHandlerTaskThreadPriority);
        m_RequestHandlerTasks[i].SetDriverManager(&m_DriverManagers[i]);
        m_DriverManagers[i].BindIrSensorEvent();
    }
    // 全 Npad 用のリソースの確保
    m_StatusManagerHolder.Initialize();
}

ResourceManager::~ResourceManager() NN_NOEXCEPT
{
    // 全 Npad 用のリソースの破棄
    m_StatusManagerHolder.Finalize();
}

::nn::Result ResourceManager::AssertValidIrCameraHandle(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    auto deviceType = GetIrCameraHandleDeviceType(handle);

    NN_SDK_REQUIRES_RANGE(playerNumber, 0, ::nn::hid::system::IrSensorSupportedNpadIdsCount);

    NN_RESULT_THROW_UNLESS(
        0 <= playerNumber && playerNumber < ::nn::hid::system::IrSensorSupportedNpadIdsCount,
        ResultIrsensorInvalidHandle());

    switch (deviceType)
    {
    case IrSensorDeviceType::Npad:
        break;
    default:
        return ResultIrsensorInvalidHandle();
    }

    NN_RESULT_SUCCESS;
}

RequestHandlerTask& ResourceManager::GetRequestHandlerTask(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    // TODO: RequestHandlerTask に問い合わせ IrSensorDeviceType が一致しなければ失敗とする
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    NN_SDK_REQUIRES_RANGE(playerNumber, 0, ::nn::hid::system::IrSensorSupportedNpadIdsCount);

    return m_RequestHandlerTasks[playerNumber];
}

::nn::Result ResourceManager::GetIrCameraHandle(
    IrCameraHandle* outValue,
    const ::nn::hid::NpadIdType& npadIdType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);

    MakeIrCameraHandle(outValue, npadIdType);
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::BindIrCameraXcdDriver(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(m_DeviceCount, 0, ::nn::hid::system::IrSensorSupportedNpadIdsCount);

    // TODO: 別プロセスにするときには IPC が必要
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    auto deviceType = GetIrCameraHandleDeviceType(handle);
    ::nn::hid::detail::IrSensorXcdDriver* pDriver = nullptr;
    switch (deviceType)
    {
    case IrSensorDeviceType::Npad:
        {
            pDriver = ::nn::hid::detail::GetResourceManager().GetNpadResourceManager().GetIrSensorXcdDriver(::nn::hid::system::IrSensorSupportedNpadIds[playerNumber]);
        }
        break;
    default:
        return ResultIrsensorInvalidHandle();
    }
    NN_SDK_ASSERT_NOT_NULL(pDriver);
    pDriver->Initialize();
    m_DriverManagers[playerNumber].SetDriver(pDriver);
    m_DriverManagers[playerNumber].SetStatusManagerHolder(&m_StatusManagerHolder);
    m_DriverManagers[playerNumber].SetIrCameraHandle(handle);
    m_DriverManagers[playerNumber].SetAppletResourceManager(&m_AppletResourceMutex, &m_AppletResourceManager);

    m_DeviceCount++;

    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::UnbindIrCameraXcdDriver(
    const IrCameraHandle& handle) NN_NOEXCEPT
{
    // TODO: 別プロセスにするときには IPC が必要
    auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
    auto deviceType = GetIrCameraHandleDeviceType(handle);
    ::nn::hid::detail::IrSensorXcdDriver* pDriver = nullptr;
    switch (deviceType)
    {
    case IrSensorDeviceType::Npad:
        {
            pDriver = ::nn::hid::detail::GetResourceManager().GetNpadResourceManager().GetIrSensorXcdDriver(::nn::hid::system::IrSensorSupportedNpadIds[playerNumber]);
        }
        break;
    default:
        return ResultIrsensorInvalidHandle();
    }
    // 終了処理をリソースマネージャ内で行っているが、
    // 実際に登録されたドライバマネージャから解除されるのは、ドライバマネージャの Deactivate 関数内
    NN_SDK_ASSERT_NOT_NULL(pDriver);
    pDriver->Finalize();
    m_DeviceCount--;

    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::GetSharedMemoryHandle(
    ::nn::os::NativeHandle* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    *outValue = m_StatusManagerHolder.GetSharedMemoryHandle();
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::ClearStatusManager() NN_NOEXCEPT
{
    NN_RESULT_DO(m_StatusManagerHolder.Clear());
    NN_RESULT_SUCCESS;
}

void ResourceManager::SetAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceIdMutex)
                      > lockerId(m_AppletResourceIdMutex);

    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);

    m_AppletResourceManager.SetAppletResourceUserId(aruid);
}

::nn::Result ResourceManager::RegisterAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid,
    bool enablesInput) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);

    return m_AppletResourceManager.RegisterAppletResourceUserId(aruid, enablesInput);
}

void ResourceManager::UnregisterAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceIdMutex)
                      > lockerId(m_AppletResourceIdMutex);

    bool isDeviceUsed = false;
    auto currentAruid = ::nn::applet::AppletResourceUserId::GetInvalidId();
    {
        ::std::lock_guard<decltype(m_AppletResourceMutex)
        > locker(m_AppletResourceMutex);

        // アプレットが IR センサーを使用中だったかどうか
        isDeviceUsed = m_AppletResourceManager.IsAppletResourceUsed(aruid) && (m_DeviceCount != 0);
        if (isDeviceUsed)
        {
            // 現在の aruid を 退避
            currentAruid = m_AppletResourceManager.GetAppletResourceUserId();
            // 一時的にアプリの aruid を登録
            m_AppletResourceManager.SetAppletResourceUserId(aruid);
        }
    }

    if (isDeviceUsed)
    {
        // 終了処理がされないままアプリが終了した場合は、ここで終了処理を行う
        for (auto i = 0; i < ::nn::hid::system::IrSensorSupportedNpadIdsCount; i++)
        {
            IrCameraHandle handle;
            MakeIrCameraHandle(&handle, ::nn::hid::system::IrSensorSupportedNpadIds[i]);
            AssertValidIrCameraHandle(handle);
            // StopAsync する。 Run していなかった場合は何もしない。
            GetRequestHandlerTask(handle).StopImageProcessorAsync(aruid);
            // Deactivate で Stop の完了を待つ
            GetRequestHandlerTask(handle).Deactivate(aruid);
            // Xcd Driver を Handle と離して、終了処理をしておく。
            UnbindIrCameraXcdDriver(handle);
            // FunctionLevel を初期値に戻す
            PackedFunctionLevel functionLevel;
            functionLevel.level = 0;
            GetRequestHandlerTask(handle).SetFunctionLevel(aruid, functionLevel);
        }
    }

    {
        ::std::lock_guard<decltype(m_AppletResourceMutex)
        > locker(m_AppletResourceMutex);

        if (isDeviceUsed)
        {
            // 元の aruid に戻す
            m_AppletResourceManager.SetAppletResourceUserId(currentAruid);
        }

        m_AppletResourceManager.UnregisterAppletResourceUserId(aruid);
    }
}

void ResourceManager::EnableAppletToGetInput(
    ::nn::applet::AppletResourceUserId aruid,
    bool enablesInput) NN_NOEXCEPT
{
    NN_UNUSED(aruid);
    NN_UNUSED(enablesInput);
    // 将来的にAll-FGとPartial-FG を分ける必要が出た場合に対応
}

void ResourceManager::ResetAppletResourceEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);

    m_AppletResourceManager.ResetAppletResourceEntry(aruid);
}

::nn::Result ResourceManager::SetAppletResourceEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);

    return m_AppletResourceManager.SetAppletResourceEntry(aruid, m_StatusManagerHolder.GetStatusManager());
}

NN_ALIGNAS(0x1000) char
    ResourceManager::s_RequestHandlerTaskThreadStacks[::nn::hid::system::IrSensorSupportedNpadIdsCount][0x1000];

ResourceManager& GetResourceManager() NN_NOEXCEPT
{
    return StaticObject<ResourceManager>::Get();
}

}}} // namespace nn::irsensor::detail
