﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/irsensor/irsensor_IrCameraTypes.h>
#include <nn/irsensor/irsensor_PackedTypes.h>

#include "irsensor_DriverManager.h"
#include "irsensor_Task.h"

namespace nn { namespace irsensor { namespace detail {

//!< DriverManager のタスクを扱うクラスです。
class RequestHandlerTask final : public Task
{
private:
    enum class IrSensorMode
    {
        None,
        Moment,
        Clustering,
        ImageTransfer,
        Pointing,
        TeraPlugin,
        IrLed,
    };

    enum class RequestType
    {
        Stop,
        Suspend,
        Run,
        Resume,
        None,
    };

private:
    //!< DriverManager 操作排他用のミューテックス
    ::nn::os::Mutex m_ManagerMutex;

    //!< IR センサーのタイマイベント
    MultiWaitTimerEvent m_TimerEvent;

    //!< IR センサーマネージャ
    DriverManager* m_pManager;

    //!< IR センサーマネージャが初期化されたかどうか
    bool m_IsManagerInitialized;

    //!< 処理の要求を排他するためのミューテックス
    //!< (ManagerMutexとのデッドロックを避けるため、常に Manager->Requestの順にロックする)
    ::nn::os::Mutex m_RequestMutex;

    //!< クライアントに要求されたモード
    IrSensorMode m_RequestMode;

    //!< 現在のモード
    IrSensorMode m_CurrentMode;

    //!< クライアントからの Run 要求を伝えるイベント
    MultiWaitEvent m_RunRequestEvent;

    //!< クライアントからの Resume 要求を伝えるイベント
    MultiWaitEvent m_ResumeRequestEvent;

    //!< クライアントからの Stop 要求を伝えるイベント
    MultiWaitEvent m_StopRequestEvent;

    //!< クライアントからの Suspend 要求を伝えるイベント
    MultiWaitEvent m_SuspendRequestEvent;

    //!< McuError の連続発生回数を管理するカウンタ
    int m_McuErrorCounter;

    //!< Mcu のバージョン取得のリクエストフラグ
    bool m_IsMcuVersionRequested;

    //!< ライブラリからの要求バージョン
    PackedMcuVersion m_RequiredVersion;

    //!< 最新の要求の種類
    RequestType m_LatestRequestType;

public:
    RequestHandlerTask() NN_NOEXCEPT;

    virtual ~RequestHandlerTask() NN_NOEXCEPT NN_OVERRIDE;

    void SetDriverManager(DriverManager* pManager) NN_NOEXCEPT;

    ::nn::Result Activate() NN_NOEXCEPT;

    ::nn::Result Activate(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedFunctionLevel& functionLevel) NN_NOEXCEPT;

    ::nn::Result Deactivate(
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    ::nn::Result RequestFirmwareVersion(
        const PackedMcuVersion& requiredVersion) NN_NOEXCEPT;

    ::nn::Result SetFunctionLevel(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedFunctionLevel& functionLevel) NN_NOEXCEPT;

    ::nn::Result StopImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    ::nn::Result StopImageProcessorAsync(
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    ::nn::Result SuspendImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    ::nn::Result StopSampling(
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    bool IsProcessorStopped() NN_NOEXCEPT;

    ::nn::Result WaitForProcessorStopped(int waitPacketCount) NN_NOEXCEPT;

    ::nn::Result RunMomentProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedMomentProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RunClusteringProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedClusteringProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RunImageTransferProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedImageTransferProcessorExConfig& config,
        ::nn::os::NativeHandle&& transferMemoryHandle,
        size_t size,
        bool isManaged
        ) NN_NOEXCEPT;

    ::nn::Result RunPointingProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedPointingProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RunTeraPluginProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedTeraPluginProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RunIrLedProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedIrLedProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result GetImageTransferProcessorState(
        ImageTransferProcessorState* pOutState,
        void* pOutImage,
        const ::nn::applet::AppletResourceUserId& aruid,
        size_t size
        ) NN_NOEXCEPT;

private:
    ::nn::Result RequestRunImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedMomentProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RequestRunImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedClusteringProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RequestRunImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedImageTransferProcessorExConfig& config,
        ::nn::os::NativeHandle&& transferMemoryHandle,
        size_t size,
        bool isManaged
        ) NN_NOEXCEPT;

    ::nn::Result RequestRunImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedPointingProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RequestRunImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedTeraPluginProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RequestRunImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedIrLedProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RequestResumeImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedMomentProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RequestResumeImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedClusteringProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RequestResumeImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedImageTransferProcessorExConfig& config,
        ::nn::os::NativeHandle&& transferMemoryHandle,
        size_t size,
        bool isManaged,
        bool isTransferMemoryHandled
        ) NN_NOEXCEPT;

    ::nn::Result RequestResumeImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedPointingProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RequestResumeImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedTeraPluginProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result RequestResumeImageProcessor(
        const ::nn::applet::AppletResourceUserId& aruid,
        const PackedIrLedProcessorConfig& config
        ) NN_NOEXCEPT;

    ::nn::Result CheckFirmwareVersion() NN_NOEXCEPT;

    void ReleaseTransferMemory(
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

protected:
    //!< スレッド名を登録します。
    virtual void SetThreadName(::nn::os::ThreadType* pThreadType
                                ) NN_NOEXCEPT NN_OVERRIDE;

    //!< イベントをリンクします。
    virtual void LinkEvent(::nn::os::MultiWaitType* pMultiWait
                                ) NN_NOEXCEPT NN_OVERRIDE;

    //!< イベントをアンリンクします。
    virtual void UnlinkEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< クライアントからの要求の対応とタイマ要求によるサンプリング処理を実行します。
    virtual void HandleEvent(const ::nn::os::MultiWaitHolderType* waitId
                            ) NN_NOEXCEPT NN_OVERRIDE;
};

}}} // namespace nn::irsensor::detail
