﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/irsensor/irsensor_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "irsensor_AppletResourceManager.h"
#include "irsensor_IrCameraHandle.h"

namespace nn { namespace irsensor { namespace detail {

AppletResourceManager::AppletResourceManager() NN_NOEXCEPT
    : m_AppletResourceUserId(::nn::applet::AppletResourceUserId::GetInvalidId())
    , m_ActivationCount()
    , m_StatusManagerHolder()
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        entry = AppletResourceEntry();
    }
}

::nn::applet::AppletResourceUserId
    AppletResourceManager::GetAppletResourceUserId() NN_NOEXCEPT
{
    return m_AppletResourceUserId;
}

void AppletResourceManager::SetAppletResourceUserId(
    ::nn::applet::AppletResourceUserId value) NN_NOEXCEPT
{
    m_AppletResourceUserId = value;

    // FG/BG フラグを更新
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>())
        {
            if (m_AppletResourceUserId == entry.aruid)
            {
                entry.flags.Set<AppletResourceFlag::IsForeground>();
            }
            else
            {
                entry.flags.Reset<AppletResourceFlag::IsForeground>();
            }
        }
    }
}

const AppletResourceEntry (&AppletResourceManager::GetAppletResourceEntries(
    ) NN_NOEXCEPT)[AppletResourceEntryCountMax]
{
    return m_Entries;
}

::nn::Result AppletResourceManager::RegisterAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid,
    bool enablesInput) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        NN_RESULT_THROW_UNLESS(
            !entry.flags.Test<AppletResourceFlag::IsRegistered>() ||
            entry.aruid != aruid,
            ::nn::irsensor::ResultAppletResourceDuplicateUserId());
    }

    for (AppletResourceEntry& entry : m_Entries)
    {
        if (!entry.flags.Test<AppletResourceFlag::IsRegistered>())
        {
            entry.flags.Set<AppletResourceFlag::IsRegistered>();
            entry.flags.Set<AppletResourceFlag::EnablesInput>(enablesInput);
            entry.aruid = aruid;

            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultAppletResourceTableOverflow());
}

void AppletResourceManager::UnregisterAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    this->ResetAppletResourceEntry(aruid);

    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            entry.flags.Reset();
            entry.aruid = ::nn::applet::AppletResourceUserId::GetInvalidId();
        }
    }
}

::nn::Result AppletResourceManager::SetAppletResourceEntry(
    ::nn::applet::AppletResourceUserId aruid,
    StatusManager* address) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(address);

    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            NN_RESULT_THROW_UNLESS(
                !entry.flags.Test<AppletResourceFlag::IsAvailable>(),
                ::nn::irsensor::ResultAppletResourceDuplicateUserId());

            entry.flags.Set<AppletResourceFlag::IsAvailable>();
            entry.address = address;
            // 参照カウントを aruid と紐づけて保存する
            ++m_ActivationCount;
            entry.clientStackLevel = m_ActivationCount.GetCount();
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

void AppletResourceManager::ResetAppletResourceEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.flags.Test<AppletResourceFlag::IsAvailable>() &&
            entry.aruid == aruid)
        {
            entry.flags.Reset<AppletResourceFlag::IsAvailable>();
            entry.address = nullptr;
            --m_ActivationCount;
            entry.clientStackLevel = 0;
            memset(&entry.deviceResource, 0, sizeof(entry.deviceResource));
        }
    }
}

bool AppletResourceManager::IsAppletResourceAvailable() NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.flags.Test<AppletResourceFlag::IsAvailable>() &&
            entry.aruid == m_AppletResourceUserId)
        {
            return true;
        }
    }
    return false;
}

bool AppletResourceManager::IsAppletResourceUsed(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.flags.Test<AppletResourceFlag::IsAvailable>() &&
            entry.aruid == aruid)
        {
            return true;
        }
    }
    return false;
}

::nn::Result AppletResourceManager::SetImageValidity(
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle,
    bool isValid) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
            entry.deviceResource[playerNumber].imageTransferStatus.isImageValid = isValid;
        }
    }
    NN_RESULT_SUCCESS;

}

::nn::Result AppletResourceManager::SetImageTransferState(
    IrCameraHandle handle,
    ImageTransferProcessorState state) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == m_AppletResourceUserId)
        {
            auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
            entry.deviceResource[playerNumber].imageTransferStatus.imageTransferState = state;
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::SetImageTransferBuffer(
    IrCameraHandle handle,
    void* address) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == m_AppletResourceUserId)
        {
            auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
            entry.deviceResource[playerNumber].imageTransferStatus.imageBufferAddress = address;
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::SetIrSensorConfig(
    const ::nn::applet::AppletResourceUserId& aruid,
    IrCameraHandle handle,
    IrSensorConfig config) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
            entry.deviceResource[playerNumber].config = config;
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::GetIrSensorDeviceResource(
    IrSensorDeviceResource* pDeviceResource,
    const ::nn::applet::AppletResourceUserId& aruid,
    IrCameraHandle handle) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
            *pDeviceResource = entry.deviceResource[playerNumber];
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::GetIrSensorMode(
    IrSensorMode* pOutMode,
    const ::nn::applet::AppletResourceUserId& aruid,
    IrCameraHandle handle) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
            *pOutMode = entry.deviceResource[playerNumber].mode;
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::SetIrSensorMode(
    ::nn::applet::AppletResourceUserId aruid,
    IrSensorMode mode,
    IrCameraHandle handle) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
            entry.deviceResource[playerNumber].mode = mode;
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::GetIrSensorFunctionLevel(
    ::nn::applet::AppletResourceUserId aruid,
    PackedFunctionLevel* pOutLevel,
    IrCameraHandle handle) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
            *pOutLevel = entry.deviceResource[playerNumber].functionLevel;
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::SetIrSensorFunctionLevel(
    ::nn::applet::AppletResourceUserId aruid,
    PackedFunctionLevel functionLevel,
    IrCameraHandle handle) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
            entry.deviceResource[playerNumber].functionLevel = functionLevel;
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::applet::AppletResourceUserId AppletResourceManager::GetLatestActivatedAruid() NN_NOEXCEPT
{
    // 最新の aruid を検索
    auto latestAruid = ::nn::applet::AppletResourceUserId::GetInvalidId();
    auto latestLevel = 0;

    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.flags.Test<AppletResourceFlag::IsAvailable>() &&
            entry.clientStackLevel >= latestLevel)
        {
            latestLevel = entry.clientStackLevel;
            latestAruid = entry.aruid;
        }
    }
    return latestAruid;
}

::nn::Result AppletResourceManager::GetTransferMemoryType(
    ::nn::os::TransferMemoryType** pTransferMemory,
    ::nn::applet::AppletResourceUserId aruid,
    IrCameraHandle handle) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
            *pTransferMemory = &entry.deviceResource[playerNumber].imageTransferStatus.transferMemory;
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::SetTransferMemoryWorkBufferAddress(
    ::nn::applet::AppletResourceUserId aruid,
    void* address,
    IrCameraHandle handle) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            auto playerNumber = GetIrCameraHandlePlayerNumber(handle);
            entry.deviceResource[playerNumber].imageTransferStatus.workBufferAddress = address;
        }
    }
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::irsensor::detail
