﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/applet/applet.h>

#include "hidbus_StaticObject.h"
#include "../detail/hidbus_PollingDataAccessor.h"
#include "../detail/hidbus_StatusManager.h"
#include "../detail/hidbus_InternalUtility.h"
#include "../detail/hidbus_InternalDefinition.h"

namespace nn { namespace hidbus { namespace server {

struct HidbusTransferMemoryHolder
{
    nn::os::TransferMemoryType transferMemoryType;
    void* pMappedAddress;
    size_t size;
};

class ResourceHolder final
{
    NN_DISALLOW_COPY(ResourceHolder);
    NN_DISALLOW_MOVE(ResourceHolder);

private:
    //!< 新規に構築されたか否か
    bool m_IsCreated;

    //!< 共有メモリオブジェクト
    ::nn::os::SharedMemoryType m_SharedMemory;

    //!< TransferMemory の Holder
    HidbusTransferMemoryHolder m_TransferMemoryHolder[nn::hidbus::detail::MaxHidbusNumber];

    //!< Initialize にした ARUID のリスト (TORIAEZU : アプリ分だけ。NOTE : アプリの生存期間中は一度登録されるとずっと有効)
    //!< 現時点では、 Initialize だけ呼ばれた状態でアプリが終了した際に、内部スレッドが動き続けてしまう問題のために使用する
    nn::applet::AppletResourceUserId m_InitializedAruid[nn::hidbus::detail::MaxHidbusNumber];

    //!< Enable にした ARUID のリスト (TORIAEZU : アプリ分だけ。NOTE : アプリの生存期間中は一度登録されるとずっと有効)
    nn::applet::AppletResourceUserId m_EnabledAruid[nn::hidbus::detail::MaxHidbusNumber];

    //!< TransferMermoy の Map / Unmap を守るための mutex
    nn::os::SdkMutexType m_TransferMemoryMutex[nn::hidbus::detail::MaxHidbusNumber];

    //!< 現在、 FG にいるアプレットの ARUID
    nn::applet::AppletResourceUserId m_InFocusAruid;

    //!< index に対する、リソース回りが初期化済みかどうかのフラグ
    bool m_IsResourceInitialized[nn::hidbus::detail::MaxHidbusNumber];

    //!< TaskScheduler やコントローラー側のステート変化を守るための Mutex
    nn::os::SdkRecursiveMutexType m_InternalStateMutex[nn::hidbus::detail::MaxHidbusNumber];

public:
    ResourceHolder() NN_NOEXCEPT;

    ~ResourceHolder() NN_NOEXCEPT;

    ::nn::Result Initialize() NN_NOEXCEPT;

    void Finalize() NN_NOEXCEPT;

    ::nn::Result ClearSharedMemory() NN_NOEXCEPT;

    void AttachSharedMemory(::nn::os::NativeHandle handle, bool managed) NN_NOEXCEPT;

    nn::hidbus::detail::StatusManager* GetStatusManager() const NN_NOEXCEPT;

    size_t GetSharedMemorySize() const NN_NOEXCEPT;

    ::nn::os::NativeHandle GetSharedMemoryHandle() const NN_NOEXCEPT;

    bool IsSharedMemoryMapped() const NN_NOEXCEPT;

    void SetTransferMemory(void* pMappedAddress, size_t size, nn::hidbus::BusHandle handle) NN_NOEXCEPT
    {
        auto index = detail::GetInternalIndexFromHandle(handle);
        m_TransferMemoryHolder[index].pMappedAddress = pMappedAddress;
        m_TransferMemoryHolder[index].size = size;
    }

    bool IsTransferMemoryMapped(nn::hidbus::BusHandle handle) NN_NOEXCEPT
    {
        auto index = detail::GetInternalIndexFromHandle(handle);
        return m_TransferMemoryHolder[index].pMappedAddress != nullptr;
    }

    nn::os::TransferMemoryType* GetTransferMemoryType(nn::hidbus::BusHandle handle) NN_NOEXCEPT
    {
        auto index = detail::GetInternalIndexFromHandle(handle);
        return &m_TransferMemoryHolder[index].transferMemoryType;
    }

    template<typename PollingDataLifoFormatT, int PollingDataSize>
    nn::hidbus::detail::PollingDataAccessor<PollingDataLifoFormatT, PollingDataSize>* GetPollingDataAccessor(int index) NN_NOEXCEPT;

    void SetAppletResourceUserId(nn::applet::AppletResourceUserId aruid, nn::hidbus::BusHandle handle) NN_NOEXCEPT
    {
        auto index = detail::GetInternalIndexFromHandle(handle);
        m_EnabledAruid[index] = aruid;
    }

    void SetInitializedAppletResourceUserId(nn::applet::AppletResourceUserId aruid, nn::hidbus::BusHandle handle) NN_NOEXCEPT
    {
        auto index = detail::GetInternalIndexFromHandle(handle);
        m_InitializedAruid[index] = aruid;
    }

    nn::applet::AppletResourceUserId GetAppletResourceUserId(int index) NN_NOEXCEPT
    {
        return m_EnabledAruid[index];
    }

    nn::applet::AppletResourceUserId GetInitializedAppletResourceUserId(int index) NN_NOEXCEPT
    {
        return m_InitializedAruid[index];
    }

    void SetInFocusAruid(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
    {
        m_InFocusAruid = aruid;
    }

    nn::applet::AppletResourceUserId GetInFocusAruid() NN_NOEXCEPT
    {
        return m_InFocusAruid;
    }

    nn::os::SdkMutexType* GetTransferMemoryMutexPointer(int index) NN_NOEXCEPT
    {
        return &m_TransferMemoryMutex[index];
    }

    bool IsResourceInitialized(int index) NN_NOEXCEPT
    {
        return m_IsResourceInitialized[index];
    }

    void SetResourceInitialized(bool isInitialized, int index) NN_NOEXCEPT
    {
        m_IsResourceInitialized[index] = isInitialized;
    }

    void LockInternalStateMutex(int index) NN_NOEXCEPT
    {
        m_InternalStateMutex[index].Lock();
    }

    void UnlockInternalStateMutex(int index) NN_NOEXCEPT
    {
        m_InternalStateMutex[index].Unlock();
    }
};

ResourceHolder& GetResourceHolder() NN_NOEXCEPT;

nn::hidbus::detail::JoyEnableSixAxisPollingDataAccessor* GetJoyEnableSixAxisPollingDataAccessor(int index) NN_NOEXCEPT;
nn::hidbus::detail::JoyDisableSixAxisPollingDataAccessor* GetJoyDisableSixAxisPollingDataAccessor(int index) NN_NOEXCEPT;

}}} // namespace nn::hidbus::server
