﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <climits>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace hidbus { namespace server {

//!< アクティブ化回数を扱うクラスです。
class ActivationCount final
{
    NN_DISALLOW_COPY(ActivationCount);
    NN_DISALLOW_MOVE(ActivationCount);

private:
    //!< アクティブ化回数の上限
    const int CountMax;

    //!< アクティブ化回数
    ::std::atomic<int> m_Count;

public:
    ActivationCount() NN_NOEXCEPT
        : CountMax(INT_MAX - 1)
        , m_Count(0)
    {
        // 何もしない
    }

    ~ActivationCount() NN_NOEXCEPT { /* 何もしない */ }

    //!< アクティブ化回数が 0 か否かを表す値です。
    bool IsZero() const NN_NOEXCEPT
    {
        return (m_Count == 0);
    }

    //!< アクティブ化回数が上限に達しているか否かを表す値です。
    bool IsMax() const NN_NOEXCEPT
    {
        return (m_Count == CountMax);
    }

    //!< アクティブ化回数をインクリメントします。
    int operator++() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(m_Count, 0, CountMax);
        return ++m_Count;
    }

    //!< アクティブ化回数をデクリメントします。
    int operator--() NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(m_Count, 1, CountMax + 1);
        return --m_Count;
    }

    //!< アクティブ化回数を強制的に 0 にします。
    void SetZero() NN_NOEXCEPT
    {
        m_Count = 0;
    }
};

}}} // namespace nn::hidbus::server
