﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os/os_Mutex.h>
#include <nn/hid/hid_VibrationMixer.h>
#include "detail/hid_LockableMutexType.h"

namespace {

//!< VibrationMixer 内の排他処理に用いるミューテックス
nn::hid::detail::LockableMutexType s_VibrationMixerMutex = { NN_OS_MUTEX_INITIALIZER(false) };

}

namespace nn { namespace hid {

VibrationMixer::VibrationMixer() NN_NOEXCEPT :
    m_MixMode(VibrationMixMode_MaxAmplitudePerSubband) {}

VibrationMixer::~VibrationMixer() NN_NOEXCEPT {}

void VibrationMixer::SetMixMode(VibrationMixMode mixMode) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationMixerMutex)> lock(s_VibrationMixerMutex);
    m_MixMode = mixMode;
}

VibrationMixMode VibrationMixer::GetMixMode() const NN_NOEXCEPT
{
    return m_MixMode;
}

void VibrationMixer::OnNextSampleRequired(
    VibrationValue* pValue,
    VibrationNodeConnection::List* pInputConnections) NN_NOEXCEPT
{
    std::lock_guard<decltype(s_VibrationMixerMutex)> lock(s_VibrationMixerMutex);

    // いったん内部値をゼロにします
    *pValue = VibrationValue::Make();

    switch(m_MixMode)
    {
    case VibrationMixMode_None:
        // 何もしません
        break;
    case VibrationMixMode_MaxAmplitude:
        {
            for(auto& c : *pInputConnections)
            {
                // 接続元から自分向けの振動値を取得します
                VibrationValue v = c.GetSource()->GetCurrentVibration();
                c.GetModulation().ApplyModulation(&v);

                // 振幅合計値が現在の値以上なら採用
                if(v.amplitudeLow + v.amplitudeHigh >= pValue->amplitudeLow + pValue->amplitudeHigh)
                {
                    *pValue = v;
                }
            }
        }
        break;
    case VibrationMixMode_MaxAmplitudePerSubband:
        {
            for(auto& c : *pInputConnections)
            {
                // 接続元から自分向けの振動値を取得します
                VibrationValue v = c.GetSource()->GetCurrentVibration();
                c.GetModulation().ApplyModulation(&v);

                // 低周波帯で Max をとる処理
                if(v.amplitudeLow >= pValue->amplitudeLow)
                {
                    pValue->amplitudeLow = v.amplitudeLow;
                    pValue->frequencyLow = v.frequencyLow;
                }

                // 高周波帯で Max をとる処理
                if(v.amplitudeHigh >= pValue->amplitudeHigh)
                {
                    pValue->amplitudeHigh = v.amplitudeHigh;
                    pValue->frequencyHigh = v.frequencyHigh;
                }
            }
        }
        break;
    case VibrationMixMode_AmplitudeSum:
        {
            float freqSumLow = 0.0f;
            float freqSumHigh = 0.0f;

            for (auto& c : *pInputConnections)
            {
                // 接続元から自分向けの振動値を取得します
                VibrationValue v = c.GetSource()->GetCurrentVibration();
                c.GetModulation().ApplyModulation(&v);

                // 振幅は単純に加算します
                pValue->amplitudeLow += v.amplitudeLow;
                pValue->amplitudeHigh += v.amplitudeHigh;

                // 周波数はあとで加重平均値を算出するために振幅で重み付けした総和を計算します
                freqSumLow += v.amplitudeLow * v.frequencyLow;
                freqSumHigh += v.amplitudeHigh * v.frequencyHigh;
            }

            // 周波数は振幅で重み付けした加重平均値を採用します
            if (pValue->amplitudeLow > 0.0f)
            {
                pValue->frequencyLow = freqSumLow / pValue->amplitudeLow;
            }
            if (pValue->amplitudeHigh > 0.0f)
            {
                pValue->frequencyHigh = freqSumHigh / pValue->amplitudeHigh;
            }
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}} // namespace nn::hid
