﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>
#include <nn/settings/system/settings_BluetoothDevices.h>

#include "hid_XcdDriver.h"

namespace nn { namespace hid { namespace detail {

XcdDriver::XcdDriver() NN_NOEXCEPT
    : m_ThreadMutex(false)
{
    // 何もしない
}

XcdDriver::~XcdDriver() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result XcdDriver::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMultiWait);

    ::std::lock_guard<decltype(m_ThreadMutex)> locker(m_ThreadMutex);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultResourceManagerActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // Xcdを初期化
        ::nn::xcd::Initialize(m_pMultiWait);

        // Uart を有効化
# ifndef ENABLE_EDEV_LOG
        ::nn::xcd::EnableNwcp();
# endif
    }

    // アクティブ化回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result XcdDriver::Deactivate() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ThreadMutex)> locker(m_ThreadMutex);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultResourceManagerDeactivationLowerLimitOver());

    // アクティブ化回数をデクリメント
    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        ::nn::xcd::Finalize();

    }

    NN_RESULT_SUCCESS;
}

void XcdDriver::LinkMultiWait(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMultiWait);

    m_pMultiWait = pMultiWait;
}

void XcdDriver::UnlinkMultiWait() NN_NOEXCEPT
{
    m_pMultiWait = nullptr;
}

bool XcdDriver::Proceed(const ::nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT
{
    return nn::xcd::Proceed(pHolder);
}

void XcdDriver::ProceedPeriodic() NN_NOEXCEPT
{
    nn::xcd::ProceedPeriodic();
}

nn::os::MultiWaitHolderType* XcdDriver::WaitEvent() NN_NOEXCEPT
{
    return nn::xcd::WaitEvent();
}

void XcdDriver::StartButtonPairing() NN_NOEXCEPT
{
    nn::xcd::StartButtonPairing();
}

Result XcdDriver::Suspend() NN_NOEXCEPT
{
    ::nn::xcd::SuspendForSleep();

    NN_RESULT_SUCCESS;
}

Result XcdDriver::Resume() NN_NOEXCEPT
{
    ::nn::xcd::ResumeFromSleep();

    NN_RESULT_SUCCESS;
}

Result XcdDriver::EnableWiredDeviceRegistration() NN_NOEXCEPT
{
    nn::xcd::EnableWiredDeviceRegistration();

    NN_RESULT_SUCCESS;
}

Result XcdDriver::DisableWiredDeviceRegistration() NN_NOEXCEPT
{
    nn::xcd::DisableWiredDeviceRegistration();

    NN_RESULT_SUCCESS;
}


}}} // namespace nn::hid::detail
