﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Windows.h>
#include <nn/hid/hid_DebugPad.h>
#include <nn/hid/hid_Keyboard.h>

#include "hid_WindowsKeyboard-os.win.h"
#include "hid_WindowsRawInput-os.win.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 仮想キーコードと対応するスキャンコードを返します。
int MapVirtualKeyToScanCode(int keyCode) NN_NOEXCEPT
{
    switch (keyCode)
    {
    case VK_PRIOR:
        return 0xE049;
    case VK_NEXT:
        return 0xE051;
    case VK_END:
        return 0xE04F;
    case VK_HOME:
        return 0xE047;
    case VK_LEFT:
        return 0xE04B;
    case VK_UP:
        return 0xE048;
    case VK_RIGHT:
        return 0xE04D;
    case VK_DOWN:
        return 0xE050;
    case VK_SNAPSHOT:
        return 0xE037;
    case VK_INSERT:
        return 0xE052;
    case VK_DELETE:
        return 0xE053;
    default:
        return ::MapVirtualKey(keyCode, MAPVK_VK_TO_VSC_EX);
    }
}

//!< スキャンコード順に並んだ HID Usage Index の配列
const int ScanCodeToHidUsageIndexMap[] = {
    0x00, 0x29, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23,
    0x24, 0x25, 0x26, 0x27, 0x2D, 0x2E, 0x2A, 0x2B,
    0x14, 0x1A, 0x08, 0x15, 0x17, 0x1C, 0x18, 0x0C,
    0x12, 0x13, 0x2F, 0x30, 0x28, 0xE0, 0x04, 0x16,
    0x07, 0x09, 0x0A, 0x0B, 0x0D, 0x0E, 0x0F, 0x33,
    0x34, 0x35, 0xE1, 0x31, 0x1D, 0x1B, 0x06, 0x19,
    0x05, 0x11, 0x10, 0x36, 0x37, 0x38, 0xE5, 0x55,
    0xE2, 0x2C, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E,
    0x3F, 0x40, 0x41, 0x42, 0x43, 0x53, 0x47, 0x5F,
    0x60, 0x61, 0x56, 0x5C, 0x5D, 0x5E, 0x57, 0x59,
    0x5A, 0x5B, 0x62, 0x63, 0x00, 0x00, 0x64, 0x44,
    0x45, 0x67, 0x00, 0x00, 0x8C, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x68, 0x69, 0x6A, 0x6B,
    0x6C, 0x6D, 0x6E, 0x6F, 0x70, 0x71, 0x72, 0x00,
    0x88, 0x00, 0x00, 0x87, 0x00, 0x00, 0x73, 0x93,
    0x92, 0x8A, 0x00, 0x8B, 0x00, 0x89, 0x85, 0x00
};

/**
 * @brief       スキャンコードと対応する HID Usage Index を返します。
 *
 * @details     USB HID to PS/2 Scan Code Translation Table 2004年版に基づき、
 *              スキャンコードと対応する HID Usage Index を返します。
 *
 * @param[in]   scanCode                    スキャンコード
 *
 * @return      スキャンコードと対応する HID Usage Index です。
 */
int MapScanCodeToHidUsageIndex(int scanCode) NN_NOEXCEPT
{
    if (scanCode >= 0 &&
        scanCode < sizeof(ScanCodeToHidUsageIndexMap) / sizeof(int))
    {
        return ScanCodeToHidUsageIndexMap[scanCode];
    }

    switch (scanCode)
    {
    case 0x00F1: return 0x91;
    case 0x00F2: return 0x90;
    case 0xE01C: return 0x58;
    case 0xE01D: return 0xE4;
    case 0xE035: return 0x54;
    case 0xE037: return 0x46;
    case 0xE038: return 0xE6;
    case 0xE046: return 0x48;
    case 0xE047: return 0x4A;
    case 0xE048: return 0x52;
    case 0xE049: return 0x4B;
    case 0xE04B: return 0x50;
    case 0xE04D: return 0x4F;
    case 0xE04F: return 0x4D;
    case 0xE050: return 0x51;
    case 0xE051: return 0x4E;
    case 0xE052: return 0x49;
    case 0xE053: return 0x4C;
    case 0xE05B: return 0xE3;
    case 0xE05C: return 0xE7;
    case 0xE05D: return 0x65;
    case 0xE05E: return 0x66;
    case 0xE11D: return 0x48;
    default: return 0x00;
    }
}

//!< 有効な仮想キーコードの配列
const int VirtualKeyCodes[] = {
    0x03, 0x08, 0x09, 0x0C, 0x0D, 0x13, 0x14, 0x15,
    0x17, 0x18, 0x19, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
    0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46,
    0x47, 0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E,
    0x4F, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56,
    0x57, 0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0x5F,
    0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
    0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F,
    0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
    0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F,
    0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
    0x90, 0x91, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5,
    0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD,
    0xAE, 0xAF, 0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5,
    0xB6, 0xB7, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF,
    0xC0, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE2, 0xE5,
    0xF6, 0xF7, 0xF8, 0xF9, 0xFA, 0xFB, 0xFD, 0xFE
};

//!< ロックキーのトグルイベントを送信します。
void SendLockKeyToggleEvent(WORD wVk) NN_NOEXCEPT
{
    INPUT inputs[2] = {};

    for (INPUT& input : inputs)
    {
        input.type = INPUT_KEYBOARD;
        input.ki = KEYBDINPUT();
        input.ki.wVk = wVk;
    }

    inputs[1].ki.dwFlags = KEYEVENTF_KEYUP;

    ::SendInput(2, inputs, sizeof(INPUT));
}

} // namespace

WindowsKeyboard::WindowsKeyboard() NN_NOEXCEPT
    : m_IsConnected(false)
    , m_Keys()
    , m_Modifiers()
{
    // 何もしない
}

DebugPadButtonSet WindowsKeyboard::GetButtons(
    ::nn::settings::DebugPadKeyboardMap& map) NN_NOEXCEPT
{
    // 割り当てを更新
    m_RawMaps[0] = map.buttonA;
    m_RawMaps[1] = map.buttonB;
    m_RawMaps[2] = map.buttonX;
    m_RawMaps[3] = map.buttonY;
    m_RawMaps[4] = map.buttonL;
    m_RawMaps[5] = map.buttonR;
    m_RawMaps[6] = map.buttonZL;
    m_RawMaps[7] = map.buttonZR;
    m_RawMaps[8] = map.buttonStart;
    m_RawMaps[9] = map.buttonSelect;
    m_RawMaps[10] = map.buttonLeft;
    m_RawMaps[11] = map.buttonUp;
    m_RawMaps[12] = map.buttonRight;
    m_RawMaps[13] = map.buttonDown;

    // 割り当ての数を設定
    m_RawMapCount = 14;

    return this->GetButtons<DebugPadButtonSet>();
}

BasicXpadButtonSet WindowsKeyboard::GetButtons(
    ::nn::settings::BasicXpadKeyboardMap& map) NN_NOEXCEPT
{
    // 割り当てを更新
    m_RawMaps[0] = map.buttonA;
    m_RawMaps[1] = map.buttonB;
    m_RawMaps[2] = map.buttonX;
    m_RawMaps[3] = map.buttonY;
    m_RawMaps[4] = map.buttonStickL;
    m_RawMaps[5] = map.buttonStickR;
    m_RawMaps[6] = map.buttonL;
    m_RawMaps[7] = map.buttonR;
    m_RawMaps[8] = map.buttonZL;
    m_RawMaps[9] = map.buttonZR;
    m_RawMaps[10] = map.buttonStart;
    m_RawMaps[11] = map.buttonSelect;
    m_RawMaps[12] = map.buttonLeft;
    m_RawMaps[13] = map.buttonUp;
    m_RawMaps[14] = map.buttonRight;
    m_RawMaps[15] = map.buttonDown;

    // 割り当ての数を設定
    m_RawMapCount = 16;

    return this->GetButtons<BasicXpadButtonSet>();
}

template<typename T>
T WindowsKeyboard::GetButtons() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(
        m_RawMapCount,
        0,
        static_cast<int>(sizeof(m_RawMaps) / sizeof(m_RawMaps[0])));

    T buttons = { 0 };

    for (int i = 0; i < m_RawMapCount; ++i)
    {
        const int& rawMap = m_RawMaps[i];

        if (0 < rawMap && rawMap < m_Keys.GetCount())
        {
            buttons.Set(i, m_Keys.Test(rawMap));
        }
    }

    return buttons;
}

void WindowsKeyboard::UpdateConnectionStatus() NN_NOEXCEPT
{
    m_IsConnected = (0 < GetRawInputPhysicalKeyboardCount());
}

void WindowsKeyboard::UpdateKeys() NN_NOEXCEPT
{
    // 入力無し状態に初期化
    m_Keys.Reset();

    for (int keyCode : VirtualKeyCodes)
    {
        // 仮想キーの状態を取得
        const int keyState = ::GetAsyncKeyState(keyCode);

        if ((keyState & 0x8000) == 0)
        {
            // 押下状態になければスキップ
            continue;
        }

        // スキャンコードに変換
        const int scanCode = MapVirtualKeyToScanCode(keyCode);

        if (scanCode == 0)
        {
            // スキャンコードに変換できなければスキップ
            continue;
        }

        // HID Usage Index に変換
        const int usageIndex = MapScanCodeToHidUsageIndex(scanCode);

        // Keyboard のキーを押下状態に設定
        m_Keys.Set(usageIndex, true);
    }
}

void WindowsKeyboard::UpdateModifiers() NN_NOEXCEPT
{
    // 修飾無し状態に初期化
    m_Modifiers.Reset();

    // Ctrl キーの押下状態を設定
    m_Modifiers.Set<KeyboardModifier::Control>(
        m_Keys.Test<KeyboardKey::LeftControl>() ||
        m_Keys.Test<KeyboardKey::RightControl>());

    // Shift キーの押下状態を設定
    m_Modifiers.Set<KeyboardModifier::Shift>(
        m_Keys.Test<KeyboardKey::LeftShift>() ||
        m_Keys.Test<KeyboardKey::RightShift>());

    // 左 Alt キーの押下状態を設定
    m_Modifiers.Set<KeyboardModifier::LeftAlt>(
        m_Keys.Test<KeyboardKey::LeftAlt>());

    // 右 Alt キーの押下状態を設定
    m_Modifiers.Set<KeyboardModifier::RightAlt>(
        m_Keys.Test<KeyboardKey::RightAlt>());

    // GUI キーの押下状態を設定
    m_Modifiers.Set<KeyboardModifier::Gui>(
        m_Keys.Test<KeyboardKey::LeftGui>() ||
        m_Keys.Test<KeyboardKey::RightGui>());

    // CapsLock が有効か否かを設定
    m_Modifiers.Set<KeyboardModifier::CapsLock>(
        (::GetKeyState(VK_CAPITAL) & 0x0001) != 0);

    // ScrollLock が有効か否かを設定
    m_Modifiers.Set<KeyboardModifier::ScrollLock>(
        (::GetKeyState(VK_SCROLL) & 0x0001) != 0);

    // NumLock が有効か否かを設定
    m_Modifiers.Set<KeyboardModifier::NumLock>(
        (::GetKeyState(VK_NUMLOCK) & 0x0001) != 0);
}

void WindowsKeyboard::Reset() NN_NOEXCEPT
{
    m_IsConnected = false;

    m_Modifiers.Reset();

    m_Keys.Reset();
}

void WindowsKeyboard::SendLockKeyEvent(
    ::nn::hid::system::KeyboardLockKeyEventSet value) NN_NOEXCEPT
{
    KeyboardModifierSet modifiers = m_Modifiers;

    // Num Lock のイベントから Num Lock の最終的な状態を決定

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::NumLockOn>())
    {
        modifiers.Set<KeyboardModifier::NumLock>();
    }

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::NumLockOff>())
    {
        modifiers.Reset<KeyboardModifier::NumLock>();
    }

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::NumLockToggle>())
    {
        modifiers.Flip<KeyboardModifier::NumLock>();
    }

    // Caps Lock のイベントから Caps Lock の最終的な状態を決定

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::CapsLockOn>())
    {
        modifiers.Set<KeyboardModifier::CapsLock>();
    }

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::CapsLockOff>())
    {
        modifiers.Reset<KeyboardModifier::CapsLock>();
    }

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::CapsLockToggle>())
    {
        modifiers.Flip<KeyboardModifier::CapsLock>();
    }

    // Scroll Lock のイベントから Scroll Lock の最終的な状態を決定

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::ScrollLockOn>())
    {
        modifiers.Set<KeyboardModifier::ScrollLock>();
    }

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::ScrollLockOff>())
    {
        modifiers.Reset<KeyboardModifier::ScrollLock>();
    }

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::ScrollLockToggle>())
    {
        modifiers.Flip<KeyboardModifier::ScrollLock>();
    }

    // 状態の変更が必要な入力モードについてキーイベントを送信

    modifiers ^= m_Modifiers;

    if (modifiers.Test<KeyboardModifier::NumLock>())
    {
        SendLockKeyToggleEvent(VK_NUMLOCK);
    }

    if (modifiers.Test<KeyboardModifier::CapsLock>())
    {
        SendLockKeyToggleEvent(VK_CAPITAL);
    }

    if (modifiers.Test<KeyboardModifier::ScrollLock>())
    {
        SendLockKeyToggleEvent(VK_SCROLL);
    }
}

}}} // namespace nn::hid::detail
