﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Windows.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/settings/settings_GenericPadAxis.h>
#include <nn/settings/settings_GenericPadButton.h>
#include <nn/util/util_BitFlagSet.h>

#include "hid_WindowsGenericPadAccessor-os.win.h"
#include "hid_WindowsHidSdi-os.win.h"

namespace nn { namespace hid { namespace detail {

//!< Windows の汎用コントローラのデジタルボタン定義です。
struct WindowsGenericPadButton final
{
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<32>
            Up;         //!< Windows の汎用コントローラの十字ボタン 上
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<33>
            Down;       //!< Windows の汎用コントローラの十字ボタン 下
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<34>
            Left;       //!< Windows の汎用コントローラの十字ボタン 左
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<35>
            Right;      //!< Windows の汎用コントローラの十字ボタン 右
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<36>
            Start;      //!< Windows の汎用コントローラの START ボタン
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<37>
            Back;       //!< Windows の汎用コントローラの BACK ボタン
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<38>
            ThumbL;     //!< Windows の汎用コントローラの L スティックボタン
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<39>
            ThumbR;     //!< Windows の汎用コントローラの R スティックボタン
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<40>
            L;          //!< Windows の汎用コントローラの L ボタン
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<41>
            R;          //!< Windows の汎用コントローラの R ボタン
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<42>
            A;          //!< Windows の汎用コントローラの A ボタン
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<43>
            B;          //!< Windows の汎用コントローラの B ボタン
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<44>
            X;          //!< Windows の汎用コントローラの X ボタン
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<45>
            Y;          //!< Windows の汎用コントローラの Y ボタン
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<46>
            TriggerL;   //!< Windows の汎用コントローラの L トリガ
    typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>::Flag<47>
            TriggerR;   //!< Windows の汎用コントローラの R トリガ
};

//!< Windows の汎用コントローラのデジタルボタンの集合を扱う型です。
typedef ::nn::util::BitFlagSet<64, WindowsGenericPadButton>
        WindowsGenericPadButtonSet;

//!< Windows の汎用コントローラを扱うためのクラスです。
class WindowsGenericPad final
{
    NN_DISALLOW_COPY(WindowsGenericPad);
    NN_DISALLOW_MOVE(WindowsGenericPad);

private:
    //!< 汎用コントローラが検出されたか否か
    bool m_IsDetected;

    //!< 汎用コントローラの識別子
    int m_GenericPadId;

    //!< 汎用コントローラのハンドル
    HidSdiHandle m_Handle;

    //!< 汎用コントローラの状態が有効か否か
    bool m_IsStateValid;

    //!< 汎用コントローラの状態
    HidSdiGamePadState m_State;

    //!< 汎用コントローラの能力
    WindowsGenericPadAbility m_Ability;

    //!< デジタルボタンの押下状態
    WindowsGenericPadButtonSet m_Buttons;

    //!< 右アナログスティックの状態
    AnalogStickState m_StickL;

    //!< 右アナログスティックの状態
    AnalogStickState m_StickR;

    //!< 座標値
    int m_Positions[6];

public:
    WindowsGenericPad() NN_NOEXCEPT;

    //!< 対象とする汎用コントローラを設定します。
    void SetGenericPadId(int genericPadId) NN_NOEXCEPT;

    //!< 汎用コントローラが検出されたか否かを表す値を返します。
    bool IsDetected() const NN_NOEXCEPT;

    //!< XInput 対応コントローラか否かを表す値を返します。
    bool IsXInput() const NN_NOEXCEPT;

    //!< デジタルボタンが押下されているか否かを表す値を返します。
    bool IsButtonDown(
        ::nn::settings::GenericPadButton button) const NN_NOEXCEPT;

    //!< デジタルボタンの押下状態を返します。
    WindowsGenericPadButtonSet GetButtons() const NN_NOEXCEPT;

    //!< 左アナログスティックの状態を返します。
    AnalogStickState GetStickL() const NN_NOEXCEPT;

    //!< 右アナログスティックの状態を返します。
    AnalogStickState GetStickR() const NN_NOEXCEPT;

    //!< 指定の割り当てに基づいた左アナログスティックの状態を返します。
    template<typename T>
    AnalogStickState GetStickL(const T& map) const NN_NOEXCEPT
    {
        AnalogStickState state = {};
        state.x = static_cast<int32_t>(
            this->GetPosition(
                static_cast<::nn::settings::GenericPadAxis>(
                    map.axisAnalogStickLX), map.attributes));
        state.y = static_cast<int32_t>(
            this->GetPosition(
                static_cast<::nn::settings::GenericPadAxis>(
                    map.axisAnalogStickLY), map.attributes));
        return state;
    }

    //!< 指定の割り当てに基づいた右アナログスティックの状態を返します。
    template<typename T>
    AnalogStickState GetStickR(const T& map) const NN_NOEXCEPT
    {
        AnalogStickState state = {};
        state.x = static_cast<int32_t>(
            this->GetPosition(
                static_cast<::nn::settings::GenericPadAxis>(
                    map.axisAnalogStickRX), map.attributes));
        state.y = static_cast<int32_t>(
            this->GetPosition(
                static_cast<::nn::settings::GenericPadAxis>(
                    map.axisAnalogStickRY), map.attributes));
        return state;
    }

    //!< 内部状態を更新します。
    void Update() NN_NOEXCEPT;

    //!< 内部状態をリセットします。
    void Reset() NN_NOEXCEPT;

private:
    //!< 指定された座標軸の座標値を返します。
    int GetPosition(
        ::nn::settings::GenericPadAxis axis,
        const ::nn::settings::GenericPadAxisAttributeSet& attribute
        ) const NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
