﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <atomic>
#include <type_traits>
#include <nn/nn_Common.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/TargetConfigs/build_Base.h>

#include "hid_RingLifo.h"

namespace nn { namespace hid { namespace detail {

class TouchScreenStateAtomicStorage final
{
    NN_DISALLOW_COPY(TouchScreenStateAtomicStorage);
    NN_DISALLOW_MOVE(TouchScreenStateAtomicStorage);

private:
    ::std::atomic<int64_t> m_SamplingNumber;
    TouchScreenState<TouchStateCountMax> m_State;

public:
    TouchScreenStateAtomicStorage() NN_NOEXCEPT
        : m_SamplingNumber(0)
        , m_State()
    {
        // 何もしない
    }

    //!< 要素をスレッドセーフに取得します。
    template<size_t N>
    bool Get(TouchScreenState<N>* pOutValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        const int64_t samplingNumber =
            m_SamplingNumber.load(::std::memory_order_acquire);

        pOutValue->samplingNumber = m_State.samplingNumber;
        pOutValue->count = ::std::min<int>(m_State.count, N);

        for (size_t i = 0; i < N; ++i)
        {
            pOutValue->touches[i] = m_State.touches[i];
        }

        // 読み出し中に値が書き換わっていなければ成功
        return (samplingNumber == m_SamplingNumber);
    }

    //!< 要素をスレッドセーフに設定します。
    template<size_t N>
    void Set(const TouchScreenState<N>& value) NN_NOEXCEPT
    {
        m_SamplingNumber.store(
            value.samplingNumber, ::std::memory_order_release);

        ::std::atomic_thread_fence(::std::memory_order_release);

        m_State.samplingNumber = value.samplingNumber;
        m_State.count = value.count;

        for (size_t i = 0; i < N; ++i)
        {
            m_State.touches[i] = value.touches[i];
        }
    }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(
    ::std::is_standard_layout<TouchScreenStateAtomicStorage>::value);
#endif

class TouchScreenLifo final
    : public RingLifo<
        TouchScreenState<TouchStateCountMax>,
        TouchScreenStateCountMax,
        TouchScreenStateAtomicStorage>
{
public:
    TouchScreenLifo() NN_NOEXCEPT { /* 何もしない */ }

    //!< タッチ数を指定して LIFO からスレッドセーフに要素を読み出します。
    template<size_t N>
    int ReadPartially(
        TouchScreenState<N>* buffer, int bufferCount) const NN_NOEXCEPT
    {
        return this->ReadImpl(buffer, bufferCount);
    }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<TouchScreenLifo>::value);
#endif

}}} // namespace nn::hid::detail
