﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_HidShellPortFile-os.win.h"
#include "hid_Point.h"
#include "hid_Rectangle.h"
#include "hid_TouchScreenDriver-os.win.h"
#include "hid_TouchScreenStateUtility.h"
#include "hid_WindowsProcedure-os.win.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< マウスのタッチ入力の識別子
const DWORD MouseTouchInputId = ::std::numeric_limits<DWORD>::max();

//!< タッチの接触面積の直径（x 軸方向）のデフォルト値
const int DefaultTouchDiameterX = 1;

//!< タッチの接触面積の直径（y 軸方向）のデフォルト値
const int DefaultTouchDiameterY = 1;

//!< タッチ入力からタッチの状態を作成します。
TouchState MakeTouchState(const TOUCHINPUT& touchInput) NN_NOEXCEPT
{
    TouchState touch = {};

    touch.x = touchInput.x;
    touch.y = touchInput.y;

    if (touchInput.dwMask & TOUCHINPUTMASKF_CONTACTAREA)
    {
        touch.diameterX = static_cast<int32_t>(touchInput.cxContact);
        touch.diameterY = static_cast<int32_t>(touchInput.cyContact);
    }
    else
    {
        touch.diameterX = static_cast<int32_t>(DefaultTouchDiameterX);
        touch.diameterY = static_cast<int32_t>(DefaultTouchDiameterY);
    }

    return touch;
}

//!< TouchScreen の HidShell 向けに共有される入力状態を更新します。
void SetTouchScreenFileState(
    const TouchScreenState<TouchStateCountMax>& state) NN_NOEXCEPT
{
    ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax
        > autoPilotState = {};

    autoPilotState.count = state.count;

    for (int i = 0; i < state.count; ++i)
    {
        autoPilotState.touches[i] = state.touches[i];
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        SetHidShellTouchScreenState(autoPilotState));
}

//!< TouchScreen の HidShell 向けに共有される自動操作状態を変換します。
void ConvertTouchScreenFileAutoPilotState(
    TouchScreenState<TouchStateCountMax>* pOutValue,
    const ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax> state
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = TouchScreenState<TouchStateCountMax>();

    pOutValue->count = state.count;

    for (int i = 0; i < state.count; ++i)
    {
        const TouchState& srcTouch = state.touches[i];
        TouchState& dstTouch = pOutValue->touches[i];
        dstTouch.fingerId = srcTouch.fingerId;
        dstTouch.attributes = srcTouch.attributes;
        dstTouch.x = srcTouch.x;
        dstTouch.y = srcTouch.y;
        dstTouch.diameterX = 64;
        dstTouch.diameterY = 64;
    }
}

} // namespace

bool TouchInputIdManager::RegisterTouchInputId(DWORD value) NN_NOEXCEPT
{
    Entry* pEntry = nullptr;

    for (size_t i = TouchStateCountMax; i > 0; --i)
    {
        Entry& entry = m_Entries[i - 1];

        if (!entry.hasValue)
        {
            pEntry = &entry;
        }
        else
        {
            if (value == entry.value)
            {
                return false;
            }
        }
    }

    if (pEntry == nullptr)
    {
        return false;
    }
    else
    {
        pEntry->hasValue = true;
        pEntry->value = value;
        return true;
    }
}

bool TouchInputIdManager::UnregisterTouchInputId(DWORD value) NN_NOEXCEPT
{
    auto found = false;

    for (Entry& entry : m_Entries)
    {
        if (entry.hasValue && value == entry.value)
        {
            found = true;
            entry = Entry();
        }
    }

    return found;
}

bool TouchInputIdManager::UnregisterTouchInputIdByIndex(
    size_t index) NN_NOEXCEPT
{
    if (index >= TouchStateCountMax)
    {
        return false;
    }

    Entry& entry = m_Entries[index];

    if (!entry.hasValue)
    {
        return false;
    }

    entry = Entry();

    return true;
}

void TouchInputIdManager::UnregisterAllTouchInputId() NN_NOEXCEPT
{
    for (Entry& entry : m_Entries)
    {
        entry = Entry();
    }
}

void TouchInputIdManager::UnregisterAllTouchInputIdExceptMouse() NN_NOEXCEPT
{
    for (Entry& entry : m_Entries)
    {
        if (entry.hasValue && entry.value != MouseTouchInputId)
        {
            entry = Entry();
        }
    }
}

bool TouchInputIdManager::FindTouchInputId(size_t* pOutValue, DWORD value
                                           ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    for (size_t i = 0; i < TouchStateCountMax; ++i)
    {
        const Entry& entry = m_Entries[i];

        if (entry.hasValue && value == entry.value)
        {
            *pOutValue = i;

            return true;
        }
    }

    return false;
}

TouchScreenDriver::TouchScreenDriver() NN_NOEXCEPT
    : m_ActivationCount()
    , m_IsAwake(false)
    , m_SamplingNumber(0)
    , m_FingerId(0)
    , m_AutoPilotState()
{
    // 何もしない
}

TouchScreenDriver::~TouchScreenDriver() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result TouchScreenDriver::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultTouchScreenDriverActivationUpperLimitOver());

    // ウィンドウプロシージャをアクティブ化
    ActivateWindowsProcedure();

    // このインスタンスからアクティブ化した回数をインクリメント
    ++m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // 起床状態に設定
        m_IsAwake = true;
    }

    // ウィンドウプロシージャを維持
    KeepWindowsProcedureAlive();

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultTouchScreenDriverDeactivationLowerLimitOver());

    // このインスタンスからアクティブ化した回数をデクリメント
    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        // HidShell による自動操作を無効化
        m_AutoPilotState =
            ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax>();

        // タッチの識別子をリセット
        m_FingerId = 0;

        // 識別子を破棄
        m_TouchIds.Clear();

        // 全てのタッチ入力の識別子の登録を解除
        m_TouchInputIdManager.UnregisterAllTouchInputId();

        // 起床状態を解除
        m_IsAwake = false;
    }

    // ウィンドウプロシージャを非アクティブ化
    DeactivateWindowsProcedure();

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenDriver::Wake() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    m_IsAwake = true;

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenDriver::Sleep() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    m_IsAwake = false;

    NN_RESULT_SUCCESS;
}

void TouchScreenDriver::GetState(
    TouchScreenState<TouchStateCountMax>* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    // ウィンドウプロシージャを維持
    KeepWindowsProcedureAlive();

    // 作業領域を初期化
    m_NewIds.Clear();
    m_OldIds.Clear();

    Rectangle rect = {};
    if (m_IsAwake && GetWindowsProcedureClientAreaSize(&rect))
    {
        // マウス入力の状態を反映
        this->UpdateMouseInputState(rect);

        // タッチ入力の状態を反映
        this->UpdateTouchInputState();
    }
    else
    {
        // 全てのタッチ入力の識別子の登録を解除
        m_TouchInputIdManager.UnregisterAllTouchInputId();

        // タッチを破棄
        m_TouchIds.Clear();
    }

    // 新たな識別子を処理
    for (size_t i = 0; i < m_NewIds.GetCount(); ++i)
    {
        const size_t touchId = m_NewIds[i];
        NN_SDK_ASSERT_LESS(touchId, static_cast<size_t>(TouchStateCountMax));
        m_TouchIds.Append(touchId);
    }

    // タッチスクリーンの入力状態を決定
    const size_t touchCount = m_TouchIds.GetCount();
    NN_SDK_ASSERT_LESS_EQUAL(touchCount,
                             static_cast<size_t>(TouchStateCountMax));
    *pOutValue = TouchScreenState<TouchStateCountMax>();
    pOutValue->samplingNumber = m_SamplingNumber++;
    pOutValue->count = static_cast<int32_t>(touchCount);
    for (size_t i = 0; i < touchCount; ++i)
    {
        const size_t touchId = m_TouchIds[i];
        NN_SDK_ASSERT_LESS(touchId, static_cast<size_t>(TouchStateCountMax));
        pOutValue->touches[i] = ScaleTouchPosition(m_Touches[touchId], rect);
    }

    // タッチの経過時間と属性をリセット
    for (size_t i = 0; i < m_TouchIds.GetCount(); ++i)
    {
        const size_t touchId = m_TouchIds[i];
        NN_SDK_ASSERT_LESS(touchId, static_cast<size_t>(TouchStateCountMax));
        m_Touches[touchId].deltaTime = ::nn::TimeSpanType();
        m_Touches[touchId].attributes.Reset();
    }

    // 全てのタッチが開放された場合はタッチの識別子をリセット
    if (m_TouchIds.GetCount() == 0)
    {
        m_FingerId = 0;
    }

    // 去った識別子を処理
    for (size_t i = 0; i < m_OldIds.GetCount(); ++i)
    {
        const size_t touchId = m_OldIds[i];
        m_TouchInputIdManager.UnregisterTouchInputIdByIndex(touchId);
        m_TouchIds.Remove(touchId);
    }

    // TouchScreen の HidShell 向けに共有される入力状態を更新
    SetTouchScreenFileState(*pOutValue);

    // TouchScreen の HidShell 向けに共有される自動操作状態をマージ
    bool isEnabled = false;
    ::nn::hid::debug::TouchScreenAutoPilotState<
        TouchStateCountMax> autoPilotState = {};
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        GetHidShellTouchScreenState(&isEnabled, &autoPilotState));
    if (!isEnabled)
    {
        m_AutoPilotState =
            ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax>();
    }
    else
    {
        UpdateTouchScreenAutoPilotState(&m_AutoPilotState, autoPilotState);
        ConvertTouchScreenFileAutoPilotState(pOutValue, m_AutoPilotState);
        FlushTouchScreenAutoPilotState(&m_AutoPilotState);
    }
}

void TouchScreenDriver::UpdateMouseInputState(const Rectangle& rect) NN_NOEXCEPT
{
    const DWORD touchInputId = MouseTouchInputId;

    auto touchId = size_t();

    bool finds = m_TouchInputIdManager.FindTouchInputId(&touchId, touchInputId);

    // タイムスタンプを取得
    const ::nn::TimeSpanType nextTimeStamp =
        ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick());

    Point position = {};
    if (!GetWindowsProcedureMousePosition(&position))
    {
        if (finds)
        {
            m_TouchInputIdManager.UnregisterTouchInputId(touchInputId);
            m_TouchIds.Remove(touchId);
        }

        // 有効なマウスカーソルの座標が取得できなければ処理を中止
        return;
    }

    const bool isButtonDown =
        GetWindowsProcedureMouseButtons().Test<MouseButton::Left>();

    if (!finds)
    {
        if (!isButtonDown)
        {
            return;
        }

        if (!m_TouchInputIdManager.RegisterTouchInputId(touchInputId))
        {
            return;
        }

        if (!m_TouchInputIdManager.FindTouchInputId(&touchId, touchInputId))
        {
            return;
        }
    }

    NN_SDK_ASSERT_LESS(touchId, static_cast<size_t>(TouchStateCountMax));

    // タイムスタンプを更新
    ::nn::TimeSpanType& timeStamp = m_TimeStamps[touchId];
    const ::nn::TimeSpanType lastTimeStamp = timeStamp;
    timeStamp = nextTimeStamp;

    // タッチの状態を更新
    TouchState& touch = m_Touches[touchId];

    if (!finds)
    {
        if (isButtonDown)
        {
            touch = TouchState();
            touch.deltaTime = ::nn::TimeSpanType();
            touch.attributes.Set<TouchAttribute::Start>();
            touch.fingerId = m_FingerId++;
            touch.diameterX = static_cast<int32_t>(DefaultTouchDiameterX);
            touch.diameterY = static_cast<int32_t>(DefaultTouchDiameterY);
            touch.rotationAngle = 0;

            // 新たな識別子に追加
            m_NewIds.Append(touchId);
        }
    }
    else
    {
        if (!rect.Contains(position))
        {
            // マウスカーソルがクライアント領域外にある場合は
            // 座標が画面外に出ない様に前回の値を再利用
            Point lastPosition = { touch.x, touch.y };
            position = rect.Clamp(lastPosition);
        }

        touch.deltaTime = timeStamp - lastTimeStamp;
        touch.attributes.Reset();

        if (!isButtonDown)
        {
            touch.attributes.Set<TouchAttribute::End>();

            // 去った識別子に追加
            m_OldIds.Append(touchId);
        }
    }

    touch.x = static_cast<int32_t>(position.x);
    touch.y = static_cast<int32_t>(position.y);
}

void TouchScreenDriver::UpdateTouchInputState() NN_NOEXCEPT
{
    int count = GetWindowsProcedureTouchInputs(m_TouchInputs,
                                               TouchInputCountMax);

    if (count == TouchInputCountMax)
    {
        // 上限までタッチ入力を取得できてしまった場合はオーバーフローと判断

        // マウス入力によるタッチが存在するか否かを確認
        bool exists = false;
        auto touchId = size_t();
        if (m_TouchInputIdManager.FindTouchInputId(&touchId, MouseTouchInputId))
        {
            exists = m_TouchIds.Contains(touchId);
        }

        // タッチ入力によるタッチを全て破棄
        m_TouchInputIdManager.UnregisterAllTouchInputIdExceptMouse();
        m_TouchIds.Clear();

        // マウス入力によるタッチを復旧
        if (exists)
        {
            m_TouchIds.Append(touchId);
        }
    }
    else
    {
        for (int i = count - 1; i >= 0; --i)
        {
            const TOUCHINPUT& touchInput = m_TouchInputs[i];

            if (touchInput.dwFlags & TOUCHEVENTF_DOWN)
            {
                this->ProcessTouchInputDownEvent(touchInput);
            }
            else if(touchInput.dwFlags & TOUCHEVENTF_UP)
            {
                this->ProcessTouchInputUpEvent(touchInput);
            }
            else if(touchInput.dwFlags & TOUCHEVENTF_MOVE)
            {
                this->ProcessTouchInputMoveEvent(touchInput);
            }
        }
    }
}

void TouchScreenDriver::ProcessTouchInputDownEvent(
    const TOUCHINPUT& touchInput) NN_NOEXCEPT
{
    const DWORD touchInputId = touchInput.dwID;

    auto touchId = size_t();

    if (!m_TouchInputIdManager.RegisterTouchInputId(touchInputId))
    {
        if (m_TouchInputIdManager.FindTouchInputId(&touchId, touchInputId))
        {
            m_TouchInputIdManager.UnregisterTouchInputId(touchInputId);
            m_TouchIds.Remove(touchId);
        }

        return;
    }

    if (m_TouchInputIdManager.FindTouchInputId(&touchId, touchInputId))
    {
        NN_SDK_ASSERT_LESS(touchId, static_cast<size_t>(TouchStateCountMax));

        if (!m_NewIds.Contains(touchId) && !m_TouchIds.Contains(touchId))
        {
            // タイムスタンプを更新
            ::nn::TimeSpanType& timeStamp = m_TimeStamps[touchId];
            timeStamp = ::nn::TimeSpanType::FromMilliSeconds(touchInput.dwTime);

            // タッチの状態を更新
            TouchState& touch = m_Touches[touchId];
            touch = MakeTouchState(touchInput);
            touch.deltaTime = ::nn::TimeSpanType();
            touch.attributes.Set<TouchAttribute::Start>();
            touch.fingerId = m_FingerId++;

            // 新たな識別子に追加
            m_NewIds.Append(touchId);
        }
    }
}

void TouchScreenDriver::ProcessTouchInputUpEvent(
    const TOUCHINPUT& touchInput) NN_NOEXCEPT
{
    const DWORD touchInputId = touchInput.dwID;

    auto touchId = size_t();

    if (m_TouchInputIdManager.FindTouchInputId(&touchId, touchInputId))
    {
        NN_SDK_ASSERT_LESS(touchId, static_cast<size_t>(TouchStateCountMax));

        if (m_NewIds.Contains(touchId))
        {
            m_TouchInputIdManager.UnregisterTouchInputId(touchInputId);
            m_NewIds.Remove(touchId);
        }
        else if (!m_OldIds.Contains(touchId) && m_TouchIds.Contains(touchId))
        {
            // タイムスタンプを更新
            ::nn::TimeSpanType& timeStamp = m_TimeStamps[touchId];
            const ::nn::TimeSpanType lastTimeStamp = timeStamp;
            timeStamp = ::nn::TimeSpanType::FromMilliSeconds(touchInput.dwTime);

            // タッチの状態を更新
            TouchState& touch = m_Touches[touchId];
            int32_t fingerId = touch.fingerId;
            touch = MakeTouchState(touchInput);
            touch.deltaTime = timeStamp - lastTimeStamp;
            touch.attributes.Set<TouchAttribute::End>();
            touch.fingerId = fingerId;

            // 去った識別子に追加
            m_OldIds.Append(touchId);
        }
    }
}

void TouchScreenDriver::ProcessTouchInputMoveEvent(
    const TOUCHINPUT& touchInput) NN_NOEXCEPT
{
    auto touchId = size_t();

    if (m_TouchInputIdManager.FindTouchInputId(&touchId, touchInput.dwID))
    {
        NN_SDK_ASSERT_LESS(touchId, static_cast<size_t>(TouchStateCountMax));

        if (m_TouchIds.Contains(touchId))
        {
            // タイムスタンプを更新
            ::nn::TimeSpanType& timeStamp = m_TimeStamps[touchId];
            const ::nn::TimeSpanType lastTimeStamp = timeStamp;
            timeStamp = ::nn::TimeSpanType::FromMilliSeconds(touchInput.dwTime);

            // タッチの状態を更新
            TouchState& touch = m_Touches[touchId];
            int32_t fingerId = touch.fingerId;
            touch = MakeTouchState(touchInput);
            touch.deltaTime = timeStamp - lastTimeStamp;
            touch.fingerId = fingerId;
        }
    }
}

}}} // namespace nn::hid::detail
