﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <mutex>
#include <new>
#include <nn/nn_Macro.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_LockableMutexType.h"

namespace nn { namespace hid { namespace detail {

//!< 静的なオブジェクトを扱うクラスです。
template<typename ObjectT, typename Tag = void>
class StaticObject final
{
    NN_DISALLOW_COPY(StaticObject);
    NN_DISALLOW_MOVE(StaticObject);

private:
    StaticObject() NN_NOEXCEPT;

public:
    //!< 静的なオブジェクトを返します。
    static ObjectT& Get() NN_NOEXCEPT
    {
        static ::nn::util::TypedStorage<ObjectT,
                                 sizeof(ObjectT),
                             NN_ALIGNOF(ObjectT)> s_Storage = {};

        NN_FUNCTION_LOCAL_STATIC(::std::atomic<bool>, s_IsInitialized, (false));

        if (!s_IsInitialized)
        {
            static LockableMutexType s_Mutex =
            {
                NN_OS_MUTEX_INITIALIZER(false)
            };

            ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

            if (!s_IsInitialized)
            {
                new(&::nn::util::Get(s_Storage)) ObjectT();

                s_IsInitialized = true;
            }
        }

        return ::nn::util::Get(s_Storage);
    }
};

}}} // namespace nn::hid::detail
