﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidSystemServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/debug/hid_SleepButton.h>
#include <nn/hid/system/hid_SleepButton.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_HidDebugServer.h"
#include "hid_HidSystemServer.h"
#include "hid_SleepButtonImpl.h"
#include "hid_SleepButtonLifo.h"
#include "hid_SharedMemoryAccessor.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_StaticObject.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< スリープボタンの共有メモリへのアクセスを扱うクラスです。
class SleepButtonSharedMemoryAccessor final : public SharedMemoryAccessor
{
private:
    //!< スリープボタンの LIFO
    ::std::atomic<SleepButtonLifo*> m_pLifo;

public:
    SleepButtonSharedMemoryAccessor() NN_NOEXCEPT
        : m_pLifo(nullptr)
    {
        this->SetResultActivationUpperLimitOver(
            ::nn::hid::ResultSleepButtonActivationUpperLimitOver());
        this->SetResultDeactivationLowerLimitOver(
            ::nn::hid::ResultSleepButtonDeactivationLowerLimitOver());
    }

    virtual ~SleepButtonSharedMemoryAccessor() NN_NOEXCEPT NN_OVERRIDE
    {
        // 何もしない
    }

    //!< スリープボタンの LIFO を返します。
    const SleepButtonLifo* GetLifo() const NN_NOEXCEPT
    {
        return m_pLifo;
    }

protected:
    virtual ::nn::Result Attach(SharedMemoryFormat* ptr) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(ptr);
        ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
        NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));
        NN_RESULT_DO(pHidSystemServer->ActivateSleepButton(
                         ::nn::applet::AppletResourceUserId::GetInvalidId()));
        m_pLifo = &::nn::util::Get(ptr->sleepButton.lifo);
        NN_RESULT_SUCCESS;
    }

    virtual ::nn::Result Detach() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
        NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
        NN_RESULT_DO(pHidDebugServer->DeactivateSleepButton());
        m_pLifo = nullptr;
        NN_RESULT_SUCCESS;
    }
};

//!< スリープボタンの共有メモリアクセッサを返します。
SleepButtonSharedMemoryAccessor& GetSleepButtonSharedMemoryAccessor(
    ) NN_NOEXCEPT
{
    return StaticObject<SleepButtonSharedMemoryAccessor>::Get();
}

} // namespace

::nn::Result BindSleepButtonEvent(::nn::os::SystemEventType* pEvent,
                                  ::nn::os::EventClearMode clearMode
                                  ) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));

    ::nn::sf::NativeHandle handle;
    NN_RESULT_DO(
        pHidSystemServer->AcquireSleepButtonEventHandle(
            &handle, ::nn::applet::AppletResourceUserId::GetInvalidId()));

    ::nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                                handle.GetOsHandle(),
                                                handle.IsManaged(),
                                                clearMode);

    // ハンドルの管理権を放棄
    handle.Detach();

    NN_RESULT_SUCCESS;
}

::nn::Result InitializeSleepButton() NN_NOEXCEPT
{
    SleepButtonSharedMemoryAccessor&
        accessor = GetSleepButtonSharedMemoryAccessor();
    NN_RESULT_DO(accessor.SetAppletResourceUserId(
                     ::nn::applet::AppletResourceUserId::GetInvalidId()));
    NN_RESULT_DO(accessor.Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeSleepButton() NN_NOEXCEPT
{
    NN_RESULT_DO(GetSleepButtonSharedMemoryAccessor().Deactivate());
    NN_RESULT_SUCCESS;
}

::nn::Result GetSleepButtonStates(
    int* pOutCount,
    ::nn::hid::system::SleepButtonState outStates[],
    int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr,
                           ResultSleepButtonNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultSleepButtonInvalidStateCount());
    const SleepButtonLifo* pLifo =
        GetSleepButtonSharedMemoryAccessor().GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultSleepButtonNotInitialized());
    NN_SDK_REQUIRES(!pLifo->IsEmpty());
    *pOutCount = pLifo->Read(outStates, count);
    NN_RESULT_SUCCESS;
}

::nn::Result SetSleepButtonAutoPilotState(
    const ::nn::hid::debug::SleepButtonAutoPilotState& value) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->SetSleepButtonAutoPilotState(value));
    NN_RESULT_SUCCESS;
}

::nn::Result UnsetSleepButtonAutoPilotState() NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->UnsetSleepButtonAutoPilotState());
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
