﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <atomic>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/os/os_Mutex.h>
#include <nn/TargetConfigs/build_Base.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/xcd/xcd_Firmware.h>
#include <nn/xcd/xcd_Tera.h>
#include <nn/xcd/xcd_Ble.h>
#endif

#include "hid_LockableMutexType.h"
#include "hid_Settings.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< ファームウェアデバッグ設定の設定名
const char* const SettingsName = "hid_debug";

//!< ファームウェアデバッグ設定の設定値を表す構造体です。
struct FirmwareDebugSettingsValue
{
    //!< DebugPad が有効か否かを表す値
    bool enablesDebugPad;

    //!< hid プロセスによるデバイスの自動管理が有効か否かを表す値
    bool managesDevices;

    //!< Switch Pro Controller の未知デバイスエミュレーションが有効化どうか
    bool emulateFutureDevice;

    //!< コントローラー MCU のハードウェアエラーエミュレーションが有効かどうか
    bool emulateHardwareErrorForMcu;

    //!< コントローラー FW 更新失敗エラーエミュレーションが有効かどうか
    bool emulateFailureForFirmwareUpdate;

    //!< コントローラー FW 更新失敗エミュレーションのモード
    uint32_t firmwareUpdateFailureEmulationMode;

    //!< hid での BLE 機能が有効かどうか
    bool isBleDisabled;

    //!< Touch IC の I2C の制御が有効化否かを表す値
    bool managesTouchIcI2c;

};

//!< FW 更新失敗のモード定義
enum FirmwareUpdateFailureMode
{
    FirmwareUpdateFailureMode_Bluetooth = 0x01,     //!< Bluetooth FW 更新失敗
    FirmwareUpdateFailureMode_Mcu       = 0x02      //!< MCU FW 更新失敗
};

//!< ファームウェアデバッグ設定の設定値を返します。
const FirmwareDebugSettingsValue& GetFirmwareDebugSettingsValue() NN_NOEXCEPT;

} // namespace

void FirmwareDebugSettings::Initialize() NN_NOEXCEPT
{
    GetFirmwareDebugSettingsValue();
}

bool FirmwareDebugSettings::EnablesDebugPad() NN_NOEXCEPT
{
    return GetFirmwareDebugSettingsValue().enablesDebugPad;
}

bool FirmwareDebugSettings::ManagesDevices() NN_NOEXCEPT
{
    return GetFirmwareDebugSettingsValue().managesDevices;
}

bool FirmwareDebugSettings::EmulateFutureDevice() NN_NOEXCEPT
{
    return GetFirmwareDebugSettingsValue().emulateFutureDevice;
}

bool FirmwareDebugSettings::ManagesTouchIcI2c() NN_NOEXCEPT
{
    return GetFirmwareDebugSettingsValue().managesTouchIcI2c;
}

namespace {

const FirmwareDebugSettingsValue& GetFirmwareDebugSettingsValue() NN_NOEXCEPT
{
    static FirmwareDebugSettingsValue s_Value = {};

    NN_FUNCTION_LOCAL_STATIC(::std::atomic<bool>, s_IsInitialized, (false));

    if (!s_IsInitialized)
    {
        static LockableMutexType s_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

        ::std::lock_guard<decltype(s_Mutex)> locker(s_Mutex);

        if (!s_IsInitialized)
        {
#if !defined(NN_BUILD_CONFIG_OS_HORIZON)
            s_Value.enablesDebugPad = true;

            s_Value.managesDevices = false;

            s_Value.emulateFutureDevice = false;

            s_Value.emulateHardwareErrorForMcu = false;

            s_Value.emulateFailureForFirmwareUpdate = false;

            s_Value.firmwareUpdateFailureEmulationMode = 0;

            s_Value.isBleDisabled = false;
#else
            s_Value.enablesDebugPad = true;
            ::nn::settings::fwdbg::GetSettingsItemValue(
                &s_Value.enablesDebugPad, sizeof(s_Value.enablesDebugPad),
                SettingsName, "enables_debugpad");

            s_Value.managesDevices = true;
            ::nn::settings::fwdbg::GetSettingsItemValue(
                &s_Value.managesDevices, sizeof(s_Value.managesDevices),
                SettingsName, "manages_devices");

            s_Value.managesTouchIcI2c = s_Value.managesDevices;
            if (s_Value.managesTouchIcI2c == true)
            {
                ::nn::settings::fwdbg::GetSettingsItemValue(
                    &s_Value.managesTouchIcI2c, sizeof(s_Value.managesTouchIcI2c),
                    SettingsName, "manages_touch_ic_i2c");
            }

            s_Value.emulateFutureDevice = false;
            ::nn::settings::fwdbg::GetSettingsItemValue(
                &s_Value.emulateFutureDevice, sizeof(s_Value.emulateFutureDevice),
                SettingsName, "emulate_future_device");

            s_Value.emulateHardwareErrorForMcu = false;
            ::nn::settings::fwdbg::GetSettingsItemValue(
                &s_Value.emulateHardwareErrorForMcu, sizeof(s_Value.emulateHardwareErrorForMcu),
                SettingsName, "emulate_mcu_hardware_error");
            ::nn::xcd::SetMcuHardwareErrorEmulationEnabled(s_Value.emulateHardwareErrorForMcu);

            s_Value.emulateFailureForFirmwareUpdate = false;
            ::nn::settings::fwdbg::GetSettingsItemValue(
                &s_Value.emulateFailureForFirmwareUpdate, sizeof(s_Value.emulateFailureForFirmwareUpdate),
                SettingsName, "emulate_firmware_update_failure");

            {
                const char key[] = "firmware_update_failure_emulation_mode";
                s_Value.firmwareUpdateFailureEmulationMode = 0;
                auto size = ::nn::settings::fwdbg::GetSettingsItemValueSize(SettingsName, key);
                if (size > 0)
                {
                    ::nn::settings::fwdbg::GetSettingsItemValue(
                        &s_Value.firmwareUpdateFailureEmulationMode,
                        std::min(size, sizeof(s_Value.firmwareUpdateFailureEmulationMode)),
                        SettingsName,
                        key);
                }
            }

            // 更新失敗モードの設定
            if (s_Value.emulateFailureForFirmwareUpdate)
            {
                ::nn::xcd::SetBtFirmwareUpdateFailureEmulationEnabled(
                    (s_Value.firmwareUpdateFailureEmulationMode & FirmwareUpdateFailureMode_Bluetooth) != 0);
                ::nn::xcd::SetMcuUpdateFailureEmulationEnabled(
                    (s_Value.firmwareUpdateFailureEmulationMode & FirmwareUpdateFailureMode_Mcu) != 0);
            }

            s_Value.isBleDisabled = false;
            ::nn::settings::fwdbg::GetSettingsItemValue(
                &s_Value.isBleDisabled, sizeof(s_Value.isBleDisabled),
                SettingsName, "ble_disabled");
            if (s_Value.isBleDisabled == true)
            {
                ::nn::xcd::DisableBle();
            }
#endif
            s_IsInitialized = true;
        }
    }

    return s_Value;
}

} // namespace

}}} // namespace nn::hid::detail
