﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/debug/hid_CaptureButton.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/hid/debug/hid_HomeButton.h>
#include <nn/hid/debug/hid_Keyboard.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/hid/system/hid_InputDetection.h>
#include <nn/hid/system/hid_Keyboard.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_Thread.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/settings/settings_Xpad.h>

#include "hid_AbstractedAutoPilotPad.h"
#include "hid_AbstractedAutoPilotPadManager.h"
#include "hid_AbstractedAutoPilotVirtualPad.h"
#include "hid_AbstractedAutoPilotVirtualPadManager.h"
#include "hid_AbstractedGenericPad-os.win.h"
#include "hid_AbstractedPadHidShellManager-os.win.h"
#include "hid_AbstractedPadManager.h"
#include "hid_AbstractedPadXcd.h"
#include "hid_AbstractedPadXcdManager.h"
#include "hid_ActivationCount.h"
#include "hid_AppletResourceManager.h"
#include "hid_AudioControlManager.h"
#include "hid_CaptureButtonManager.h"
#include "hid_ConsoleSixAxisSensorAppletSettingManager.h"
#include "hid_ConsoleSixAxisSensorDriver.win32.h"
#include "hid_ConsoleSixAxisSensorManager.h"
#include "hid_ConsoleSixAxisSensorProcessor.h"
#include "hid_DebugPadDriver-os.win.h"
#include "hid_DebugPadManager.h"
#include "hid_DebugPadMappingManager.h"
#include "hid_GcAdapterDriver.h"
#include "hid_HandheldManager.h"
#include "hid_HomeButtonManager.h"
#include "hid_IConsoleSixAxisSensorDriver.h"
#include "hid_IDebugPadDriver.h"
#include "hid_IKeyboardDriver.h"
#include "hid_IMouseDriver.h"
#include "hid_InterruptSceneNotifier.h"
#include "hid_InputDetectorManager.h"
#include "hid_IrSensorXcdDriver.h"
#include "hid_ITouchScreenDriver.h"
#include "hid_IXpadDriver.h"
#include "hid_KeyboardDriver-os.win.h"
#include "hid_KeyboardManager.h"
#include "hid_MouseDriver-os.win.h"
#include "hid_MouseManager.h"
#include "hid_MultiWaitEvent.h"
#include "hid_NpadResourceManager.h"
#include "hid_OvlnSenderManager.h"
#include "hid_PalmaResourceManager.h"
#include "hid_PlayReportManager.h"
#include "hid_RegisteredDeviceManager.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_SixAxisSensorAppletSettingManager.h"
#include "hid_SixAxisSensorXcdDriver.h"
#include "hid_SleepButtonDriver.win32.h"
#include "hid_SleepButtonManager.h"
#include "hid_Task.h"
#include "hid_TouchScreenDriver-os.win.h"
#include "hid_TouchScreenManager.h"
#include "hid_UniquePadResourceManager.h"
#include "hid_VibratorXcdDriver.h"
#include "hid_WindowsGenericPadBroker-os.win.h"
#include "hid_XcdDriver.h"
#include "hid_XpadDriver.win32.h"
#include "hid_XpadIdManager.h"
#include "hid_XpadLifo.h"
#include "hid_XpadManager.h"
#include "hid_XpadMappingManager.h"
#include "hid_AudioControlDriver.h"

namespace nn { namespace hid { namespace detail {

class WindowsGenericPadBrokerTask;

//!< 共有タスクを扱うクラスです。
class SharedTask final : public Task
{
private:
    //!< Keyboard 操作排他用のミューテックス
    ::nn::os::SdkMutex m_KeyboardMutex;

    //!< Mouse 操作排他用のミューテックス
    ::nn::os::SdkMutex m_MouseMutex;

    //!< TouchScreen 操作排他用のミューテックス
    ::nn::os::SdkMutex m_TouchScreenMutex;

    //!< ホームボタン操作排他用のミューテックス
    ::nn::os::SdkMutex m_HomeButtonMutex;

    //!< スリープボタン操作排他用のミューテックス
    ::nn::os::SdkMutex m_SleepButtonMutex;

    //!< 撮影ボタン操作排他用のミューテックス
    ::nn::os::SdkMutex m_CaptureButtonMutex;

    //!< ConsoleSixAxisSensor 操作排他用のミューテックス
    ::nn::os::SdkMutex m_ConsoleSixAxisSensorMutex;

    //!< InputDetector 操作排他用のミューテックス
    ::nn::os::SdkMutex m_InputDetectorMutex;

    //!< RegisteredDevice 操作排他用のミューテックス
    ::nn::os::SdkMutex m_RegisteredDeviceMutex;

    //!< オーディオコントロール操作排他用のミューテックス
    ::nn::os::SdkMutex m_AudioControlMutex;

    //!< Keyboard のタイマーイベント
    SynchronizedTimer m_KeyboardTimerEvent;

    //!< Mouse のタイマーイベント
    SynchronizedTimer m_MouseTimerEvent;

    //!< TouchScreen のタイマーイベント
    MultiWaitTimerEvent m_TouchScreenTimerEvent;

    //!< ホームボタンのタイマーイベント
    SynchronizedTimer m_HomeButtonTimerEvent;

    //!< スリープボタンのタイマーイベント
    SynchronizedTimer m_SleepButtonTimerEvent;

    //!< 撮影ボタンのタイマーイベント
    SynchronizedTimer m_CaptureButtonTimerEvent;

    //!< プレイレポートのタイマーイベント
    SynchronizedTimer m_PlayReportTimerEvent;

    //!< ConsoleSixAxisSensor のタイマーイベント
    SynchronizedTimer m_ConsoleSixAxisSensorTimerEvent;

    //!< Npad / UniquePad 共通のタイマーイベント
    SynchronizedTimer m_PadCommonTimerEvent;

    //!< Pad 共通のタイマーイベントがアクティブ化された回数
    ActivationCount m_PadCommonTimerEventActivationCount;

    //!< Pad 共通のタイマーイベントの動作間隔
    static const nn::TimeSpan PadCommonTimerInterval;

    //!< Xcd の有線ペアリング完了通知用のシステムイベント
    MultiWaitSystemEvent m_PairingCompleteEvent;

    //!< Xcd のコントローラーへの接続タイムアウト通知用のシステムイベント
    MultiWaitSystemEvent m_ConnectionTriggerTimeoutEvent;

    //!< Bluetooth ファームウェア更新の完了検知用のシステムイベント
    MultiWaitSystemEvent m_BluetoothFirmwareUpdateSystemEvent;

    //!< Mcu ファームウェア更新の完了検知用のシステムイベント
    MultiWaitSystemEvent m_McuFirmwareUpdateSystemEvent;

    //!< ファームウェア更新中のサンプル受信検知用のシステムイベント
    MultiWaitSystemEvent m_FirmwareUpdateNewSampleSystemEvent;

    //!< ファームウェア更新中のリセットタイムアウト検知用のタイマーイベント
    MultiWaitTimerEvent m_FirmwareUpdateResetTimeoutEvent;

    //!< Xcd の状態変化検知用のシステムイベント
    MultiWaitSystemEvent m_XcdDeviceUpdateEvent;

    //!< オーディオコントロールのタイマーイベント
    SynchronizedTimer m_AudioControlTimerEvent;

    //!< Keyboard マネージャ
    KeyboardManager* m_pKeyboardManager;

    //!< Mouse マネージャ
    MouseManager* m_pMouseManager;

    //!< TouchScreen マネージャ
    TouchScreenManager* m_pTouchScreenManager;

    //!< ホームボタンマネージャ
    HomeButtonManager* m_pHomeButtonManager;

    //!< スリープボタンマネージャ
    SleepButtonManager* m_pSleepButtonManager;

    //!< 撮影ボタンマネージャ
    CaptureButtonManager* m_pCaptureButtonManager;

    //!< ConsoleSixAxisSensor マネージャ
    ConsoleSixAxisSensorManager* m_pConsoleSixAxisSensorManager;

    //!< InputDetector マネージャ
    InputDetectorManager* m_pInputDetectorManager;

    //!< プレイレポートマネージャー
    PlayReportManager* m_pPlayReportManager;

    //!< 登録情報マネージャー
    RegisteredDeviceManager* m_pRegisteredDeviceManager;

    //!< Xcd の抽象化インタフェース
    AbstractedPadXcd* m_pAbstractedPadXcd;

    //!< Xcd の抽象化 インタフェースの数
    int m_AbstractedPadXcdCount;

    //!< Xcd の割り当てマネージャー
    AbstractedPadXcdManager* m_pAbstractedPadXcdManager;

    //!< 自動入力の抽象化 Pad を管理するマネージャー
    AbstractedAutoPilotPadManager* m_pAbstractedAutoPilotPadManager;

    //!< 自動入力の抽象化 Pad を管理するマネージャー
    AbstractedAutoPilotVirtualPadManager* m_pAbstractedAutoPilotVirtualPadManager;

    //!< AbstractedPad の HidShell 対応を管理するマネージャー
    AbstractedPadHidShellManager* m_pAbstractedPadHidShellManager;

    //!< Npad リソースマネージャー
    NpadResourceManager* m_pNpadResourceManager;

    //!< UniquePad リソースマネージャー
    UniquePadResourceManager* m_pUniquePadResourceManager;

    //!< オーディオコントロールマネージャ
    AudioControlManager* m_pAudioControlManager;

public:
    SharedTask() NN_NOEXCEPT;

    virtual ~SharedTask() NN_NOEXCEPT NN_OVERRIDE;

    //!< Npad リソースマネージャを設定します。
    void SetNpadResourceManager(NpadResourceManager* pManager) NN_NOEXCEPT;

    //!< UniquePad リソースマネージャを設定します。
    void SetUniquePadResourceManager(UniquePadResourceManager* pManager) NN_NOEXCEPT;

    //!< Keyboard マネージャを設定します。
    void SetKeyboardManager(KeyboardManager* pManager) NN_NOEXCEPT;

    //!< Mouse マネージャを設定します。
    void SetMouseManager(MouseManager* pManager) NN_NOEXCEPT;

    //!< TouchScreen マネージャを設定します。
    void SetTouchScreenManager(TouchScreenManager* pManager) NN_NOEXCEPT;

    //!< ホームボタンマネージャを設定します。
    void SetHomeButtonManager(HomeButtonManager* pManager) NN_NOEXCEPT;

    //!< スリープボタンマネージャを設定します。
    void SetSleepButtonManager(SleepButtonManager* pManager) NN_NOEXCEPT;

    //!< 撮影ボタンマネージャを設定します。
    void SetCaptureButtonManager(CaptureButtonManager* pManager) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor マネージャを設定します。
    void SetConsoleSixAxisSensorManager(ConsoleSixAxisSensorManager* pManager) NN_NOEXCEPT;

    //!< プレイレポートマネージャーを設定します。
    void SetPlayReportManager(PlayReportManager* pManager) NN_NOEXCEPT;

    //!< 登録情報マネージャーを設定します。
    void SetRegisteredDeviceManager(RegisteredDeviceManager* pManager) NN_NOEXCEPT;

    //!< InputDetector マネージャを設定します。
    void SetInputDetectorManager(InputDetectorManager* pManager) NN_NOEXCEPT;

    //!< Xcd の抽象化インタフェースを設定します。
    void SetAbstractedPadXcd(AbstractedPadXcd* pXcd, int count) NN_NOEXCEPT;

    //!< Xcd の割り当てマネージャーを設定します。
    void SetAbstractedPadXcdManager(AbstractedPadXcdManager* pManager) NN_NOEXCEPT;

    //!< 自動入力の抽象化 Pad を管理するマネージャーを設定します。
    void SetAbstractedAutoPilotPadManager(AbstractedAutoPilotPadManager* pManager) NN_NOEXCEPT;

    //!< 自動入力の抽象化 Pad を管理するマネージャーを設定します。
    void SetAbstractedAutoPilotVirtualPadManager(
        AbstractedAutoPilotVirtualPadManager* pManager) NN_NOEXCEPT;

    //!< AbstractedPad の HidShell 対応を管理するマネージャーを設定します。
    void SetAbstractedPadHidShellManager(
        AbstractedPadHidShellManager* pManager) NN_NOEXCEPT;

    //!< オーディオコントロールマネージャーを設定します。
    void SetAudioControlManager(AudioControlManager* pManager) NN_NOEXCEPT;

    //!< Keyboard をアクティブ化します。
    ::nn::Result ActivateKeyboard() NN_NOEXCEPT;

    //!< Keyboard を非アクティブ化します。
    ::nn::Result DeactivateKeyboard() NN_NOEXCEPT;

    //!< Keyboard のアプレットリソースの状態を保証します。
    ::nn::Result EnsureKeyboardAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Keyboard に自動操作状態を設定します。
    ::nn::Result SetKeyboardAutoPilotState(
        const ::nn::hid::debug::KeyboardAutoPilotState& value) NN_NOEXCEPT;

    //!< Keyboard の自動操作状態を解除します。
    ::nn::Result UnsetKeyboardAutoPilotState() NN_NOEXCEPT;

    //!< Keyboard のロックキーイベントを送信します。
    ::nn::Result SendKeyboardLockKeyEvent(
        ::nn::hid::system::KeyboardLockKeyEventSet value) NN_NOEXCEPT;

    //!< Mouse をアクティブ化します。
    ::nn::Result ActivateMouse() NN_NOEXCEPT;

    //!< Mouse を非アクティブ化します。
    ::nn::Result DeactivateMouse() NN_NOEXCEPT;

    //!< Mouse のアプレットリソースの状態を保証します。
    ::nn::Result EnsureMouseAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Mouse に自動操作状態を設定します。
    ::nn::Result SetMouseAutoPilotState(
        const ::nn::hid::debug::MouseAutoPilotState& value) NN_NOEXCEPT;

    //!< Mouse の自動操作状態を解除します。
    ::nn::Result UnsetMouseAutoPilotState() NN_NOEXCEPT;

    //!< Gesture をアクティブ化します。
    ::nn::Result ActivateGesture() NN_NOEXCEPT;

    //!< Gesture を非アクティブ化します。
    ::nn::Result DeactivateGesture() NN_NOEXCEPT;

    //!< Gesture のアプレットリソースの状態を保証します。
    ::nn::Result EnsureGestureAppletResource(
        ::nn::applet::AppletResourceUserId aruid, int32_t version) NN_NOEXCEPT;

    //!< TouchScreen をアクティブ化します。
    ::nn::Result ActivateTouchScreen() NN_NOEXCEPT;

    //!< TouchScreen を非アクティブ化します。
    ::nn::Result DeactivateTouchScreen() NN_NOEXCEPT;

    //!< TouchScreen のアプレットリソースの状態を保証します。
    ::nn::Result EnsureTouchScreenAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< TouchScreen に自動操作状態を設定します。
    ::nn::Result SetTouchScreenAutoPilotState(
        const ::nn::hid::debug::TouchScreenAutoPilotState<
            TouchStateCountMax>& value) NN_NOEXCEPT;

    //!< TouchScreen の自動操作状態を解除します。
    ::nn::Result UnsetTouchScreenAutoPilotState() NN_NOEXCEPT;

    //!< ホームボタンの入力通知のハンドルを確保します。
    ::nn::Result AcquireHomeButtonEventHandle(
        ::nn::os::NativeHandle* pOutValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ホームボタンをアクティブ化します。
    ::nn::Result ActivateHomeButton() NN_NOEXCEPT;

    //!< ホームボタンを非アクティブ化します。
    ::nn::Result DeactivateHomeButton() NN_NOEXCEPT;

    //!< ホームボタンのアプレットリソースの状態を保証します。
    ::nn::Result EnsureHomeButtonAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ホームボタンに自動操作状態を設定します。
    ::nn::Result SetHomeButtonAutoPilotState(
        const ::nn::hid::debug::HomeButtonAutoPilotState& value) NN_NOEXCEPT;

    //!< ホームボタンの自動操作状態を解除します。
    ::nn::Result UnsetHomeButtonAutoPilotState() NN_NOEXCEPT;

    //!< スリープボタンの入力通知のハンドルを確保します。
    ::nn::Result AcquireSleepButtonEventHandle(
        ::nn::os::NativeHandle* pOutValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< スリープボタンをアクティブ化します。
    ::nn::Result ActivateSleepButton() NN_NOEXCEPT;

    //!< スリープボタンを非アクティブ化します。
    ::nn::Result DeactivateSleepButton() NN_NOEXCEPT;

    //!< スリープボタンのアプレットリソースの状態を保証します。
    ::nn::Result EnsureSleepButtonAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< スリープボタンに自動操作状態を設定します。
    ::nn::Result SetSleepButtonAutoPilotState(
        const ::nn::hid::debug::SleepButtonAutoPilotState& value) NN_NOEXCEPT;

    //!< スリープボタンの自動操作状態を解除します。
    ::nn::Result UnsetSleepButtonAutoPilotState() NN_NOEXCEPT;

    //!< 撮影ボタンの入力通知のハンドルを確保します。
    ::nn::Result AcquireCaptureButtonEventHandle(
        ::nn::os::NativeHandle* pOutValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 撮影ボタンをアクティブ化します。
    ::nn::Result ActivateCaptureButton() NN_NOEXCEPT;

    //!< 撮影ボタンを非アクティブ化します。
    ::nn::Result DeactivateCaptureButton() NN_NOEXCEPT;

    //!< 撮影ボタンのアプレットリソースの状態を保証します。
    ::nn::Result EnsureCaptureButtonAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 撮影ボタンに自動操作状態を設定します。
    ::nn::Result SetCaptureButtonAutoPilotState(
        const ::nn::hid::debug::CaptureButtonAutoPilotState& value) NN_NOEXCEPT;

    //!< 撮影ボタンの自動操作状態を解除します。
    ::nn::Result UnsetCaptureButtonAutoPilotState() NN_NOEXCEPT;

    //!< Pro Controller 有線 USB 通信機能が有効かどうかを取得します。
    ::nn::Result IsUsbFullKeyControllerEnabled(bool* pOutEnabled) NN_NOEXCEPT;

    //!< Pro Controller 有線 USB 通信機能を有効にします。
    ::nn::Result EnableUsbFullKeyController(bool enabled) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor をアクティブ化します。
    ::nn::Result ActivateConsoleSixAxisSensor() NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor を非アクティブ化します。
    ::nn::Result DeactivateConsoleSixAxisSensor() NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor の生値を取得します。
    ::nn::Result GetConsoleSixAxisSensorCountStates(
        int* pOutCount,
        ::nn::hid::debug::ConsoleSixAxisSensorCountState* pOutValues, int count
        ) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor のサンプリング間隔を取得します。
    ::nn::Result GetConsoleSixAxisSensorSamplingFrequency(
        int64_t* pOutValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor のアプレットリソースの状態を保証します。
    ::nn::Result EnsureConsoleSixAxisSensorAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 廃止予定: ConsoleSixAxisSensor のキャリブレーション値を取得します。
    ::nn::Result GetConsoleSixAxisSensorCalibrationValues(
        tmp::ConsoleSixAxisSensorCalibrationValues* pOutValues) NN_NOEXCEPT;

    //!< 本体 6 軸センサーのユーザーキャリブレーション機能のサポート有無を取得します。
    ::nn::Result IsConsoleSixAxisSensorUserCalibrationSupported(bool* pOutIsSupported) NN_NOEXCEPT;

    //!< 工程出荷時の本体 6 軸センサーのキャリブレーション値を書き込みます。
    ::nn::Result ResetConsoleSixAxisSensorCalibrationValues() NN_NOEXCEPT;

    //!< 本体 6 軸センサーのユーザーキャリブレーション処理を開始します。
    ::nn::Result StartConsoleSixAxisSensorUserCalibration() NN_NOEXCEPT;

    //!< 進行中の本体 6 軸センサーのユーザーキャリブレーション処理をキャンセルします。
    ::nn::Result CancelConsoleSixAxisSensorUserCalibration() NN_NOEXCEPT;

    //!< 6 軸センサーの高精度ユーザーキャリブレーションの進捗を取得します。
    ::nn::Result GetSixAxisSensorAccurateUserCalibrationState(
        ::nn::hid::system::SixAxisSensorAccurateUserCalibrationState* pOutState
        ) NN_NOEXCEPT;

    //!< アプリから割り当てられた入力状態格納用バッファの Handle を設定します。
    ::nn::Result SetStateBufferMemoryHandle(
        const ::nn::applet::AppletResourceUserId& aruid,
        ::nn::os::NativeHandle&& transferMemoryHandle,
        size_t size, bool isManaged) NN_NOEXCEPT;

    //!< アプリから割り当てられたワークバッファの Handle を設定します。
    ::nn::Result SetWorkBufferMemoryHandle(
        const ::nn::applet::AppletResourceUserId& aruid,
        ::nn::os::NativeHandle&& transferMemoryHandle,
        size_t size, bool isManaged) NN_NOEXCEPT;

    //!< Seven 向け SixAxisSensor を初期化します。
    ::nn::Result InitializeSevenSixAxisSensor(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Seven 向け SixAxisSensor を終了します。
    ::nn::Result FinalizeSevenSixAxisSensor(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< InputDetector をアクティブ化します。
    ::nn::Result ActivateInputDetector() NN_NOEXCEPT;

    //!< InputDetector を非アクティブ化します。
    ::nn::Result DeactivateInputDetector() NN_NOEXCEPT;

    //!< InputDetector に入力変化を通知します。
    ::nn::Result NotifyInputDetector(const system::InputSourceIdSet& id) NN_NOEXCEPT;

    //!< InputDetector のアプレットリソースの状態を保証します。
    ::nn::Result EnsureInputDetectorAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< プレイレポート送信用のサービスをアクティブ化します。
    void ActivatePlayReport() NN_NOEXCEPT;

    //!< プレイレポート送信用のサービスをを非アクティブにします。
    void DeactivatePlayReport() NN_NOEXCEPT;

    //!< コントローラーの登録処理のサービスをアクティブ化します。
    void ActivateRegisteredDeviceManagement() NN_NOEXCEPT;

    //!< コントローラーの登録処理のサービスをを非アクティブにします。
    void DeactivateRegisteredDeviceManagement() NN_NOEXCEPT;

    //!< コントローラーの利用状態が変化したときに受ける通知をイベントオブジェクトにバインドします
    Result AcquirePlayReportControllerUsageUpdateEventHandle(
        ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< 現在のコントローラーの利用状態の取得を行います。
    int GetPlayReportControllerUsages(
        system::PlayReportControllerUsage* pOutValues, int count) NN_NOEXCEPT;

    //!< コントローラーの登録情報が変化したときに受ける通知をイベントオブジェクトにバインドします
    Result AcquirePlayReportRegisteredDevicesEventHandle(
        ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< 現在のコントローラーの登録情報の取得を行います。【互換性維持用】
    int GetRegisteredDevices(system::RegisteredDeviceOld* pOutValues, int count) NN_NOEXCEPT;

    //!< 現在のコントローラーの登録情報の取得を行います。
    int GetRegisteredDevices(system::RegisteredDevice* pOutValues, int count) NN_NOEXCEPT;

    //!< コントローラーの登録情報を追加します。
    void AddRegisteredDevices(const system::RegisteredDevice& device) NN_NOEXCEPT;

    //!< コントローラーへの接続処理がタイムアウトした際の通知をうけるイベントハンドルを取得します
    ::nn::Result AcquireConnectionTriggerTimeoutEvent(
        ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< コントローラーに対して接続を試みます
    ::nn::Result SendConnectionTrigger(nn::bluetooth::Address& address) NN_NOEXCEPT;

    //!< コントローラーが新たに登録された際に通知をうけるイベントハンドルを取得します
    ::nn::Result AcquireDeviceRegisteredEventForControllerSupport(
        ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< AbstractedXcdManager をアクティブ化します。
    ::nn::Result ActivateAbstractedXcdManager() NN_NOEXCEPT;

    //!< AbstractedXcdManager を非アクティブ化します。
    ::nn::Result DeactivateAbstractedXcdManager() NN_NOEXCEPT;

    //!< 抽象化されたコントローラーに BasicXapd 相当の自動操作状態を設定します。
    ::nn::Result SetAbstractedPadAutoPilotStateWithBasicXpad(
        int playerNumber,
        const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT;

    //!< 抽象化されたコントローラーの自動操作状態を解除します。
    ::nn::Result UnsetAbstractedPadAutoPilotState(int playerNumber) NN_NOEXCEPT;

    //!< 抽象化されたコントローラーに自動操作状態を設定します。
    ::nn::Result SetAbstractedPadAutoPilotVirtualPadState(
        const ::nn::hid::debug::AbstractedVirtualPadId& virtualPadId,
        const ::nn::hid::debug::AbstractedPadState& value) NN_NOEXCEPT;

    //!< 抽象化されたコントローラーの自動操作状態を解除します。
    ::nn::Result UnsetAbstractedPadAutoPilotVirtualPadState(
        const ::nn::hid::debug::AbstractedVirtualPadId& virtualPadId
        ) NN_NOEXCEPT;

    //<! 抽象化されたコントローラーの自動操作状態を全て解除します。
    ::nn::Result UnsetAllAbstractedPadAutoPilotVirtualPadState() NN_NOEXCEPT;

    //!< Pad 共通のタイマーを初期化します
    ::nn::Result ActivatePadCommonTimer() NN_NOEXCEPT;

    //!< Pad 共通のタイマーを初期化します
    ::nn::Result DeactivatePadCommonTimer() NN_NOEXCEPT;

    ::nn::Result ActivateAudioControl() NN_NOEXCEPT;

    ::nn::Result DeactivateAudioControl() NN_NOEXCEPT;

    //!< オーディオコントロールの入力通知のハンドルを確保します。
    ::nn::Result AcquireAudioControlEventHandle(
        ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    ::nn::Result GetAudioControlStates(
        int* pOutCount,
        system::AudioControlState outStates[], int count) NN_NOEXCEPT;

protected:
    //!< タイマーイベントをリンクします。
    virtual void LinkTimerEvent(
        ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE;

    //!< タイマーイベントをアンリンクします。
    virtual void UnlinkTimerEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< サンプリング処理を実行します。
    virtual void DoSampling(
        const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT NN_OVERRIDE;

    virtual void LinkTimers(SynchronizedTimerGroup* group) NN_NOEXCEPT NN_OVERRIDE;

    virtual void UnlinkTimers() NN_NOEXCEPT NN_OVERRIDE;

    virtual void DoSamplingForTimer(const SynchronizedTimer* waitId) NN_NOEXCEPT NN_OVERRIDE;

private:
    //!< Pad 共通のタイマーイベントをアクティブ化します。
    ::nn::Result ActivatePadCommonTimerEvent() NN_NOEXCEPT;

    //!< Pad 共通のタイマーイベントをアクティブ化します。
    ::nn::Result DeactivatePadCommonTimerEvent() NN_NOEXCEPT;

    //!< Keyboard のタイマーイベントを解決します。
    bool ResolveKeyboardTimerEvent(const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< Mouse のタイマーイベントを解決します。
    bool ResolveMouseTimerEvent(const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< ホームボタンのタイマーイベントを解決します。
    bool ResolveHomeButtonTimerEvent(
        const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< スリープボタンのタイマーイベントを解決します。
    bool ResolveSleepButtonTimerEvent(
        const SynchronizedTimer* waitId) NN_NOEXCEPT;

    //!< 撮影ボタンのタイマーイベントを解決します。
    bool ResolveCaptureButtonTimerEvent(
        const SynchronizedTimer* waitId) NN_NOEXCEPT;
};

//!< DebugPad のタスクを扱うクラスです。
class DebugPadTask final : public Task
{
private:
    //!< DebugPad 操作排他用のミューテックス
    ::nn::os::SdkMutex m_DebugPadMutex;

    //!< DebugPad のタイマーイベント
    SynchronizedTimer m_DebugPadTimerEvent;

    //!< DebugPad マネージャ
    DebugPadManager* m_pDebugPadManager;

    //!< WindowsGenericPadBroker タスク
    WindowsGenericPadBrokerTask* m_pWindowsGenericPadBrokerTask;

public:
    DebugPadTask() NN_NOEXCEPT;

    virtual ~DebugPadTask() NN_NOEXCEPT NN_OVERRIDE;

    //!< DebugPad マネージャを設定します。
    void SetDebugPadManager(DebugPadManager* pManager) NN_NOEXCEPT;

    //!< WindowsGenericPadBroker タスクを設定します。
    void SetWindowsGenericPadBrokerTask(
        WindowsGenericPadBrokerTask* pWindowsGenericPadBrokerTask) NN_NOEXCEPT;

    //!< DebugPad をアクティブ化します。
    ::nn::Result ActivateDebugPad() NN_NOEXCEPT;

    //!< DebugPad を非アクティブ化します。
    ::nn::Result DeactivateDebugPad() NN_NOEXCEPT;

    //!< DebugPad のアプレットリソースの状態を保証します。
    ::nn::Result EnsureDebugPadAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< DebugPad に自動操作状態を設定します。
    ::nn::Result SetDebugPadAutoPilotState(
        const ::nn::hid::debug::DebugPadAutoPilotState& value) NN_NOEXCEPT;

    //!< DebugPad の自動操作状態を解除します。
    ::nn::Result UnsetDebugPadAutoPilotState() NN_NOEXCEPT;

protected:
    //!< タイマーイベントをリンクします。
    virtual void LinkTimerEvent(
        ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE;

    //!< タイマーイベントをアンリンクします。
    virtual void UnlinkTimerEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< サンプリング処理を実行します。
    virtual void DoSampling(
        const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT NN_OVERRIDE;

    virtual void LinkTimers(SynchronizedTimerGroup* group) NN_NOEXCEPT NN_OVERRIDE;

    virtual void UnlinkTimers() NN_NOEXCEPT NN_OVERRIDE;

    virtual void DoSamplingForTimer(const SynchronizedTimer* waitId) NN_NOEXCEPT NN_OVERRIDE;
};

//!< Xpad のタスクを扱うクラスです。
class XpadTask final : public Task
{
private:
    //!< Xpad 操作排他用のミューテックス
    ::nn::os::SdkMutex m_XpadMutex;

    //!< Xpad のタイマーイベント
    SynchronizedTimer m_XpadTimerEvent;

    //!< Xpad マネージャ
    XpadManager* m_pXpadManager;

    //!< WindowsGenericPadBroker タスク
    WindowsGenericPadBrokerTask* m_pWindowsGenericPadBrokerTask;

public:
    XpadTask() NN_NOEXCEPT;

    virtual ~XpadTask() NN_NOEXCEPT NN_OVERRIDE;

    //!< Xpad マネージャを設定します。
    void SetXpadManager(XpadManager* pManager) NN_NOEXCEPT;

    //!< WindowsGenericPadBroker タスクを設定します。
    void SetWindowsGenericPadBrokerTask(
        WindowsGenericPadBrokerTask* pWindowsGenericPadBrokerTask) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad をアクティブ化します。
    ::nn::Result ActivateBasicXpad() NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad を非アクティブ化します。
    ::nn::Result DeactivateBasicXpad() NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad のサンプリングを有効化します。
    void EnableBasicXpadSampling() NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad のアプレットリソースの状態を保証します。
    ::nn::Result EnsureBasicXpadAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad に自動操作状態を設定します。
    ::nn::Result SetBasicXpadAutoPilotState(
        const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad の自動操作状態を解除します。
    ::nn::Result UnsetBasicXpadAutoPilotState() NN_NOEXCEPT;

    //!< JoyXpad をアクティブ化します。
    ::nn::Result ActivateJoyXpad() NN_NOEXCEPT;

    //!< JoyXpad を非アクティブ化します。
    ::nn::Result DeactivateJoyXpad() NN_NOEXCEPT;

    //!< JoyXpad のサンプリングを有効化します。
    void EnableJoyXpadSampling() NN_NOEXCEPT;

    //!< JoyXpad の LIFO のハンドルを取得します。
    ::nn::Result GetJoyXpadLifoHandle(
        ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< 6軸センサーをアクティブ化します。
    ::nn::Result ActivateSixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーを非アクティブ化します。
    ::nn::Result DeactivateSixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーの LIFO のハンドルを取得します。
    ::nn::Result GetSixAxisSensorLifoHandle(
        ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< 6軸センサーをアクティブ化します。
    ::nn::Result ActivateJoySixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーを非アクティブ化します。
    ::nn::Result DeactivateJoySixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーの LIFO のハンドルを取得します。
    ::nn::Result GetJoySixAxisSensorLifoHandle(
        ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

protected:
    //!< タイマーイベントをリンクします。
    virtual void LinkTimerEvent(
        ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE;

    //!< タイマーイベントをアンリンクします。
    virtual void UnlinkTimerEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< サンプリング処理を実行します。
    virtual void DoSampling(
        const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT NN_OVERRIDE;

    virtual void LinkTimers(
        SynchronizedTimerGroup* group) NN_NOEXCEPT NN_OVERRIDE;

    virtual void UnlinkTimers() NN_NOEXCEPT NN_OVERRIDE;

    virtual void DoSamplingForTimer(
        const SynchronizedTimer* waitId) NN_NOEXCEPT NN_OVERRIDE;
};

//!< GenericPad に依存した AbstractedPad を扱うタスクです
class AbstractedGenericPadTask final : public Task
{
private:
    //!< AbstractedGenericPad 操作排他用のミューテックス
    ::nn::os::SdkMutex m_AbstractedGenericPadMutex;

    //!< AbstractedGenericPad のタイマーイベント
    MultiWaitTimerEvent m_AbstractedGenericPadTimerEvent;

    //!< WindowsGenericPadBroker タスク
    WindowsGenericPadBrokerTask* m_pWindowsGenericPadBrokerTask;

    //!< GenericPad の抽象化インタフェース
    AbstractedGenericPad* m_pAbstractedGenericPad;

public:
    AbstractedGenericPadTask() NN_NOEXCEPT;

    virtual ~AbstractedGenericPadTask() NN_NOEXCEPT NN_OVERRIDE;

    //!< WindowsGenericPadBroker タスクを設定します。
    void SetWindowsGenericPadBrokerTask(
        WindowsGenericPadBrokerTask* pWindowsGenericPadBrokerTask) NN_NOEXCEPT;

    //!< AbstractedGenericPad をアクティブ化します。
    ::nn::Result ActivateAbstractedGenericPad() NN_NOEXCEPT;

    //!< AbstractedGenericPad を非アクティブ化します。
    ::nn::Result DeactivateAbstractedGenericPad() NN_NOEXCEPT;

    //!< GenericPad の抽象化インタフェースを設定します。
    void SetAbstractedGenericPad(AbstractedGenericPad* pGenricPad) NN_NOEXCEPT;

protected:
    //!< タイマーイベントをリンクします。
    virtual void LinkTimerEvent(
        ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE;

    //!< タイマーイベントをアンリンクします。
    virtual void UnlinkTimerEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< サンプリング処理を実行します。
    virtual void DoSampling(
        const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT NN_OVERRIDE;
};

//!< XcdDriver のタスクを扱うクラスです。
class XcdTask final : public Task
{
private:
    //!< Xcd 操作排他用のミューテックス
    ::nn::os::SdkMutex m_XcdMutex;

    //!< Xcd ドライバ
    XcdDriver* m_pXcdDriver;

public:
    XcdTask() NN_NOEXCEPT;

    virtual ~XcdTask() NN_NOEXCEPT NN_OVERRIDE;

    //!< XpadId マネージャを設定します。
    void SetXcdDriver(XcdDriver* pXcdDriver) NN_NOEXCEPT;

    //!< Xcd ドライバをアクティブ化します。
    ::nn::Result ActivateXcdDriver() NN_NOEXCEPT;

    //!< Xcd ドライバを非アクティブ化します。
    ::nn::Result DeactivateXcdDriver() NN_NOEXCEPT;

protected:
    //!< イベントをリンクします。(名称はいったんTaskを引き継ぎます)
    virtual void LinkTimerEvent(
        ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE;

    //!< イベントをアンリンクします。(名称はいったんTaskを引き継ぎます)
    virtual void UnlinkTimerEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< 定期処理を実行します。(名称はいったんTaskを引き継ぎます)
    virtual void DoSampling(
        const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT NN_OVERRIDE;

    //!< イベントの受付を開始します。
    virtual void InvokeEventLoop() NN_NOEXCEPT NN_OVERRIDE;
};

//!< WindowsGenericPadBroker のタスクを扱うクラスです。
class WindowsGenericPadBrokerTask final : public Task
{
private:
    //!< WindowsGenericPadBroker のイベント操作排他用のミューテックス
    ::nn::os::SdkMutex m_EventMutex;

    //!< WindowsGenericPadBroker の完了イベント
    ::nn::os::Event m_BrokerCompletionEvent;

    //!< WindowsGenericPadBroker のアクティブ化イベント
    MultiWaitEvent m_BrokerActivationEvent;

    //!< WindowsGenericPadBroker の非アクティブ化イベント
    MultiWaitEvent m_BrokerDeactivationEvent;

    //!< WindowsGenericPadBroker のタイマーイベント
    MultiWaitTimerEvent m_BrokerTimerEvent;

    //!< 汎用コントローラのブローカ
    WindowsGenericPadBroker* m_pWindowsGenericPadBroker;

public:
    WindowsGenericPadBrokerTask() NN_NOEXCEPT;

    virtual ~WindowsGenericPadBrokerTask() NN_NOEXCEPT NN_OVERRIDE;

    //!< 汎用コントローラのブローカを設定します。
    void SetWindowsGenericPadBroker(
        WindowsGenericPadBroker* pWindowsGenericPadBroker) NN_NOEXCEPT;

    //!< WindowsGenericPadBroker をアクティブ化します。
    ::nn::Result ActivateWindowsGenericPadBroker() NN_NOEXCEPT;

    //!< WindowsGenericPadBroker を非アクティブ化します。
    ::nn::Result DeactivateWindowsGenericPadBroker() NN_NOEXCEPT;

protected:
    //!< タイマーイベントをリンクします。
    virtual void LinkTimerEvent(
        ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT NN_OVERRIDE;

    //!< タイマーイベントをアンリンクします。
    virtual void UnlinkTimerEvent() NN_NOEXCEPT NN_OVERRIDE;

    //!< サンプリング処理を実行します。
    virtual void DoSampling(
        const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT NN_OVERRIDE;
};

//!< リソースマネージャを扱うクラスです。
class ResourceManager final
{
    NN_DISALLOW_COPY(ResourceManager);
    NN_DISALLOW_MOVE(ResourceManager);

private:
    //!< Xpad のプレイヤーの最大数
    static const int XpadPlayerCountMax = 4;

    //!< UniquePad スレッドの名前
    static const char* const s_UniquePadSamplerThreadName;

    //!< UniquePad スレッドの優先度
    static const int s_UniquePadTaskThreadPriority =
        ::nn::os::DefaultThreadPriority - 6;

    //!< 共有タスクのスレッドのスタック領域
    static NN_ALIGNAS(0x1000) char s_SharedTaskThreadStack[0x1000];

    //!< DebugPad タスクのスレッドのスタック領域
    static NN_ALIGNAS(0x1000) char s_DebugPadTaskThreadStack[0x1000];

    //!< Xpad タスクのスレッドのスタック領域
    static NN_ALIGNAS(0x1000) char s_XpadTaskThreadStacks[
        XpadPlayerCountMax][0x1000];

    //!< XCD を駆動させるスレッドのスタック領域
    static NN_ALIGNAS(0x1000) char s_XcdTaskThreadStack[0x2000];

    //!< WindowsGenericPadBroker タスクのスレッドのスタック領域
    static NN_ALIGNAS(0x1000) char s_WindowsGenericPadBrokerTaskThreadStack[
        0x1000];

    //!< AbstractedGenericPad タスクのスレッドのスタック領域
    static NN_ALIGNAS(0x1000) char s_AbstractedGenericPadTaskThreadStack[
        AbstractedGenericPadCountMax][0x1000];

    //!< 共有タスク
    SharedTask m_SharedTask;

    //!< DebugPad タスク
    DebugPadTask m_DebugPadTask;

    //!< Xpad タスク
    XpadTask m_XpadTasks[XpadPlayerCountMax];

    //!< Xcd タスク
    XcdTask m_XcdTask;

    //!< WindowsGenericPadBroker タスク
    WindowsGenericPadBrokerTask m_WindowsGenericPadBrokerTask;

    //!< AbstractedGenericPad タスク
    AbstractedGenericPadTask m_AbstractedGenericPadTask[AbstractedGenericPadCountMax];

    //!< DebugPad ドライバ
    DebugPadDriver m_DebugPadDriver;

    //!< Keyboard ドライバ
    KeyboardDriver m_KeyboardDriver;

    //!< Mouse ドライバ
    MouseDriver m_MouseDriver;

    //!< TouchScreen ドライバ
    TouchScreenDriver m_TouchScreenDriver;

    //!< Xpad ドライバ
    XpadDriver m_XpadDrivers[XpadPlayerCountMax];

    //!< スリープボタンドライバ
    SleepButtonDriver m_SleepButtonDriver;

    //!< ConsoleSixAxisSensor ドライバ
    ConsoleSixAxisSensorDriver m_ConsoleSixAxisSensorDriver;

    //!< オーディオコントロールドライバ
    AudioControlDriver m_AudioControlDriver;

    //!< DebugPad マネージャ
    DebugPadManager m_DebugPadManager;

    //!< Keyboard マネージャ
    KeyboardManager m_KeyboardManager;

    //!< Mouse マネージャ
    MouseManager m_MouseManager;

    //!< TouchScreen マネージャ
    TouchScreenManager m_TouchScreenManager;

    //!< Xpad マネージャ
    XpadManager m_XpadManagers[XpadPlayerCountMax];

    //!< InputDetector 操作排他用のミューテックス
    ::nn::os::SdkMutex m_InputDetectorMutex;

    //!< InputDetector マネージャ
    InputDetectorManager m_InputDetectorManager;

    //!< ホームボタンマネージャ
    HomeButtonManager m_HomeButtonManager;

    //!< スリープボタンマネージャ
    SleepButtonManager m_SleepButtonManager;

    //!< 撮影ボタンマネージャ
    CaptureButtonManager m_CaptureButtonManager;

    //!< ConsoleSixAxisSensor マネージャ
    ConsoleSixAxisSensorManager m_ConsoleSixAxisSensorManager;

    //!< アプレットリソース操作排他用のミューテックス
    ::nn::os::SdkRecursiveMutex m_AppletResourceMutex;

    //!< アプレットリソースマネージャ
    AppletResourceManager m_AppletResourceManager;

    //!< SixAxisSensor の設定値管理マネージャ
    SixAxisSensorAppletSettingManager m_SixAxisSensorAppletSettingManager;

    //!< ConsoleSixAxisSensor の設定値管理マネージャ
    ConsoleSixAxisSensorAppletSettingManager m_ConsoleSixAxisSensorAppletSettingManager;

    //!< Handheld マネージャ
    HandheldManager m_HandheldManager;

    //!< ConsoleSixAxisSensorProcessor
    ConsoleSixAxisSensorProcessor m_ConsoleSixAxisSensorProcessor;

    //!< DebugPad 用のマッピングマネージャ
    DebugPadMappingManager m_DebugPadMappingManager;

    //!< Xpad 用のマッピングマネージャ
    XpadMappingManager m_XpadMappingManagers[XpadPlayerCountMax];

    //!< Xpad ID マネージャのミューテックス
    ::nn::os::SdkMutex m_XpadIdManagerMutex;

    //!< Xpad ID マネージャ
    XpadIdManager m_XpadIdManager;

    //!< 汎用コントローラのブローカ
    WindowsGenericPadBroker m_WindowsGenericPadBroker;

    //!< Xcd 駆動用のドライバ
    XcdDriver m_XcdDriver;

    //!< オーバーレイ通知を送信するためのマネージャー
    OvlnSenderManager m_OvlnSender;

    //!< 割り込みシーン通知をハンドリングするクラス
    InterruptSceneNotifier m_InterruptSceneNotifier;

    //!< プレイレポートを送信するためのマネージャー
    PlayReportManager m_PlayReportManager;

    //!< コントローラーの登録情報を管理するためのマネージャー
    RegisteredDeviceManager m_RegisteredDeviceManager;

    //!< 自動入力の抽象化インタフェース
    AbstractedAutoPilotPad m_AbstractedAutoPilotPad[AbstractedAutoPilotPadCount];

    //!< 自動入力の抽象化 Pad を管理するマネージャー
    AbstractedAutoPilotPadManager m_AbstractedAutoPilotPadManager;

    //!< 自動入力の抽象化インタフェース
    AbstractedAutoPilotVirtualPad m_AbstractedAutoPilotVirtualPad[
        AbstractedAutoPilotVirtualPadCount];

    //!< 自動入力の抽象化 Pad を管理するマネージャー
    AbstractedAutoPilotVirtualPadManager m_AbstractedAutoPilotVirtualPadManager;

    //!< Xcd の抽象化インタフェース
    AbstractedPadXcd m_AbstractedPadXcd[AbstractedPadXcdCountMax];

    //!< Xcd の割り当てマネージャー
    AbstractedPadXcdManager m_AbstractedPadXcdManager;

    //!< GenericPad の抽象化インタフェース
    AbstractedGenericPad m_AbstractedGenericPad[AbstractedGenericPadCountMax];

    //!< AbstractedPadId 発行マネージャー
    AbstractedPadIdPublisher m_AbstractedIdPublisher;

    //!< AbstractedPad の HidShell 対応を管理するマネージャー
    AbstractedPadHidShellManager m_AbstractedPadHidShellManager;

    //!< Npad のリソースマネージャー
    NpadResourceManager m_NpadResourceManager;

    //!< UniquePad のリソースマネージャー
    UniquePadResourceManager m_UniquePadResourceManager;

    //!< Palma のリソースマネージャー
    PalmaResourceManager m_PalmaResourceManager;

    //!< オーディオコントロールマネージャ
    AudioControlManager m_AudioControlManager;

    //!< Gc コントローラーアダプターのドライバー
    GcAdapterDriver m_GcAdapterDriver;

    //!< AbstractedPad のマネージャー
    AbstractedPadManager m_AbstractedPadManager;

public:
    ResourceManager() NN_NOEXCEPT;

    ~ResourceManager() NN_NOEXCEPT;

    //!< DebugPad タスクを返します。
    DebugPadTask& GetDebugPadTask() NN_NOEXCEPT;

    //!< Gesture タスクを返します。
    SharedTask& GetGestureTask() NN_NOEXCEPT;

    //!< Keyboard タスクを返します。
    SharedTask& GetKeyboardTask() NN_NOEXCEPT;

    //!< Mouse タスクを返します。
    SharedTask& GetMouseTask() NN_NOEXCEPT;

    //!< TouchScreen タスクを返します。
    SharedTask& GetTouchScreenTask() NN_NOEXCEPT;

    //!< ホームボタンタスクを返します。
    SharedTask& GetHomeButtonTask() NN_NOEXCEPT;

    //!< スリープボタンタスクを返します。
    SharedTask& GetSleepButtonTask() NN_NOEXCEPT;

    //!< 撮影ボタンタスクを返します。
    SharedTask& GetCaptureButtonTask() NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor タスクを返します。
    SharedTask& GetConsoleSixAxisSensorTask() NN_NOEXCEPT;

    //!< Xpad タスクを返します。
    XpadTask& GetXpadTask(int playerNumber) NN_NOEXCEPT;

    //!< InputDetector タスクを返します。
    SharedTask& GetInputDetectorTask() NN_NOEXCEPT;

    //!< AbstractedPadXcd タスクを返します。
    SharedTask& GetAbstractedPadXcdTask() NN_NOEXCEPT;

    //!< AbstractedAutoPilotPad タスクを返します。
    SharedTask& GetAbstractedAutoPilotPadTask() NN_NOEXCEPT;

    //!< プレイレポートタスク返します。
    SharedTask& GetPlayReportTask() NN_NOEXCEPT;

    //!< 登録デバイスタスク返します。
    SharedTask& GetRegisteredDeviceTask() NN_NOEXCEPT;

    //!< Xcd ファームウェアアップデータタスク を返します。
    SharedTask& GetFirmwareUpdaterTask() NN_NOEXCEPT;

    //!< オーディオコントロールタスク を返します。
    SharedTask& GetAudioControlTask() NN_NOEXCEPT;

    //!< Npad のリソースマネージャーを返します。
    NpadResourceManager& GetNpadResourceManager() NN_NOEXCEPT;

    //!< UniquePad のリソースマネージャーを返します。
    UniquePadResourceManager& GetUniquePadResourceManager() NN_NOEXCEPT;

    //!< AbstractedPad のマネージャーを返します。
    AbstractedPadManager& GetAbstractedPadManager() NN_NOEXCEPT;

    //!< Palma のリソースマネージャーを返します。
    PalmaResourceManager& GetPalmaResourceManager() NN_NOEXCEPT;

    //!< ARUID を設定します。
    void SetAppletResourceUserId(
        ::nn::applet::AppletResourceUserId value) NN_NOEXCEPT;

    //!< 携帯機で利用できる Hid を有効にします。
    void EnableHandheldHids() NN_NOEXCEPT;

    //!< 携帯機で利用できる Hid を無効にします。
    void DisableHandheldHids() NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID を登録します。
    ::nn::Result RegisterAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID の登録を解除します。
    void UnregisterAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースのエントリを設定します。
    ::nn::Result SetAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースのエントリを削除します。
    void ResetAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< HID 入力の提供を行うか否かを設定します。
    void EnableAppletToGetInput(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< アプレットリソースの共有メモリのハンドルを返します。
    ::nn::Result GetAppletResourceSharedMemoryHandle(
        ::nn::os::NativeHandle* pOutValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 6 軸センサーの提供を行うか否かを設定します。
    void EnableAppletToGetSixAxisSensor(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< 共有アプレットリソースをアクティブ化します。
    ::nn::Result ActivateSharedAppletResource() NN_NOEXCEPT;

    //!< 共有アプレットリソースを非アクティブ化します。
    ::nn::Result DeactivateSharedAppletResource() NN_NOEXCEPT;

    // VibrationDeviceHandle が有効であることを表明します。
    ::nn::Result AssertValidVibrationDeviceHandle(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< VibrationDeviceHandle から VibrationDeviceInfo を取得します。
    ::nn::Result GetVibrationDeviceInfo(
        VibrationDeviceInfo* pOutValue,
        const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< IVibratorDriver を返します。
    IVibratorDriver& GetVibrator(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< VibratorXcdDriver を返します。
    VibratorXcdDriver* GetVibratorXcd(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< VibratorGcDriver を返します。
    VibratorGcDriver* GetVibratorGc(const VibrationDeviceHandle& handle) NN_NOEXCEPT;

    //!< 振動の制御権を持つアプレットリソースユーザ ID であるかを判定します。
    ::nn::Result IsAruidValidForVibration(
        ::nn::applet::AppletResourceUserId aruid, bool* pOutValue) NN_NOEXCEPT;

    //!< 振動の制御権を持つアプレットリソースユーザ ID を設定します。
    ::nn::Result SetAruidValidForVibration(
        ::nn::applet::AppletResourceUserId aruid, bool enable) NN_NOEXCEPT;

    //!< 一時的な振動強制ON期間を開始します。
    ::nn::Result BeginPermitVibrationSession(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 一時的な振動強制ON期間を停止します。
    ::nn::Result EndPermitVibrationSession() NN_NOEXCEPT;

    //!< 全ての振動子を一旦リセットします。
    ::nn::Result ResetAllVibrationDevices() NN_NOEXCEPT;

    //!< SixAxisSensorProcessor を返します。
    const SixAxisSensorProcessor& GetSixAxisSensorProcessor(
        const SixAxisSensorHandle& handle) const NN_NOEXCEPT;

    //!< SixAxisSensorAppletSettingManager を返します。
    SixAxisSensorAppletSettingManager& GetSixAxisSensorAppletSettingManager() NN_NOEXCEPT;

    //!< ConsoleSixAxisSensorAppletSettingManager を返します。
    ConsoleSixAxisSensorAppletSettingManager& GetConsoleSixAxisSensorAppletSettingManager(
        ) NN_NOEXCEPT;

    //!< XcdDriver を返します。
    XcdDriver& GetXcdDriver() NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad のプレイヤー番号が有効であることを表明します。
    ::nn::Result AssertValidBasicXpadPlayerNumber(int playerNumber) NN_NOEXCEPT;

    //!< JoyXpad のプレイヤー番号が有効であることを表明します。
    ::nn::Result AssertValidJoyXpadPlayerNumber(int playerNumber) NN_NOEXCEPT;

    //!< Xpad ID の再解決イベントのハンドルを確保します。
    ::nn::Result AcquireXpadIdEventHandle(
        ::nn::os::NativeHandle* outHandle, uint64_t clientId) NN_NOEXCEPT;

    //!< Xpad ID の再解決イベントのハンドルを開放します。
    ::nn::Result ReleaseXpadIdEventHandle(uint64_t clientId) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad の解決された ID を取得します。
    ::nn::Result GetXpadIds(
        int* outCount, BasicXpadId* outXpadIds, int count) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad の解決された ID を取得します。
    ::nn::Result GetXpadIds(
        int* outCount, JoyXpadId* outXpadIds, int count) NN_NOEXCEPT;

    //!< Bluetooth オフ中のジョイコン取り外し通知のハンドルを確保します
    ::nn::Result AcquireJoyDetachOnBluetoothOffEventHandle(
        ::nn::os::NativeHandle* pOutValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Pad 共通のタイマーを初期化します
    ::nn::Result ActivatePadCommonTimer() NN_NOEXCEPT;

    //!< Pad 共通のタイマーを初期化します
    ::nn::Result DeactivatePadCommonTimer() NN_NOEXCEPT;

    //!< DebugPad 用のマッピングマネージャを返します。
    DebugPadMappingManager& GetDebugPadMappingManager() NN_NOEXCEPT;

    //!< Xpad 用のマッピングマネージャを返します。
    XpadMappingManager& GetXpadMappingManager(int playerNumber) NN_NOEXCEPT;

    //!< GenericPad のアクティブ化する
    ::nn::Result ActivateAbstractedGenericPad() NN_NOEXCEPT;

    //!< GenericPad の非アクティブ化する
    ::nn::Result DeactivateAbstractedGenericPad() NN_NOEXCEPT;

private:
    //!< 共有タスクを初期化します。
    void InitializeSharedTask() NN_NOEXCEPT;

    //!< DebugPad タスクを初期化します。
    void InitializeDebugPadTask() NN_NOEXCEPT;

    //!< AbstractedPad 関連の初期化を行います。
    void InitializeAbstractedPad() NN_NOEXCEPT;

    //!< Xpad 関連の初期化を行います。
    void InitializeXpad() NN_NOEXCEPT;

    //!< Npad 関連の初期化を行います。
    void InitializeNpad() NN_NOEXCEPT;

    //!< UniquePad 関連の初期化を行います。
    void InitializeUniquePad() NN_NOEXCEPT;

    //!< WindowsGenericPadBroker タスクを初期化します。
    void InitializeWindowsGenericPadBrokerTask() NN_NOEXCEPT;

    //!< AbstractedPadManager の初期化を行います。
    void InitializeAbstractedPadManager() NN_NOEXCEPT;
};

//!< リソースマネージャを返します。
ResourceManager& GetResourceManager() NN_NOEXCEPT;

}}} // namespace nn::hid::detail
