﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_SevenSixAxisSensor.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/debug/hid_CaptureButton.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/hid/debug/hid_HomeButton.h>
#include <nn/hid/debug/hid_Keyboard.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/hid/debug/hid_SleepButton.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/hid/system/hid_Keyboard.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_AhidManager-os.horizon.h"
#include "hid_CaptureButtonManager.h"
#include "hid_IConsoleSixAxisSensorDriver.h"
#include "hid_IDebugPadDriver.h"
#include "hid_IKeyboardDriver.h"
#include "hid_IMouseDriver.h"
#include "hid_InputDetectorId.h"
#include "hid_ITouchScreenDriver.h"
#include "hid_IXpadDriver.h"
#include "hid_ResourceManager-os.horizon.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_StaticObject.h"
#include "hid_UniquePadId.h"
#include "hid_XpadId.h"

namespace nn { namespace hid { namespace detail {

const nn::TimeSpan SharedTask::PadCommonTimerInterval = nn::TimeSpan::FromMilliSeconds(5);

SharedTask::SharedTask() NN_NOEXCEPT
{
    // 何もしない
}

SharedTask::~SharedTask() NN_NOEXCEPT
{
    // 何もしない
}

void SharedTask::SetNpadResourceManager(NpadResourceManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    m_pNpadResourceManager = pManager;
}

void SharedTask::SetUniquePadResourceManager(UniquePadResourceManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    pManager->SetSystemEvents(m_BluetoothFirmwareUpdateSystemEvent.GetBase(),
                              m_McuFirmwareUpdateSystemEvent.GetBase(),
                              m_FirmwareUpdateNewSampleSystemEvent.GetBase(),
                              m_FirmwareUpdateResetTimeoutEvent.GetBase());
    m_pUniquePadResourceManager = pManager;
}

void SharedTask::SetPalmaResourceManager(PalmaResourceManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pPalmaResourceManager = pManager;
}

void SharedTask::SetDebugPadManager(DebugPadManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_DebugPadTimerEvent);
    m_pDebugPadManager = pManager;
}

void SharedTask::SetKeyboardManager(KeyboardManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_KeyboardTimerEvent);
    m_pKeyboardManager = pManager;
}

void SharedTask::SetMouseManager(MouseManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_MouseTimerEvent);
    m_pMouseManager = pManager;
}

void SharedTask::SetXpadManagers(XpadManager* managers, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(managers);
    NN_SDK_REQUIRES_GREATER(count, 0);
    managers[0].SetTimer(&m_XpadTimerEvent);
    m_XpadManagers = managers;
    m_XpadManagerCount = count;
}

void SharedTask::SetInputDetectorManager(InputDetectorManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    m_pInputDetectorManager = pManager;
}

void SharedTask::SetHomeButtonManager(HomeButtonManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_HomeButtonTimerEvent);
    m_pHomeButtonManager = pManager;
}

void SharedTask::SetSleepButtonManager(SleepButtonManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_SleepButtonTimerEvent);
    m_pSleepButtonManager = pManager;
}

void SharedTask::SetCaptureButtonManager(
    CaptureButtonManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_CaptureButtonTimerEvent);
    m_pCaptureButtonManager = pManager;
}

void SharedTask::SetPlayReportManager(PlayReportManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    pManager->SetTimer(&m_PlayReportTimerEvent);
    m_pPlayReportManager = pManager;
}

void SharedTask::SetRegisteredDeviceManager(RegisteredDeviceManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    pManager->BindXcdRegistrationEvent(m_PairingCompleteEvent.GetBase());
    pManager->BindXcdConnectionTriggerTimeoutEvent(m_ConnectionTriggerTimeoutEvent.GetBase());
    m_pRegisteredDeviceManager = pManager;
}

void SharedTask::SetAbstractedPadXcd(AbstractedPadXcd* pXcd, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pXcd);
    NN_SDK_REQUIRES_GREATER(count, 0);

    m_pAbstractedPadXcd = pXcd;
    m_AbstractedPadXcdCount = count;
}

void SharedTask::SetAbstractedPadXcdManager(AbstractedPadXcdManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pAbstractedPadXcdManager = pManager;
}

void SharedTask::SetAbstractedPalmaPad(AbstractedPalmaPad* pPalma, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPalma);
    NN_SDK_REQUIRES_GREATER(count, 0);

    m_pAbstractedPalmaPad = pPalma;
    m_AbstractedPalmaPadCount = count;
}

void SharedTask::SetAbstractedPalmaPadManager(AbstractedPalmaPadManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pAbstractedPalmaPadManager = pManager;
}

void SharedTask::SetAbstractedAutoPilotPadManager(
    AbstractedAutoPilotPadManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pAbstractedAutoPilotPadManager = pManager;
}

void SharedTask::SetAbstractedAutoPilotVirtualPadManager(
    AbstractedAutoPilotVirtualPadManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pAbstractedAutoPilotVirtualPadManager = pManager;
}

void SharedTask::SetAbstractedGcAdapterPadManager(
    AbstractedGcAdapterPadManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pAbstractedGcAdapterPadManager = pManager;
}

void SharedTask::SetAbstractedFiftyPad(AbstractedFiftyPad* pFifty) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFifty);

    m_pAbstractedFiftyPad = pFifty;
}

void SharedTask::SetRailManager(RailManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pRailManager = pManager;
    pManager->BindRailUpdateEvent(m_RailUpdateEvent.GetBase());
}

void SharedTask::SetAudioControlManager(AudioControlManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimer(&m_AudioControlTimerEvent);
    m_pAudioControlManager = pManager;
}

::nn::Result SharedTask::ActivateDebugPad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_DebugPadMutex)> locker(m_DebugPadMutex);
    NN_RESULT_DO(m_pDebugPadManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateDebugPad() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);

    {
        ::std::lock_guard<decltype(m_DebugPadMutex)> locker(m_DebugPadMutex);
        NN_RESULT_DO(m_pDebugPadManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureDebugPadAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);
    ::std::lock_guard<decltype(m_DebugPadMutex)> locker(m_DebugPadMutex);
    NN_RESULT_DO(m_pDebugPadManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetDebugPadAutoPilotState(
    const ::nn::hid::debug::DebugPadAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);
    ::std::lock_guard<decltype(m_DebugPadMutex)> locker(m_DebugPadMutex);
    NN_RESULT_THROW(m_pDebugPadManager->SetAutoPilotState(value));
}

::nn::Result SharedTask::UnsetDebugPadAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);
    ::std::lock_guard<decltype(m_DebugPadMutex)> locker(m_DebugPadMutex);
    NN_RESULT_THROW(m_pDebugPadManager->UnsetAutoPilotState());
}

::nn::Result SharedTask::ActivateKeyboard() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);
    NN_RESULT_DO(m_pKeyboardManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateKeyboard() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);

    {
        ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);
        NN_RESULT_DO(m_pKeyboardManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureKeyboardAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);
    ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);
    NN_RESULT_DO(m_pKeyboardManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetKeyboardAutoPilotState(
    const ::nn::hid::debug::KeyboardAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);
    ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);
    NN_RESULT_THROW(m_pKeyboardManager->SetAutoPilotState(value));
}

::nn::Result SharedTask::UnsetKeyboardAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);
    ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);
    NN_RESULT_THROW(m_pKeyboardManager->UnsetAutoPilotState());
}

::nn::Result SharedTask::SendKeyboardLockKeyEvent(
    ::nn::hid::system::KeyboardLockKeyEventSet value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);
    ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);
    NN_RESULT_DO(m_pKeyboardManager->SendLockKeyEvent(value));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateMouse() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_MouseMutex)> locker(m_MouseMutex);
    NN_RESULT_DO(m_pMouseManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateMouse() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);

    {
        ::std::lock_guard<decltype(m_MouseMutex)> locker(m_MouseMutex);
        NN_RESULT_DO(m_pMouseManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateAudioControl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_AudioControlMutex)> locker(m_AudioControlMutex);
    NN_RESULT_DO(m_pAudioControlManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateAudioControl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlManager);

    {
        ::std::lock_guard<decltype(m_AudioControlMutex)> locker(m_AudioControlMutex);
        NN_RESULT_DO(m_pAudioControlManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureMouseAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);
    ::std::lock_guard<decltype(m_MouseMutex)> locker(m_MouseMutex);
    NN_RESULT_DO(m_pMouseManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetMouseAutoPilotState(
    const ::nn::hid::debug::MouseAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);
    ::std::lock_guard<decltype(m_MouseMutex)> locker(m_MouseMutex);
    NN_RESULT_THROW(m_pMouseManager->SetAutoPilotState(value));
}

::nn::Result SharedTask::UnsetMouseAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);
    ::std::lock_guard<decltype(m_MouseMutex)> locker(m_MouseMutex);
    NN_RESULT_THROW(m_pMouseManager->UnsetAutoPilotState());
}

::nn::Result SharedTask::ActivateAbstractedPadXcdManager() NN_NOEXCEPT
{
    auto needsRollback = true;

    // デフォルトはシグナル状態
    nn::os::SignalSystemEvent(m_XcdDeviceUpdateEvent.GetBase());
    // XCD にSystemEvent を登録
    nn::xcd::BindLinkUpdateEvent(m_XcdDeviceUpdateEvent.GetBase());

    NN_RESULT_DO(m_pAbstractedPadXcdManager->Activate());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_pAbstractedPadXcdManager->Deactivate();
        }
    };

    NN_RESULT_DO(m_pAbstractedGcAdapterPadManager->Activate());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_pAbstractedGcAdapterPadManager->Deactivate();
        }
    };

    NN_RESULT_DO(m_pAbstractedPalmaPadManager->Activate());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_pAbstractedPalmaPadManager->Deactivate();
        }
    };

    NN_RESULT_DO(m_pAbstractedPalmaPadManager->Activate());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            m_pAbstractedPalmaPadManager->Deactivate();
        }
    };

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateAbstractedPadXcdManager() NN_NOEXCEPT
{
    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_DO(this->m_pAbstractedPadXcdManager->Deactivate());

    NN_RESULT_DO(this->m_pAbstractedGcAdapterPadManager->Deactivate());

    NN_RESULT_DO(this->m_pAbstractedPalmaPadManager->Deactivate());

    // XCD に登録されているSystemEvent をアンバインド
    nn::xcd::UnbindLinkUpdateEvent(m_XcdDeviceUpdateEvent.GetBase());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateXpad() NN_NOEXCEPT
{
    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);

    int i = 0;

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            while (i > 0)
            {
                --i;
                m_XpadManagers[i].Deactivate();
            }
        }
    };

    while (i < m_XpadManagerCount)
    {
        NN_RESULT_DO(m_XpadManagers[i].Activate());
        ++i;
    }

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateXpad() NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
        for (int i = m_XpadManagerCount - 1; i >= 0; --i)
        {
            NN_RESULT_DO(m_XpadManagers[i].Deactivate());
        }
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

void SharedTask::EnableXpadSampling() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);

    for (int i = 0; i < m_XpadManagerCount; ++i)
    {
        m_XpadManagers[i].EnableSampling();
    }
}

::nn::Result SharedTask::EnsureBasicXpadAppletResource(
    ::nn::applet::AppletResourceUserId aruid,
    int playerNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(playerNumber, 0, m_XpadManagerCount);
    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_DO(
        m_XpadManagers[playerNumber].EnsureBasicXpadAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::GetJoyXpadLifoHandle(
    ::nn::os::NativeHandle* pOutValue, int playerNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_RANGE(playerNumber, 0, m_XpadManagerCount);

    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_THROW(
        m_XpadManagers[playerNumber].GetJoyXpadLifoHandle(pOutValue));
}

::nn::Result SharedTask::ActivatePadCommonTimer() NN_NOEXCEPT
{
    auto needsRollback = true;

    NN_RESULT_DO(this->ActivatePadCommonTimerEvent());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivatePadCommonTimerEvent();
        }
    };

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivatePadCommonTimer() NN_NOEXCEPT
{
    NN_RESULT_DO(this->DeactivatePadCommonTimerEvent());

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

void SharedTask::BindPairingEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)> locker(m_RegisteredDeviceMutex);
    m_pRegisteredDeviceManager->BindXcdRegistrationEvent(pSystemEvent);
}

void SharedTask::UnbindPairingEvent(::nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)> locker(m_RegisteredDeviceMutex);
    m_pRegisteredDeviceManager->UnbindXcdRegistrationEvent(pSystemEvent);
}

::nn::Result SharedTask::ActivateSixAxisSensor() NN_NOEXCEPT
{
    int i = 0;
    while (i < m_XpadManagerCount)
    {
        NN_RESULT_DO(m_XpadManagers[i].ActivateSixAxisSensor());
        ++i;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateSixAxisSensor() NN_NOEXCEPT
{
    int i = 0;
    while (i < m_XpadManagerCount)
    {
        NN_RESULT_DO(m_XpadManagers[i].DeactivateSixAxisSensor());
        ++i;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::GetSixAxisSensorLifoHandle(
    ::nn::os::NativeHandle* pOutValue, int playerNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_RANGE(playerNumber, 0, m_XpadManagerCount);

    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_THROW(
        m_XpadManagers[playerNumber].GetSixAxisSensorLifoHandle(pOutValue));
}

::nn::Result SharedTask::ActivateJoySixAxisSensor() NN_NOEXCEPT
{
    int i = 0;
    while (i < m_XpadManagerCount)
    {
        NN_RESULT_DO(m_XpadManagers[i].ActivateJoySixAxisSensor());
        ++i;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateJoySixAxisSensor() NN_NOEXCEPT
{
    int i = 0;
    while (i < m_XpadManagerCount)
    {
        NN_RESULT_DO(m_XpadManagers[i].DeactivateJoySixAxisSensor());
        ++i;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::GetJoySixAxisSensorLifoHandle(
    ::nn::os::NativeHandle* pOutValue, int playerNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_RANGE(playerNumber, 0, m_XpadManagerCount);

    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_THROW(
        m_XpadManagers[playerNumber].GetJoySixAxisSensorLifoHandle(pOutValue));
}

::nn::Result SharedTask::SetBasicXpadAutoPilotState(
    int playerNumber,
    const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(playerNumber, 0, m_XpadManagerCount);
    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_THROW(
        m_XpadManagers[playerNumber].SetBasicXpadAutoPilotState(value));
}

::nn::Result SharedTask::UnsetBasicXpadAutoPilotState(
    int playerNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(playerNumber, 0, m_XpadManagerCount);
    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);
    NN_RESULT_THROW(
        m_XpadManagers[playerNumber].UnsetBasicXpadAutoPilotState());
}

::nn::Result SharedTask::SetAbstractedPadAutoPilotStateWithBasicXpad(
    int playerNumber,
    const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(playerNumber, 0, AbstractedAutoPilotPadCount);
    NN_RESULT_THROW(
        m_pAbstractedAutoPilotPadManager->SetAutoPilotState(playerNumber, value));
}

::nn::Result SharedTask::UnsetAbstractedPadAutoPilotState(
    int playerNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(playerNumber, 0, AbstractedAutoPilotPadCount);
    NN_RESULT_THROW(
        m_pAbstractedAutoPilotPadManager->UnsetAutoPilotState(playerNumber));
}

::nn::Result SharedTask::SetAbstractedPadAutoPilotVirtualPadState(
    const ::nn::hid::debug::AbstractedVirtualPadId& virtualPadId,
    const ::nn::hid::debug::AbstractedPadState& value) NN_NOEXCEPT
{
    NN_RESULT_THROW(
        m_pAbstractedAutoPilotVirtualPadManager->SetAutoPilotState(virtualPadId, value));
}

::nn::Result SharedTask::UnsetAbstractedPadAutoPilotVirtualPadState(const ::nn::hid::debug::AbstractedVirtualPadId& virtualPadId) NN_NOEXCEPT
{
    NN_RESULT_THROW(
        m_pAbstractedAutoPilotVirtualPadManager->UnsetAutoPilotState(virtualPadId));
}

::nn::Result SharedTask::UnsetAllAbstractedPadAutoPilotVirtualPadState() NN_NOEXCEPT
{
    NN_RESULT_THROW(
        m_pAbstractedAutoPilotVirtualPadManager->UnsetAllAutoPilotState());
}

::nn::Result SharedTask::ActivateInputDetector() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);

    ::std::lock_guard<decltype(m_InputDetectorMutex)> locker(m_InputDetectorMutex);
    NN_RESULT_DO(m_pInputDetectorManager->Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateInputDetector() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);

    {
        ::std::lock_guard<decltype(m_InputDetectorMutex)
        > locker(m_InputDetectorMutex);
        NN_RESULT_DO(m_pInputDetectorManager->Deactivate());
    }
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureInputDetectorAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);
    ::std::lock_guard<decltype(m_InputDetectorMutex)> locker(m_InputDetectorMutex);
    NN_RESULT_DO(m_pInputDetectorManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::NotifyInputDetector(const ::nn::hid::system::InputSourceIdSet& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);
    ::std::lock_guard<decltype(m_InputDetectorMutex)> locker(m_InputDetectorMutex);
    NN_RESULT_THROW(m_pInputDetectorManager->Notify(id));
}

::nn::Result SharedTask::AcquireHomeButtonEventHandle(
    ::nn::os::NativeHandle* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultHomeButtonEventNullHandleBuffer());
    ::std::lock_guard<decltype(m_HomeButtonMutex)> locker(m_HomeButtonMutex);
    NN_RESULT_DO(
        m_pHomeButtonManager->AcquireHomeButtonEventHandle(pOutValue, aruid));
    // ハンドルの取得直後はシグナル状態であることを保証
    m_pHomeButtonManager->SignalHomeButtonEvent();
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateHomeButton() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_HomeButtonMutex)> locker(m_HomeButtonMutex);
    NN_RESULT_DO(m_pHomeButtonManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateHomeButton() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);

    {
        ::std::lock_guard<decltype(m_HomeButtonMutex)
                          > locker(m_HomeButtonMutex);
        NN_RESULT_DO(m_pHomeButtonManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureHomeButtonAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);
    ::std::lock_guard<decltype(m_HomeButtonMutex)> locker(m_HomeButtonMutex);
    NN_RESULT_DO(m_pHomeButtonManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetHomeButtonAutoPilotState(
    const ::nn::hid::debug::HomeButtonAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);
    ::std::lock_guard<decltype(m_HomeButtonMutex)> locker(m_HomeButtonMutex);
    NN_RESULT_THROW(m_pHomeButtonManager->SetAutoPilotState(value));
}

::nn::Result SharedTask::UnsetHomeButtonAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);
    ::std::lock_guard<decltype(m_HomeButtonMutex)> locker(m_HomeButtonMutex);
    NN_RESULT_THROW(m_pHomeButtonManager->UnsetAutoPilotState());
}

::nn::Result SharedTask::AcquireSleepButtonEventHandle(
    ::nn::os::NativeHandle* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultSleepButtonEventNullHandleBuffer());
    ::std::lock_guard<decltype(m_SleepButtonMutex)> locker(m_SleepButtonMutex);
    NN_RESULT_DO(
        m_pSleepButtonManager->AcquireSleepButtonEventHandle(pOutValue, aruid));
    // ハンドルの取得直後はシグナル状態であることを保証
    m_pSleepButtonManager->SignalSleepButtonEvent();
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateSleepButton() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_SleepButtonMutex)> locker(m_SleepButtonMutex);
    NN_RESULT_DO(m_pSleepButtonManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateSleepButton() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);

    {
        ::std::lock_guard<decltype(m_SleepButtonMutex)
                          > locker(m_SleepButtonMutex);
        NN_RESULT_DO(m_pSleepButtonManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureSleepButtonAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);
    ::std::lock_guard<decltype(m_SleepButtonMutex)> locker(m_SleepButtonMutex);
    NN_RESULT_DO(m_pSleepButtonManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetSleepButtonAutoPilotState(
    const ::nn::hid::debug::SleepButtonAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);
    ::std::lock_guard<decltype(m_SleepButtonMutex)> locker(m_SleepButtonMutex);
    NN_RESULT_THROW(m_pSleepButtonManager->SetAutoPilotState(value));
}

::nn::Result SharedTask::UnsetSleepButtonAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);
    ::std::lock_guard<decltype(m_SleepButtonMutex)> locker(m_SleepButtonMutex);
    NN_RESULT_THROW(m_pSleepButtonManager->UnsetAutoPilotState());
}

::nn::Result SharedTask::AcquireCaptureButtonEventHandle(
    ::nn::os::NativeHandle* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultCaptureButtonEventNullHandleBuffer());
    ::std::lock_guard<decltype(m_CaptureButtonMutex)
                      > locker(m_CaptureButtonMutex);
    NN_RESULT_DO(
        m_pCaptureButtonManager->AcquireCaptureButtonEventHandle(
            pOutValue, aruid));
    // ハンドルの取得直後はシグナル状態であることを保証
    m_pCaptureButtonManager->SignalCaptureButtonEvent();
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ActivateCaptureButton() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);

    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    ::std::lock_guard<decltype(m_CaptureButtonMutex)
                      > locker(m_CaptureButtonMutex);
    NN_RESULT_DO(m_pCaptureButtonManager->Activate());

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivateCaptureButton() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);

    {
        ::std::lock_guard<decltype(m_CaptureButtonMutex)
                          > locker(m_CaptureButtonMutex);
        NN_RESULT_DO(m_pCaptureButtonManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnsureCaptureButtonAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);
    ::std::lock_guard<decltype(m_CaptureButtonMutex)
                      > locker(m_CaptureButtonMutex);
    NN_RESULT_DO(m_pCaptureButtonManager->EnsureAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::SetCaptureButtonAutoPilotState(
    const ::nn::hid::debug::CaptureButtonAutoPilotState& value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);
    ::std::lock_guard<decltype(m_CaptureButtonMutex)
                      > locker(m_CaptureButtonMutex);
    NN_RESULT_DO(m_pCaptureButtonManager->SetAutoPilotState(value));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::UnsetCaptureButtonAutoPilotState() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);
    ::std::lock_guard<decltype(m_CaptureButtonMutex)
                      > locker(m_CaptureButtonMutex);
    NN_RESULT_DO(m_pCaptureButtonManager->UnsetAutoPilotState());
    NN_RESULT_SUCCESS;
}

void SharedTask::ActivatePlayReport() NN_NOEXCEPT
{
    m_pPlayReportManager->Activate();
}

void SharedTask::DeactivatePlayReport() NN_NOEXCEPT
{
    m_pPlayReportManager->Deactivate();
}

void SharedTask::ActivateRegisteredDeviceManagement() NN_NOEXCEPT
{
    m_pRegisteredDeviceManager->Activate();
}

void SharedTask::DeactivateRegisteredDeviceManagement() NN_NOEXCEPT
{
    m_pRegisteredDeviceManager->Deactivate();
}

Result SharedTask::AcquirePlayReportControllerUsageUpdateEventHandle(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NN_RESULT_DO(m_pPlayReportManager->AcquireUsageUpdateEventHandle(pOutValue));
    NN_RESULT_SUCCESS;
}

int SharedTask::GetPlayReportControllerUsages(
    system::PlayReportControllerUsage* pOutValues, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    return m_pPlayReportManager->GetControllerUsages(pOutValues, count);
}

Result SharedTask::AcquirePlayReportRegisteredDevicesEventHandle(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NN_RESULT_DO(m_pRegisteredDeviceManager->AcquireUpdateForPlayReportEventHandle(pOutValue));
    NN_RESULT_SUCCESS;
}

int SharedTask::GetRegisteredDevices(system::RegisteredDeviceOld* pOutValues, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                      > locker(m_RegisteredDeviceMutex);
    return m_pRegisteredDeviceManager->GetRegisteredDevices(pOutValues, count);
}

int SharedTask::GetRegisteredDevices(system::RegisteredDevice* pOutValues, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                      > locker(m_RegisteredDeviceMutex);
    return m_pRegisteredDeviceManager->GetRegisteredDevices(pOutValues, count);
}

void SharedTask::AddRegisteredDevices(const system::RegisteredDevice& device) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                      > locker(m_RegisteredDeviceMutex);
    m_pRegisteredDeviceManager->AddRegisteredDevice(device);
}

Result SharedTask::AcquireConnectionTriggerTimeoutEvent(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                      > locker(m_RegisteredDeviceMutex);
    NN_RESULT_DO(m_pRegisteredDeviceManager->AcquireConnectionTriggerTimeoutEvent(pOutValue));
    NN_RESULT_SUCCESS;
}

Result SharedTask::SendConnectionTrigger(::nn::bluetooth::Address& address) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                      > locker(m_RegisteredDeviceMutex);
    NN_RESULT_DO(m_pRegisteredDeviceManager->SendConnectionTrigger(address));
    NN_RESULT_SUCCESS;
}

Result SharedTask::AcquireDeviceRegisteredEventForControllerSupport(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                      > locker(m_RegisteredDeviceMutex);
    NN_RESULT_DO(
        m_pRegisteredDeviceManager->AcquireDeviceRegisteredEventForControllerSupport(pOutValue));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::AcquireAudioControlEventHandle(
    ::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlManager);
    NN_RESULT_THROW_UNLESS(
        pOutValue != nullptr, ResultAudioControlEventNullHandleBuffer());
    ::std::lock_guard<decltype(m_AudioControlMutex)> locker(m_AudioControlMutex);
    NN_RESULT_DO(
        m_pAudioControlManager->AcquireAudioControlEventHandle(pOutValue));
    // ハンドルの取得直後はシグナル状態であることを保証
    m_pAudioControlManager->SignalAudioControlEvent();
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::GetAudioControlStates(
    int* pOutCount, system::AudioControlState outStates[], int count
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlManager);
    NN_RESULT_THROW_UNLESS(
        outStates != nullptr, ResultAudioControlEventNullHandleBuffer());
    ::std::lock_guard<decltype(m_AudioControlMutex)> locker(m_AudioControlMutex);
    NN_RESULT_DO(
        m_pAudioControlManager->GetStates(pOutCount, outStates, count));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::IsUsbFullKeyControllerEnabled(bool* pOutEnabled) NN_NOEXCEPT
{
    NN_RESULT_DO(m_pAbstractedPadXcdManager->IsUsbFullKeyControllerEnabled(pOutEnabled));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::EnableUsbFullKeyController(bool enabled) NN_NOEXCEPT
{
    NN_RESULT_DO(m_pAbstractedPadXcdManager->EnableUsbFullKeyController(enabled));
    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::ResetDevices() NN_NOEXCEPT
{
    m_pAbstractedPadXcdManager->ResetAllDevices();
    NN_RESULT_SUCCESS;
}

void SharedTask::LinkTimerEvent(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    m_XcdDeviceUpdateEvent.Link(pMultiWait);
    m_PairingCompleteEvent.Link(pMultiWait);
    m_ConnectionTriggerTimeoutEvent.Link(pMultiWait);
    m_RailUpdateEvent.Link(pMultiWait);
    m_BluetoothFirmwareUpdateSystemEvent.Link(pMultiWait);
    m_McuFirmwareUpdateSystemEvent.Link(pMultiWait);
    m_FirmwareUpdateNewSampleSystemEvent.Link(pMultiWait);
    m_FirmwareUpdateResetTimeoutEvent.Link(pMultiWait);
}

void SharedTask::UnlinkTimerEvent() NN_NOEXCEPT
{
    m_XcdDeviceUpdateEvent.Unlink();
    m_PairingCompleteEvent.Unlink();
    m_ConnectionTriggerTimeoutEvent.Unlink();
    m_RailUpdateEvent.Unlink();
    m_BluetoothFirmwareUpdateSystemEvent.Unlink();
    m_McuFirmwareUpdateSystemEvent.Unlink();
    m_FirmwareUpdateNewSampleSystemEvent.Unlink();
    m_FirmwareUpdateResetTimeoutEvent.Unlink();
}

void SharedTask::LinkTimers(SynchronizedTimerGroup* group) NN_NOEXCEPT
{
    m_DebugPadTimerEvent.Link(group);
    m_KeyboardTimerEvent.Link(group);
    m_MouseTimerEvent.Link(group);
    m_XpadTimerEvent.Link(group);
    m_PadCommonTimerEvent.Link(group);
    m_HomeButtonTimerEvent.Link(group);
    m_SleepButtonTimerEvent.Link(group);
    m_CaptureButtonTimerEvent.Link(group);
    m_PlayReportTimerEvent.Link(group);
    m_AudioControlTimerEvent.Link(group);
}

void SharedTask::UnlinkTimers() NN_NOEXCEPT
{
    m_DebugPadTimerEvent.Unlink();
    m_KeyboardTimerEvent.Unlink();
    m_MouseTimerEvent.Unlink();
    m_XpadTimerEvent.Unlink();
    m_PadCommonTimerEvent.Unlink();
    m_HomeButtonTimerEvent.Unlink();
    m_SleepButtonTimerEvent.Unlink();
    m_CaptureButtonTimerEvent.Unlink();
    m_PlayReportTimerEvent.Unlink();
    m_AudioControlTimerEvent.Unlink();
}

void SharedTask::DoSampling(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);
    NN_SDK_REQUIRES_NOT_NULL(m_XpadManagers);
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pPlayReportManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pRegisteredDeviceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pRailManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlManager);


    if (waitId == m_XcdDeviceUpdateEvent.GetWaitId())
    {
        m_XcdDeviceUpdateEvent.Clear();
        m_pAbstractedPadXcdManager->UpdateDevices();
        m_pUniquePadResourceManager->HandleDeviceUpdate();
        return;
    }

    if (waitId == m_BluetoothFirmwareUpdateSystemEvent.GetWaitId())
    {
        m_BluetoothFirmwareUpdateSystemEvent.Clear();
        m_pUniquePadResourceManager->HandleBluetoothFirmwareUpdate();
        return;
    }

    if (waitId == m_McuFirmwareUpdateSystemEvent.GetWaitId())
    {
        m_McuFirmwareUpdateSystemEvent.Clear();
        m_pUniquePadResourceManager->HandleMcuFirmwareUpdate();
        return;
    }

    if (waitId == m_FirmwareUpdateNewSampleSystemEvent.GetWaitId())
    {
        m_FirmwareUpdateNewSampleSystemEvent.Clear();
        m_pUniquePadResourceManager->HandleNewSample();
        return;
    }

    if (waitId == m_FirmwareUpdateResetTimeoutEvent.GetWaitId())
    {
        m_FirmwareUpdateResetTimeoutEvent.Clear();
        m_pUniquePadResourceManager->HandleResetTimedOut();
        return;
    }

    if (waitId == m_PairingCompleteEvent.GetWaitId())
    {
        m_PairingCompleteEvent.Clear();
        ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                          > locker(m_RegisteredDeviceMutex);
        m_pRegisteredDeviceManager->UpdateRegisteredDevice();
        return;
    }

    if (waitId == m_ConnectionTriggerTimeoutEvent.GetWaitId())
    {
        m_ConnectionTriggerTimeoutEvent.Clear();
        ::std::lock_guard<decltype(m_RegisteredDeviceMutex)
                          > locker(m_RegisteredDeviceMutex);
        m_pRegisteredDeviceManager->SignalConnectionTriggerTimeout();
        m_pUniquePadResourceManager->HandleConnectionTriggerTimeout();
        return;
    }

    if (waitId == m_RailUpdateEvent.GetWaitId())
    {
        m_RailUpdateEvent.Clear();
        ::std::lock_guard<decltype(m_RailMutex)
                          > locker(m_RailMutex);
        m_pRailManager->HandleRailUpdateEvent();
        return;
    }
}

void SharedTask::DoSamplingForTimer(const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    if (this->ResolvePadCommonTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolveDebugPadTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolveKeyboardTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolveMouseTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolveXpadTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolveHomeButtonTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolveSleepButtonTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolveCaptureButtonTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolvePlayReportTimerEvent(waitId))
    {
        return;
    }

    if (this->ResolveAudioControlTimerEvent(waitId))
    {
        return;
    }
}

void SharedTask::IsNpadUsed(bool isUsed) NN_NOEXCEPT
{
    // Npad を常時有効にする
    NN_UNUSED(isUsed);
}

::nn::Result SharedTask::ActivatePadCommonTimerEvent() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_PadCommonTimerEventActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_PadCommonTimerEventActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
        m_PadCommonTimerEvent.Enable(PadCommonTimerInterval);
    }

    ++m_PadCommonTimerEventActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result SharedTask::DeactivatePadCommonTimerEvent() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_PadCommonTimerEventActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    --m_PadCommonTimerEventActivationCount;

    if (m_PadCommonTimerEventActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
        m_PadCommonTimerEvent.Disable();
    }

    NN_RESULT_SUCCESS;
}

bool SharedTask::ResolvePadCommonTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    if (waitId != &m_PadCommonTimerEvent)
    {
        return false;
    }

    m_PadCommonTimerEvent.Clear();

    m_pAbstractedPadXcdManager->HandlePeriodicalEvent();
    for (int i = 0; i < m_AbstractedPadXcdCount; ++i)
    {
        m_pAbstractedPadXcd[i].Update();
    }
    m_pAbstractedPalmaPadManager->UpdateDevices();
    for (int i = 0; i < m_AbstractedPalmaPadCount; ++i)
    {
        m_pAbstractedPalmaPad[i].Update();
    }
    m_pAbstractedGcAdapterPadManager->Update();
    m_pAbstractedFiftyPad->Update();
    m_pNpadResourceManager->HandlePeriodcialEvent();
    m_pUniquePadResourceManager->HandlePeriodicalEvent();

    return true;
}

bool SharedTask::ResolveDebugPadTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDebugPadManager);

    if (waitId != &m_DebugPadTimerEvent)
    {
        return false;
    }

    m_DebugPadTimerEvent.Clear();

    ::std::lock_guard<decltype(m_DebugPadMutex)> locker(m_DebugPadMutex);

    m_pDebugPadManager->Sample();

    return true;
}

bool SharedTask::ResolveKeyboardTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pKeyboardManager);

    if (waitId != &m_KeyboardTimerEvent)
    {
        return false;
    }

    m_KeyboardTimerEvent.Clear();

    ::std::lock_guard<decltype(m_KeyboardMutex)> locker(m_KeyboardMutex);

    m_pKeyboardManager->Sample();

    return true;
}

bool SharedTask::ResolveMouseTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pMouseManager);

    if (waitId != &m_MouseTimerEvent)
    {
        return false;
    }

    m_MouseTimerEvent.Clear();

    ::std::lock_guard<decltype(m_MouseMutex)> locker(m_MouseMutex);

    m_pMouseManager->Sample();

    return true;
}

bool SharedTask::ResolveXpadTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_XpadManagers);

    if (waitId != &m_XpadTimerEvent)
    {
        return false;
    }

    m_XpadTimerEvent.Clear();

    ::std::lock_guard<decltype(m_XpadMutex)> locker(m_XpadMutex);

    for (int i = 0; i < m_XpadManagerCount; ++i)
    {
        m_XpadManagers[i].Sample();
    }

    return true;
}

bool SharedTask::ResolveHomeButtonTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pHomeButtonManager);

    if (waitId != &m_HomeButtonTimerEvent)
    {
        return false;
    }

    m_HomeButtonTimerEvent.Clear();

    ::std::lock_guard<decltype(m_HomeButtonMutex)> locker(m_HomeButtonMutex);

    m_pHomeButtonManager->Sample();

    return true;
}

bool SharedTask::ResolveSleepButtonTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pSleepButtonManager);

    if (waitId != &m_SleepButtonTimerEvent)
    {
        return false;
    }

    m_SleepButtonTimerEvent.Clear();

    ::std::lock_guard<decltype(m_SleepButtonMutex)> locker(m_SleepButtonMutex);

    m_pSleepButtonManager->Sample();

    return true;
}

bool SharedTask::ResolveCaptureButtonTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCaptureButtonManager);

    if (waitId != &m_CaptureButtonTimerEvent)
    {
        return false;
    }

    m_CaptureButtonTimerEvent.Clear();

    ::std::lock_guard<decltype(m_CaptureButtonMutex)
                      > locker(m_CaptureButtonMutex);

    m_pCaptureButtonManager->Sample();

    return true;
}

bool SharedTask::ResolvePlayReportTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pPlayReportManager);

    if (waitId != &m_PlayReportTimerEvent)
    {
        return false;
    }

    m_PlayReportTimerEvent.Clear();

    m_pPlayReportManager->CheckForUpdates();

    return true;
}

bool SharedTask::ResolveAudioControlTimerEvent(
    const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAudioControlManager);

    if (waitId != &m_AudioControlTimerEvent)
    {
        return false;
    }

    m_AudioControlTimerEvent.Clear();

    ::std::lock_guard<decltype(m_AudioControlMutex)
                      > locker(m_AudioControlMutex);

    m_pAudioControlManager->Sample();

    return true;
}

::nn::os::SdkMutexType AhidTask::s_AhidMutex = NN_OS_SDK_MUTEX_INITIALIZER();

AhidTask::AhidTask() NN_NOEXCEPT
    : m_pAhidManager(nullptr)
{
    // 何もしない
}

AhidTask::~AhidTask() NN_NOEXCEPT
{
    // 何もしない
}

void AhidTask::SetAhidManager(AhidManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimerEvent(&m_AhidTimerEvent);
    m_pAhidManager = pManager;
}

::nn::Result AhidTask::ActivateAhid() NN_NOEXCEPT
{
    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    {
        ::std::lock_guard<decltype(s_AhidMutex)> locker(s_AhidMutex);

        NN_RESULT_DO(m_pAhidManager->Activate());
    }

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result AhidTask::DeactivateAhid() NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(s_AhidMutex)> locker(s_AhidMutex);

        NN_RESULT_DO(m_pAhidManager->Deactivate());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

void AhidTask::LinkTimerEvent(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    NN_UNUSED(pMultiWait);
}

void AhidTask::UnlinkTimerEvent() NN_NOEXCEPT
{
    // 何もしない
}

void AhidTask::DoSampling(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    NN_UNUSED(waitId);
}

void AhidTask::LinkTimers(SynchronizedTimerGroup* group) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(group);
    m_AhidTimerEvent.Link(group);
}

void AhidTask::UnlinkTimers() NN_NOEXCEPT
{
    m_AhidTimerEvent.Unlink();
}

void AhidTask::DoSamplingForTimer(const SynchronizedTimer* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(waitId);
    NN_SDK_REQUIRES_NOT_NULL(m_pAhidManager);

    if (waitId == &m_AhidTimerEvent)
    {
        m_AhidTimerEvent.Clear();

        {
            ::std::lock_guard<decltype(s_AhidMutex)> locker(s_AhidMutex);

            m_pAhidManager->DoPreProcessing();
        }

        m_pAhidManager->Communicate();

        {
            ::std::lock_guard<decltype(s_AhidMutex)> locker(s_AhidMutex);

            m_pAhidManager->DoPostProcessing();
        }

        return;
    }
}

TouchScreenTask::TouchScreenTask() NN_NOEXCEPT
{
    // 何もしない
}

TouchScreenTask::~TouchScreenTask() NN_NOEXCEPT
{
    // 何もしない
}

void TouchScreenTask::SetTouchScreenManager(
    TouchScreenManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimerEvent(m_TouchScreenTimerEvent.GetBase());
    m_pTouchScreenManager = pManager;
}

::nn::Result TouchScreenTask::ActivateGesture() NN_NOEXCEPT
{
    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    {
        ::std::lock_guard<decltype(m_TouchScreenMutex)
                          > locker(m_TouchScreenMutex);
        NN_RESULT_DO(m_pTouchScreenManager->ActivateForGesture());
    }

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenTask::DeactivateGesture() NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(m_TouchScreenMutex)
                          > locker(m_TouchScreenMutex);
        NN_RESULT_DO(m_pTouchScreenManager->DeactivateForGesture());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenTask::EnsureGestureAppletResource(
    ::nn::applet::AppletResourceUserId aruid, int32_t version) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTouchScreenManager);
    ::std::lock_guard<decltype(m_TouchScreenMutex)> locker(m_TouchScreenMutex);
    NN_RESULT_DO(
        m_pTouchScreenManager->EnsureAppletResourceForGesture(aruid, version));
    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenTask::ActivateTouchScreen() NN_NOEXCEPT
{
    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    {
        ::std::lock_guard<decltype(m_TouchScreenMutex)
                          > locker(m_TouchScreenMutex);
        NN_RESULT_DO(m_pTouchScreenManager->ActivateForTouchScreen());
    }

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenTask::DeactivateTouchScreen() NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(m_TouchScreenMutex)
                          > locker(m_TouchScreenMutex);
        NN_RESULT_DO(m_pTouchScreenManager->DeactivateForTouchScreen());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenTask::EnsureTouchScreenAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTouchScreenManager);
    ::std::lock_guard<decltype(m_TouchScreenMutex)> locker(m_TouchScreenMutex);
    NN_RESULT_DO(
        m_pTouchScreenManager->EnsureAppletResourceForTouchScreen(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenTask::SetTouchScreenAutoPilotState(
    const ::nn::hid::debug::TouchScreenAutoPilotState<TouchStateCountMax>& value
    ) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_TouchScreenMutex)> locker(m_TouchScreenMutex);
    NN_RESULT_THROW(m_pTouchScreenManager->SetAutoPilotState(value));
}

::nn::Result TouchScreenTask::UnsetTouchScreenAutoPilotState() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_TouchScreenMutex)> locker(m_TouchScreenMutex);
    NN_RESULT_THROW(m_pTouchScreenManager->UnsetAutoPilotState());
}

::nn::Result TouchScreenTask::WakeTouchScreenUp() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_TouchScreenMutex)> locker(m_TouchScreenMutex);
    NN_RESULT_DO(m_pTouchScreenManager->WakeDeviceUp());
    NN_RESULT_SUCCESS;
}

::nn::Result TouchScreenTask::PutTouchScreenToSleep() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_TouchScreenMutex)> locker(m_TouchScreenMutex);
    NN_RESULT_DO(m_pTouchScreenManager->PutDeviceToSleep());
    NN_RESULT_SUCCESS;
}

void TouchScreenTask::LinkTimerEvent(
    ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    m_TouchScreenTimerEvent.Link(pMultiWait);
}

void TouchScreenTask::UnlinkTimerEvent() NN_NOEXCEPT
{
    m_TouchScreenTimerEvent.Unlink();
}

void TouchScreenTask::DoSampling(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTouchScreenManager);

    if (waitId == m_TouchScreenTimerEvent.GetWaitId())
    {
        m_TouchScreenTimerEvent.Clear();
        ::std::lock_guard<decltype(m_TouchScreenMutex)
                          > locker(m_TouchScreenMutex);
        m_pTouchScreenManager->Sample();
        return;
    }
}

ConsoleSixAxisSensorTask::ConsoleSixAxisSensorTask() NN_NOEXCEPT
{
    // 何もしない
}

ConsoleSixAxisSensorTask::~ConsoleSixAxisSensorTask() NN_NOEXCEPT
{
    // 何もしない
}

void ConsoleSixAxisSensorTask::SetConsoleSixAxisSensorManager(
    ConsoleSixAxisSensorManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetTimerEvent(m_ConsoleSixAxisSensorTimerEvent.GetBase());
    m_pConsoleSixAxisSensorManager = pManager;
}

::nn::Result ConsoleSixAxisSensorTask::ActivateConsoleSixAxisSensor() NN_NOEXCEPT
{
    auto needsRollback = true;

    NN_RESULT_DO(this->ActivateThread());

    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    {
        ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)
                          > locker(m_ConsoleSixAxisSensorMutex);
        NN_RESULT_DO(m_pConsoleSixAxisSensorManager->ActivateForConsoleSixAxisSensor());
    }

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::DeactivateConsoleSixAxisSensor() NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)
                          > locker(m_ConsoleSixAxisSensorMutex);
        NN_RESULT_DO(m_pConsoleSixAxisSensorManager->DeactivateForConsoleSixAxisSensor());
    }

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::EnsureConsoleSixAxisSensorAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);
    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->EnsureAppletResourceForConsoleSixAxisSensor(aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::WakeConsoleSixAxisSensorUp() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);
    NN_RESULT_DO(m_pConsoleSixAxisSensorManager->WakeDeviceUp());
    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::PutConsoleSixAxisSensorToSleep() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);
    NN_RESULT_DO(m_pConsoleSixAxisSensorManager->PutDeviceToSleep());
    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::GetConsoleSixAxisSensorCountStates(
    int* pOutCount,
    ::nn::hid::debug::ConsoleSixAxisSensorCountState* pOutValues,
    int count) NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)
        > locker(m_ConsoleSixAxisSensorMutex);

        *pOutCount = m_pConsoleSixAxisSensorManager
            ->GetConsoleSixAxisSensorCountStates(pOutValues, count);
    }
    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::GetConsoleSixAxisSensorSamplingFrequency(
    int64_t* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_UNUSED(aruid);

    {
        ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)
        > locker(m_ConsoleSixAxisSensorMutex);
        *pOutValue = m_pConsoleSixAxisSensorManager->GetSamplingFrequency();
    }

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::GetConsoleSixAxisSensorCalibrationValues(
    tmp::ConsoleSixAxisSensorCalibrationValues* pOutValues) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValues);
    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->GetConsoleSixAxisSensorCalibrationValues(pOutValues));
    NN_RESULT_SUCCESS;
}

void ConsoleSixAxisSensorTask::LinkTimerEvent(
    ::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    m_ConsoleSixAxisSensorTimerEvent.Link(pMultiWait);
}

void ConsoleSixAxisSensorTask::UnlinkTimerEvent() NN_NOEXCEPT
{
    m_ConsoleSixAxisSensorTimerEvent.Unlink();
}

void ConsoleSixAxisSensorTask::DoSampling(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);

    if (waitId == m_ConsoleSixAxisSensorTimerEvent.GetWaitId())
    {
        m_ConsoleSixAxisSensorTimerEvent.Clear();
        ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)
                          > locker(m_ConsoleSixAxisSensorMutex);
        m_pConsoleSixAxisSensorManager->Sample();
        return;
    }
}

::nn::Result ConsoleSixAxisSensorTask::IsConsoleSixAxisSensorUserCalibrationSupported(bool* pOutIsSupported) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);

    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager
            ->IsConsoleSixAxisSensorUserCalibrationSupported(pOutIsSupported));

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::ResetConsoleSixAxisSensorCalibrationValues() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);

    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->ResetConsoleSixAxisSensorCalibrationValues());

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::StartConsoleSixAxisSensorUserCalibration() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);

    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->StartConsoleSixAxisSensorUserCalibration());

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::CancelConsoleSixAxisSensorUserCalibration() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);

    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->CancelConsoleSixAxisSensorUserCalibration());

    NN_RESULT_SUCCESS;
}

::nn::Result
ConsoleSixAxisSensorTask::GetSixAxisSensorAccurateUserCalibrationState(
    ::nn::hid::system::SixAxisSensorAccurateUserCalibrationState* pOutState
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);

    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);

    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->GetSixAxisSensorAccurateUserCalibrationState(pOutState));

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::SetStateBufferMemoryHandle(
    const ::nn::applet::AppletResourceUserId& aruid,
    ::nn::os::NativeHandle&& transferMemoryHandle,
    size_t size, bool isManaged) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);

    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);

    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->SetStateBufferMemoryHandle(
            aruid, ::std::move(transferMemoryHandle), size, isManaged));

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::SetWorkBufferMemoryHandle(
    const ::nn::applet::AppletResourceUserId& aruid,
    ::nn::os::NativeHandle&& transferMemoryHandle,
    size_t size, bool isManaged) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);

    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);

    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->SetWorkBufferMemoryHandle(
            aruid, ::std::move(transferMemoryHandle), size, isManaged));

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::InitializeSevenSixAxisSensor(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);

    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);

    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->InitializeSevenSixAxisSensor(aruid));

    NN_RESULT_SUCCESS;
}

::nn::Result ConsoleSixAxisSensorTask::FinalizeSevenSixAxisSensor(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);

    ::std::lock_guard<decltype(m_ConsoleSixAxisSensorMutex)> locker(m_ConsoleSixAxisSensorMutex);

    NN_RESULT_DO(
        m_pConsoleSixAxisSensorManager->FinalizeSevenSixAxisSensor(aruid));

    NN_RESULT_SUCCESS;
}

XcdTask::XcdTask() NN_NOEXCEPT
    : m_XcdMutex()
{
    // 何もしない
}

XcdTask::~XcdTask() NN_NOEXCEPT
{
    // 何もしない
}

void XcdTask::SetXcdDriver(XcdDriver* pXcdDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pXcdDriver);
    m_pXcdDriver = pXcdDriver;
}

void XcdTask::LinkTimerEvent(::nn::os::MultiWaitType* pMultiWait) NN_NOEXCEPT
{
    m_pXcdDriver->LinkMultiWait(pMultiWait);

    // Xcd 内部で多重待ちオブジェクトへのイベントのリンクを行うので、ここで初期化する
    ::std::lock_guard<decltype(m_XcdMutex)> locker(m_XcdMutex);
    m_pXcdDriver->Activate();
}

void XcdTask::UnlinkTimerEvent() NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(m_XcdMutex)> locker(m_XcdMutex);
        m_pXcdDriver->Deactivate();
    }

    m_pXcdDriver->UnlinkMultiWait();
}

void XcdTask::DoSampling(
    const ::nn::os::MultiWaitHolderType* waitId) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_XcdMutex)> lock(m_XcdMutex);

    if (m_pXcdDriver->Proceed(waitId) == true)
    {
        return;
    }
}

void XcdTask::InvokeEventLoop() NN_NOEXCEPT
{
    while (!nn::os::TryWaitEvent(m_TerminationEvent.GetBase()))
    {
        auto* pWaitId = m_pXcdDriver->WaitEvent();
        if (pWaitId == nullptr)
        {
            std::lock_guard<decltype(m_XcdMutex)> lock(m_XcdMutex);
            m_pXcdDriver->ProceedPeriodic();
        }
        else
        {
            std::lock_guard<decltype(m_XcdMutex)> lock(m_XcdMutex);
            m_pXcdDriver->Proceed(pWaitId);
        }
    }
}

::nn::Result XcdTask::ActivateXcdDriver() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXcdDriver);

    auto needsRollback = true;
    NN_RESULT_DO(this->ActivateThread());
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            this->DeactivateThread();
        }
    };

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

::nn::Result XcdTask::DeactivateXcdDriver() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXcdDriver);

    NN_RESULT_DO(this->DeactivateThread());

    NN_RESULT_SUCCESS;
}

::nn::Result XcdTask::Suspend() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXcdDriver);

    NN_RESULT_DO(m_pXcdDriver->Suspend());

    NN_RESULT_SUCCESS;
}

::nn::Result XcdTask::Resume() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXcdDriver);

    NN_RESULT_DO(m_pXcdDriver->Resume());

    NN_RESULT_SUCCESS;
}

::nn::Result XcdTask::EnableWiredDeviceRegistration() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXcdDriver);

    NN_RESULT_DO(m_pXcdDriver->EnableWiredDeviceRegistration());

    NN_RESULT_SUCCESS;
}

::nn::Result XcdTask::DisableWiredDeviceRegistration() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pXcdDriver);

    NN_RESULT_DO(m_pXcdDriver->DisableWiredDeviceRegistration());

    NN_RESULT_SUCCESS;
}

ResourceManager::ResourceManager() NN_NOEXCEPT
{
    // 共有タスクを初期化
    this->InitializeSharedTask();

    // TouchScreen タスクを初期化
    this->InitializeTouchScreenTask();

    // ConsoleSixAxisSensor タスクを初期化
    this->InitializeConsoleSixAxisSensorTask();

    // ahid タスクを初期化
    this->InitializeAhidTask();

    // Xcd タスクを初期化
    m_XcdTask.SetThreadStack(s_XcdTaskThreadStack,
                             sizeof(s_XcdTaskThreadStack));
    m_XcdTask.SetThreadPriority(NN_SYSTEM_THREAD_PRIORITY(hid, XcdDriver));
    m_XcdTask.SetThreadNamePointer(NN_SYSTEM_THREAD_NAME(hid, XcdDriver));
    m_XcdTask.SetXcdDriver(&m_XcdDriver);

    // 基本的な構成を持つ Xpad の解決された ID の初期値
    const BasicXpadId BasicXpadIds[] =
    {
        MakeXpadId<BasicXpadId>(0),
        MakeXpadId<BasicXpadId>(1),
        MakeXpadId<BasicXpadId>(2),
        MakeXpadId<BasicXpadId>(3)
    };

    // JoyXpad の解決された ID の初期値
    const JoyXpadId JoyXpadIds[] =
    {
        MakeXpadId<JoyXpadId>(0),
        MakeXpadId<JoyXpadId>(1),
        MakeXpadId<JoyXpadId>(2),
        MakeXpadId<JoyXpadId>(3)
    };

    // Xpad ID マネージャを初期化
    m_XpadIdManager.SetBasicXpadIds(BasicXpadIds, NN_ARRAY_SIZE(BasicXpadIds));
    m_XpadIdManager.SetJoyXpadIds(JoyXpadIds, NN_ARRAY_SIZE(JoyXpadIds));

    // DebugPadDriver に Xcd ドライバをセット
    // ペアリングトリガーのため (削除予定)
    m_DebugPadDriver.SetXcdDriver(&m_XcdDriver);
}

SharedTask& ResourceManager::GetDebugPadTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetKeyboardTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetMouseTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

TouchScreenTask& ResourceManager::GetGestureTask() NN_NOEXCEPT
{
    return m_TouchScreenTask;
}

TouchScreenTask& ResourceManager::GetTouchScreenTask() NN_NOEXCEPT
{
    return m_TouchScreenTask;
}

ConsoleSixAxisSensorTask& ResourceManager::GetConsoleSixAxisSensorTask() NN_NOEXCEPT
{
    return m_ConsoleSixAxisSensorTask;
}

SharedTask& ResourceManager::GetXpadTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetInputDetectorTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetHomeButtonTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetSleepButtonTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetCaptureButtonTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

AhidTask (&ResourceManager::GetAhidTasks() NN_NOEXCEPT)[2 * AhidPortCount + 1]
{
    return m_AhidTasks;
}

SharedTask& ResourceManager::GetXpadControlTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

XcdTask& ResourceManager::GetXcdTask() NN_NOEXCEPT
{
    return m_XcdTask;
}

SharedTask& ResourceManager::GetAbstractedPadXcdTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetAbstractedAutoPilotPadTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetPlayReportTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetRegisteredDeviceTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

SharedTask& ResourceManager::GetAudioControlTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

void ResourceManager::SetAppletResourceUserId(
    ::nn::applet::AppletResourceUserId value) NN_NOEXCEPT
{
    {
        ::std::lock_guard<decltype(m_AppletResourceMutex)
        > locker(m_AppletResourceMutex);
        m_AppletResourceManager.SetAppletResourceUserId(value);
        m_UniquePadResourceManager.NotifyAppletResourceUserId(value);
    }
}

void ResourceManager::EnableHandheldHids() NN_NOEXCEPT
{
    m_HandheldManager.SetHandheldEnabled(true);
}

void ResourceManager::DisableHandheldHids() NN_NOEXCEPT
{
    m_HandheldManager.SetHandheldEnabled(false);
}

::nn::Result ResourceManager::RegisterAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid,
    bool enablesInput) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    NN_RESULT_DO(
        m_AppletResourceManager
            .RegisterAppletResourceUserId(aruid, enablesInput));
    NN_RESULT_SUCCESS;
}

void ResourceManager::UnregisterAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    m_AppletResourceManager.UnregisterAppletResourceUserId(aruid);
}

::nn::Result ResourceManager::SetAppletResourceEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    NN_RESULT_DO(m_AppletResourceManager.SetAppletResourceEntry(aruid));
    NN_RESULT_SUCCESS;
}

void ResourceManager::ResetAppletResourceEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    m_AppletResourceManager.ResetAppletResourceEntry(aruid);
}

void ResourceManager::EnableAppletToGetInput(
    ::nn::applet::AppletResourceUserId aruid,
    bool enablesInput) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    m_AppletResourceManager.EnableAppletResourceInput(aruid, enablesInput);
}

void ResourceManager::EnableAppletToGetSixAxisSensor(
    ::nn::applet::AppletResourceUserId aruid,
    bool enablesInput) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    m_AppletResourceManager.EnableAppletResourceSixAxisSensor(aruid, enablesInput);
}

::nn::Result ResourceManager::GetAppletResourceSharedMemoryHandle(
    ::nn::os::NativeHandle* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    NN_RESULT_DO(
        m_AppletResourceManager.GetAppletResourceSharedMemoryHandle(
            pOutValue, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::ActivateSharedAppletResource() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    NN_RESULT_DO(m_AppletResourceManager.ActivateSharedAppletResource());
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::DeactivateSharedAppletResource() NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_AppletResourceMutex)
                      > locker(m_AppletResourceMutex);
    NN_RESULT_DO(m_AppletResourceManager.DeactivateSharedAppletResource());
    NN_RESULT_SUCCESS;
}

const SixAxisSensorProcessor& ResourceManager::GetSixAxisSensorProcessor(
    const SixAxisSensorHandle& handle) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        IsSixAxisSensorHandleNpadTypeEqualTo(handle));

    auto pProcessor = m_NpadResourceManager.GetSixAxisSensor(handle);
    NN_SDK_REQUIRES_NOT_NULL(pProcessor);
    return *pProcessor;
}

SixAxisSensorAppletSettingManager&
ResourceManager::GetSixAxisSensorAppletSettingManager() NN_NOEXCEPT
{
    return m_SixAxisSensorAppletSettingManager;
}

ConsoleSixAxisSensorAppletSettingManager&
ResourceManager::GetConsoleSixAxisSensorAppletSettingManager() NN_NOEXCEPT
{
    return m_ConsoleSixAxisSensorAppletSettingManager;
}

SharedTask& ResourceManager::GetFirmwareUpdaterTask() NN_NOEXCEPT
{
    return m_SharedTask;
}

XcdDriver& ResourceManager::GetXcdDriver() NN_NOEXCEPT
{
    return m_XcdDriver;
}

::nn::Result ResourceManager::AssertValidBasicXpadPlayerNumber(
    int playerNumber) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        0 <= playerNumber && playerNumber < XpadPlayerCountMax,
        ResultBasicXpadPlayerNumberOutOfRange());
    NN_RESULT_SUCCESS;
}


::nn::Result ResourceManager::AssertValidJoyXpadPlayerNumber(
    int playerNumber) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        0 <= playerNumber && playerNumber < XpadPlayerCountMax,
        ResultJoyXpadPlayerNumberOutOfRange());
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::AcquireXpadIdEventHandle(
    ::nn::os::NativeHandle* outHandle,
    uint64_t clientId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outHandle);
    ::std::lock_guard<decltype(m_XpadIdManagerMutex)
                      > locker(m_XpadIdManagerMutex);
    NN_RESULT_DO(m_XpadIdManager.AcquireXpadIdEventHandle(outHandle, clientId));
    // ハンドルの取得直後はシグナル状態であることを保証
    m_XpadIdManager.SignalXpadIdSystemEvent();
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::ReleaseXpadIdEventHandle(
    uint64_t clientId) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_XpadIdManagerMutex)
                      > locker(m_XpadIdManagerMutex);
    NN_RESULT_THROW(m_XpadIdManager.ReleaseXpadIdEventHandle(clientId));
}

::nn::Result ResourceManager::GetXpadIds(
    int* outCount, BasicXpadId* outXpadIds, int count) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_XpadIdManagerMutex)
                      > locker(m_XpadIdManagerMutex);
    NN_RESULT_THROW(
        m_XpadIdManager.GetBasicXpadIds(outCount, outXpadIds, count));
}

::nn::Result ResourceManager::GetJoyXpadIds(
    int* outCount, JoyXpadId* outXpadIds, int count) NN_NOEXCEPT
{
    ::std::lock_guard<decltype(m_XpadIdManagerMutex)
                      > locker(m_XpadIdManagerMutex);
    NN_RESULT_THROW(
        m_XpadIdManager.GetJoyXpadIds(outCount, outXpadIds, count));
}

void ResourceManager::ActivateSenderForOverlay() NN_NOEXCEPT
{
    m_OvlnSender.Activate();
}

void ResourceManager::DeactivateSenderForOverlay() NN_NOEXCEPT
{
    m_OvlnSender.Deactivate();
}

::nn::Result ResourceManager::AcquireJoyDetachOnBluetoothOffEventHandle(
    ::nn::os::NativeHandle* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_DO(
        m_InterruptSceneNotifier.AcquireJoyDetachOnBluetoothOffEventHandle(pOutValue, aruid));
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::ActivatePadCommonTimer() NN_NOEXCEPT
{
    NN_RESULT_DO(m_SharedTask.ActivatePadCommonTimer());
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::DeactivatePadCommonTimer() NN_NOEXCEPT
{
    NN_RESULT_DO(m_SharedTask.DeactivatePadCommonTimer());
    NN_RESULT_SUCCESS;
}

::nn::Result ResourceManager::DeassertTouchIcReset() NN_NOEXCEPT
{
    m_TouchScreenDriver.DeassertReset();
    NN_RESULT_SUCCESS;
}

NpadResourceManager& ResourceManager::GetNpadResourceManager() NN_NOEXCEPT
{
    return m_NpadResourceManager;
}

UniquePadResourceManager& ResourceManager::GetUniquePadResourceManager() NN_NOEXCEPT
{
    return m_UniquePadResourceManager;
}

AbstractedPadManager& ResourceManager::GetAbstractedPadManager() NN_NOEXCEPT
{
    return m_AbstractedPadManager;
}

PalmaResourceManager& ResourceManager::GetPalmaResourceManager() NN_NOEXCEPT
{
    return m_PalmaResourceManager;
}

template<typename ManagerT>
void ResourceManager::InjectAppletResourceManager(
    ManagerT* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetAppletResourceManager(
        &m_AppletResourceManager, &m_AppletResourceMutex);
}

template<typename ManagerT>
void ResourceManager::InjectInputDetectorManager(ManagerT* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);
    pManager->SetInputDetectorManager(
        &m_InputDetectorManager, &m_InputDetectorMutex);
}

void ResourceManager::InitializeDebugPadManager() NN_NOEXCEPT
{
    m_DebugPadDriver.SetHomeButtonNotifier(
        m_HomeButtonManager.MakeSystemButtonNotifier());
    m_DebugPadManager.SetDriver(&m_DebugPadDriver);
    this->InjectAppletResourceManager(&m_DebugPadManager);
    this->InjectInputDetectorManager(&m_DebugPadManager);
}

void ResourceManager::InitializeKeyboardManager() NN_NOEXCEPT
{
    m_KeyboardManager.SetDriver(&m_KeyboardDriver);
    this->InjectAppletResourceManager(&m_KeyboardManager);
    this->InjectInputDetectorManager(&m_KeyboardManager);
}

void ResourceManager::InitializeMouseManager() NN_NOEXCEPT
{
    m_MouseManager.SetDriver(&m_MouseDriver);
    this->InjectAppletResourceManager(&m_MouseManager);
    this->InjectInputDetectorManager(&m_MouseManager);
}

void ResourceManager::InitializeHomeButtonManager() NN_NOEXCEPT
{
    this->InjectAppletResourceManager(&m_HomeButtonManager);
    this->InjectInputDetectorManager(&m_HomeButtonManager);
}

void ResourceManager::InitializeSleepButtonManager() NN_NOEXCEPT
{
    m_SleepButtonManager.SetDriver(&m_SleepButtonDriver);
    this->InjectAppletResourceManager(&m_SleepButtonManager);
    this->InjectInputDetectorManager(&m_SleepButtonManager);
}

void ResourceManager::InitializeCaptureButtonManager() NN_NOEXCEPT
{
    this->InjectAppletResourceManager(&m_CaptureButtonManager);
    this->InjectInputDetectorManager(&m_CaptureButtonManager);
}

void ResourceManager::InitializeRegisteredDeviceManager() NN_NOEXCEPT
{
    m_RegisteredDeviceManager.SetOvlnSenderManager(&m_OvlnSender);
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPadUsb); i++)
    {
        m_RegisteredDeviceManager.AddIAbstractedPad(&m_AbstractedPadUsb[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPadXcd); i++)
    {
        m_RegisteredDeviceManager.AddIAbstractedPad(&m_AbstractedPadXcd[i]);
    }
}

void ResourceManager::InitializeXpadManagers() NN_NOEXCEPT
{
    for (int i = 0; i < XpadPlayerCountMax; ++i)
    {
        m_XpadManagers[i].SetDriver(&m_XpadDrivers[i]);
        this->InjectAppletResourceManager(&m_XpadManagers[i]);
        m_XpadManagers[i].SetXpadId(MakeXpadId<BasicXpadId>(i));
    }
}

void ResourceManager::InitializeAbstractedPadXcdManager() NN_NOEXCEPT
{
    m_AbstractedPadXcdManager.SetAbstractedPadXcds(
        m_AbstractedPadXcd, NN_ARRAY_SIZE(m_AbstractedPadXcd));
    m_AbstractedPadXcdManager.SetAbstractedPadIdPublisher(
        &m_AbstractedIdPublisher);
}

void ResourceManager::InitializeAbstractedPalmaPadManager() NN_NOEXCEPT
{
    m_AbstractedPalmaPadManager.SetAbstractedPalmaPads(
        m_AbstractedPalmaPad, NN_ARRAY_SIZE(m_AbstractedPalmaPad));
    m_AbstractedPalmaPadManager.SetAbstractedPadIdPublisher(
        &m_AbstractedIdPublisher);
}

void ResourceManager::InitializeAbstractedGcAdapterPadManager() NN_NOEXCEPT
{
    for (auto& pad : m_AbstractedGcAdapterPads)
    {
        pad.SetGcAdapterDriver(&m_GcAdapterDriver);
    }
    m_AbstractedGcAdapterPadManager.SetAbstractedGcAdapterPads(
        m_AbstractedGcAdapterPads, NN_ARRAY_SIZE(m_AbstractedGcAdapterPads));
    m_AbstractedGcAdapterPadManager.SetAbstractedPadIdPublisher(
        &m_AbstractedIdPublisher);
    m_AbstractedGcAdapterPadManager.SetGcAdapterDriver(&m_GcAdapterDriver);
}

void ResourceManager::InitializeAbstractedAutoPilotPadManager() NN_NOEXCEPT
{
    m_AbstractedAutoPilotPadManager.SetAbstractedAutoPilotPads(
        m_AbstractedAutoPilotPad, NN_ARRAY_SIZE(m_AbstractedAutoPilotPad));
    for (auto& pad : m_AbstractedAutoPilotPad)
    {
        pad.SetAbstractedPadIdPublisher(&m_AbstractedIdPublisher);
    }
}

void ResourceManager::InitializeAbstractedAutoPilotVirtualPadManager() NN_NOEXCEPT
{
    m_AbstractedAutoPilotVirtualPadManager.SetAbstractedAutoPilotVirtualPads(
        m_AbstractedAutoPilotVirtualPad,
        NN_ARRAY_SIZE(m_AbstractedAutoPilotVirtualPad));
    for (auto &pad : m_AbstractedAutoPilotVirtualPad)
    {
        pad.SetAbstractedPadIdPublisher(&m_AbstractedIdPublisher);
    }
}

void ResourceManager::InitializeSharedTask() NN_NOEXCEPT
{
    m_SharedTask.SetThreadStack(
        s_SharedTaskThreadStack, sizeof(s_SharedTaskThreadStack));
    m_SharedTask.SetThreadPriority(
        NN_SYSTEM_THREAD_PRIORITY(hid, CommonSampler));
    m_SharedTask.SetThreadNamePointer(
        NN_SYSTEM_THREAD_NAME(hid, CommonSampler));

    this->InitializeDebugPadManager();
    m_SharedTask.SetDebugPadManager(&m_DebugPadManager);

    this->InitializeKeyboardManager();
    m_SharedTask.SetKeyboardManager(&m_KeyboardManager);

    this->InitializeMouseManager();
    m_SharedTask.SetMouseManager(&m_MouseManager);

    m_SharedTask.SetInputDetectorManager(&m_InputDetectorManager);

    this->InjectAppletResourceManager(&m_InputDetectorManager);
    m_SharedTask.SetInputDetectorManager(&m_InputDetectorManager);

    this->InitializeHomeButtonManager();
    m_SharedTask.SetHomeButtonManager(&m_HomeButtonManager);

    this->InitializeSleepButtonManager();
    m_SharedTask.SetSleepButtonManager(&m_SleepButtonManager);

    this->InitializeCaptureButtonManager();
    m_SharedTask.SetCaptureButtonManager(&m_CaptureButtonManager);

    m_SharedTask.SetPlayReportManager(&m_PlayReportManager);

    this->InitializeRegisteredDeviceManager();
    m_SharedTask.SetRegisteredDeviceManager(&m_RegisteredDeviceManager);

    this->InjectAppletResourceManager(&m_SixAxisSensorAppletSettingManager);
    m_ConsoleSixAxisSensorManager.SetDriver(&m_ConsoleSixAxisSensorDriver);
    this->InjectAppletResourceManager(&m_ConsoleSixAxisSensorManager);
    m_ConsoleSixAxisSensorManager.SetSixAxisSensorAppletSettingManager(
        &m_SixAxisSensorAppletSettingManager);
    m_ConsoleSixAxisSensorManager.SetConsoleSixAxisSensorAppletSettingManager(
        &m_ConsoleSixAxisSensorAppletSettingManager);
    m_ConsoleSixAxisSensorManager.SetHandheldManager(&m_HandheldManager);
    this->InjectInputDetectorManager(&m_ConsoleSixAxisSensorManager);
    m_ConsoleSixAxisSensorManager.SetConsoleSixAxisSensorProcessor(
        &m_ConsoleSixAxisSensorProcessor);
    m_ConsoleSixAxisSensorManager.SetSixAxisSensorFilter(
        &m_SixAxisSensorSanjiroFilter);
    this->InjectAppletResourceManager(
        &m_ConsoleSixAxisSensorAppletSettingManager);

    this->InitializeXpadManagers();
    m_SharedTask.SetXpadManagers(m_XpadManagers, XpadPlayerCountMax);

    m_AbstractedPadUsbDriver.SetAbstractedPadUsbs(
        m_AbstractedPadUsb, NN_ARRAY_SIZE(m_AbstractedPadUsb));
    m_AbstractedPadUsbDriver.SetAbstractedPadIdPublisher(
        &m_AbstractedIdPublisher);

    this->InitializeAbstractedPadXcdManager();
    m_SharedTask.SetAbstractedPadXcd(
        m_AbstractedPadXcd, NN_ARRAY_SIZE(m_AbstractedPadXcd));
    m_SharedTask.SetAbstractedPadXcdManager(&m_AbstractedPadXcdManager);

    this->InitializeAbstractedPalmaPadManager();
    m_SharedTask.SetAbstractedPalmaPad(
        m_AbstractedPalmaPad, NN_ARRAY_SIZE(m_AbstractedPalmaPad));
    m_SharedTask.SetAbstractedPalmaPadManager(&m_AbstractedPalmaPadManager);

    this->InitializeAbstractedGcAdapterPadManager();
    m_SharedTask.SetAbstractedGcAdapterPadManager(&m_AbstractedGcAdapterPadManager);

    this->InitializeAbstractedAutoPilotPadManager();
    m_SharedTask.SetAbstractedAutoPilotPadManager(&m_AbstractedAutoPilotPadManager);

    this->InitializeAbstractedAutoPilotVirtualPadManager();
    m_SharedTask.SetAbstractedAutoPilotVirtualPadManager(&m_AbstractedAutoPilotVirtualPadManager);

    m_AbstractedFiftyPad.SetAbstractedPadIdPublisher(&m_AbstractedIdPublisher);
    m_SharedTask.SetAbstractedFiftyPad(&m_AbstractedFiftyPad);

    m_RailManager.SetOvlnSenderManager(&m_OvlnSender);
    m_RailManager.SetInterruptSceneNotifier(&m_InterruptSceneNotifier);
    m_SharedTask.SetRailManager(&m_RailManager);

    m_AudioControlManager.SetDriver(&m_AudioControlDriver);
    m_SharedTask.SetAudioControlManager(&m_AudioControlManager);

    this->InitializeUniquePad();
    this->InitializeNpad();
    this->InitializeAbstractedPadManager();

    // Palma の初期化
    this->InjectAppletResourceManager(&m_PalmaResourceManager);
    m_SharedTask.SetPalmaResourceManager(&m_PalmaResourceManager);
}

void ResourceManager::InitializeTouchScreenTask() NN_NOEXCEPT
{
    m_TouchScreenTask.SetThreadStack(
        s_TouchScreenTaskThreadStack, sizeof(s_TouchScreenTaskThreadStack));
    m_TouchScreenTask.SetThreadPriority(
        NN_SYSTEM_THREAD_PRIORITY(hid, TouchScreenSampler));
    m_TouchScreenTask.SetThreadNamePointer(
        NN_SYSTEM_THREAD_NAME(hid, TouchScreenSampler));

    m_TouchScreenManager.SetDriver(&m_TouchScreenDriver);
    this->InjectAppletResourceManager(&m_TouchScreenManager);
    this->InjectInputDetectorManager(&m_TouchScreenManager);
    m_TouchScreenManager.SetHandheldManager(&m_HandheldManager);
    m_TouchScreenTask.SetTouchScreenManager(&m_TouchScreenManager);
}

void ResourceManager::InitializeConsoleSixAxisSensorTask() NN_NOEXCEPT
{
    m_ConsoleSixAxisSensorTask.SetThreadStack(
        s_ConsoleSixAxisSensorTaskThreadStack,
        sizeof(s_ConsoleSixAxisSensorTaskThreadStack));
    m_ConsoleSixAxisSensorTask.SetThreadPriority(
        NN_SYSTEM_THREAD_PRIORITY(hid, ConsoleSixAxisSensorSampler));
    m_ConsoleSixAxisSensorTask.SetThreadNamePointer(
        NN_SYSTEM_THREAD_NAME(hid, ConsoleSixAxisSensorSampler));

    m_ConsoleSixAxisSensorManager.SetDriver(&m_ConsoleSixAxisSensorDriver);
    this->InjectAppletResourceManager(&m_ConsoleSixAxisSensorManager);
    this->InjectInputDetectorManager(&m_ConsoleSixAxisSensorManager);
    m_ConsoleSixAxisSensorManager.SetHandheldManager(&m_HandheldManager);
    m_ConsoleSixAxisSensorTask.SetConsoleSixAxisSensorManager(
        &m_ConsoleSixAxisSensorManager);
}

void ResourceManager::InitializeAhidTask() NN_NOEXCEPT
{
    // ahid タスクの受信タスク向けの初期化処理を実施
    for (size_t i = 0; i < AhidPortCount; ++i)
    {
        m_AhidManagers[i].SetPort(AhidManager::PortType_In, i);
    }

    // ahid タスクの送信タスク向けの初期化処理を実施
    for (size_t i = 0; i < AhidPortCount; ++i)
    {
        m_AhidManagers[i + AhidPortCount].SetPort(AhidManager::PortType_Out, i);
    }

    // ahid タスクの制御タスク向けの初期化処理を実施
    m_AhidManagers[2 * AhidPortCount].SetPort(AhidManager::PortType_Control, 0);

    // ahid タスクの共通の初期化処理を実施
    for (size_t i = 0; i < NN_ARRAY_SIZE(m_AhidTasks); ++i)
    {
        m_AhidManagers[i].SetKeyboardDriver(&m_KeyboardDriver);
        m_AhidManagers[i].SetMouseDriver(&m_MouseDriver);
        m_AhidManagers[i].SetAbstractedPadUsbDriver(&m_AbstractedPadUsbDriver);
        m_AhidManagers[i].SetAudioControlDriver(&m_AudioControlDriver);
        m_AhidManagers[i].SetGcAdapterDriver(&m_GcAdapterDriver);
        m_AhidTasks[i].SetAhidManager(&m_AhidManagers[i]);
        m_AhidTasks[i].SetThreadStack(
            s_AhidTaskThreadStacks[i], sizeof(s_AhidTaskThreadStacks[i]));
        m_AhidTasks[i].SetThreadPriority(
            NN_SYSTEM_THREAD_PRIORITY(hid, AhidSampler));
        m_AhidTasks[i].SetThreadNamePointer(
            NN_SYSTEM_THREAD_NAME(hid, AhidSampler));
    }
}

void ResourceManager::InitializeUniquePad() NN_NOEXCEPT
{
    m_UniquePadResourceManager.Initialize(&m_HandheldManager,
                                          &m_InterruptSceneNotifier);
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPadUsb); i++)
    {
        m_UniquePadResourceManager.AddIAbstractedPad(&m_AbstractedPadUsb[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPadXcd); i++)
    {
        m_UniquePadResourceManager.AddIAbstractedPad(&m_AbstractedPadXcd[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPalmaPad); i++)
    {
        m_UniquePadResourceManager.AddIAbstractedPad(&m_AbstractedPalmaPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedAutoPilotPad); i++)
    {
        m_UniquePadResourceManager.AddIAbstractedPad(&m_AbstractedAutoPilotPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedAutoPilotVirtualPad); i++)
    {
        m_UniquePadResourceManager.AddIAbstractedPad(&m_AbstractedAutoPilotVirtualPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedGcAdapterPads); i++)
    {
        m_UniquePadResourceManager.AddIAbstractedPad(&m_AbstractedGcAdapterPads[i]);
    }
    m_UniquePadResourceManager.AddIAbstractedPad(&m_AbstractedFiftyPad);
    m_SharedTask.SetUniquePadResourceManager(&m_UniquePadResourceManager);
}

void ResourceManager::InitializeNpad() NN_NOEXCEPT
{
    m_NpadResourceManager.Initialize(&m_CaptureButtonManager,
        &m_HomeButtonManager,
        &m_OvlnSender,
        &m_PlayReportManager,
        &m_AppletResourceManager,
        &m_AppletResourceMutex,
        &m_SixAxisSensorAppletSettingManager,
        &m_ConsoleSixAxisSensorManager,
        &m_HandheldManager,
        &m_InputDetectorManager,
        &m_InputDetectorMutex,
        &m_UniquePadResourceManager,
        &m_InterruptSceneNotifier,
        &m_GcAdapterDriver,
        &m_PalmaResourceManager
    );
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPadUsb); i++)
    {
        m_NpadResourceManager.AddIAbstractedPad(&m_AbstractedPadUsb[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPadXcd); i++)
    {
        m_NpadResourceManager.AddIAbstractedPad(&m_AbstractedPadXcd[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPalmaPad); i++)
    {
        m_NpadResourceManager.AddIAbstractedPad(&m_AbstractedPalmaPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedAutoPilotPad); i++)
    {
        m_NpadResourceManager.AddIAbstractedPad(&m_AbstractedAutoPilotPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedAutoPilotVirtualPad); i++)
    {
        m_NpadResourceManager.AddIAbstractedPad(&m_AbstractedAutoPilotVirtualPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedGcAdapterPads); i++)
    {
        m_NpadResourceManager.AddIAbstractedPad(&m_AbstractedGcAdapterPads[i]);
    }
    m_NpadResourceManager.AddIAbstractedPad(&m_AbstractedFiftyPad);
    m_SharedTask.SetNpadResourceManager(&m_NpadResourceManager);
}

void ResourceManager::InitializeAbstractedPadManager() NN_NOEXCEPT
{
    m_AbstractedPadManager.Initialize();
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPadUsb); i++)
    {
        m_AbstractedPadManager.AddIAbstractedPad(&m_AbstractedPadUsb[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPadXcd); i++)
    {
        m_AbstractedPadManager.AddIAbstractedPad(&m_AbstractedPadXcd[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedPalmaPad); i++)
    {
        m_AbstractedPadManager.AddIAbstractedPad(&m_AbstractedPalmaPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedAutoPilotPad); i++)
    {
        m_AbstractedPadManager.AddIAbstractedPad(&m_AbstractedAutoPilotPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedAutoPilotVirtualPad); i++)
    {
        m_AbstractedPadManager.AddIAbstractedPad(&m_AbstractedAutoPilotVirtualPad[i]);
    }
    for (int i = 0; i < NN_ARRAY_SIZE(m_AbstractedGcAdapterPads); i++)
    {
        m_AbstractedPadManager.AddIAbstractedPad(&m_AbstractedGcAdapterPads[i]);
    }
}

NN_ALIGNAS(0x1000) char ResourceManager::s_SharedTaskThreadStack[0x2000];

NN_ALIGNAS(0x1000) char ResourceManager::s_AhidTaskThreadStacks[
    2 * AhidPortCount + 1][0x1000];

NN_ALIGNAS(0x1000) char ResourceManager::s_TouchScreenTaskThreadStack[0x1000];

NN_ALIGNAS(0x1000) char ResourceManager::s_ConsoleSixAxisSensorTaskThreadStack[0x2000];

NN_ALIGNAS(0x1000) char ResourceManager::s_XcdTaskThreadStack[0x3000];

ResourceManager& GetResourceManager() NN_NOEXCEPT
{
    return StaticObject<ResourceManager>::Get();
}

}}} // namespace nn::hid::detail
