﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_IHidSystemServer.sfdl.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>

#include "hid_RegisteredDeviceImpl.h"
#include "hid_HidDebugServer.h"
#include "hid_HidSystemServer.h"

namespace nn { namespace hid { namespace detail {

::nn::Result GetRegisteredDevices(int* pOutCount,
                                  system::RegisteredDevice* outValues,
                                  int count) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidSystemServer> pProxy;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pProxy));
    auto gotCount = int64_t();
    NN_RESULT_DO(
        pProxy->GetRegisteredDevices(::nn::sf::Out<int64_t>(&gotCount),
                                     ::nn::sf::OutArray<system::RegisteredDevice>(
                                     outValues, static_cast<size_t>(count))));
    *pOutCount = static_cast<int>(gotCount);
    NN_RESULT_SUCCESS;
}

::nn::Result GetConnectableRegisteredDevices(int* pOutCount,
                                             system::RegisteredDevice* outValues,
                                             int count) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidSystemServer> pProxy;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pProxy));
    auto gotCount = int64_t();
    NN_RESULT_DO(
        pProxy->GetConnectableRegisteredDevices(::nn::sf::Out<int64_t>(&gotCount),
                                                ::nn::sf::OutArray<system::RegisteredDevice>(
                                                outValues, static_cast<size_t>(count))));
    *pOutCount = static_cast<int>(gotCount);
    NN_RESULT_SUCCESS;
}

::nn::Result BindConnectionTriggerTimeoutEvent(nn::os::SystemEventType* pSystemEvent, nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    auto handle = ::nn::sf::NativeHandle();
    NN_RESULT_DO(pProxy->AcquireConnectionTriggerTimeoutEvent(&handle));

    ::nn::os::AttachReadableHandleToSystemEvent(pSystemEvent,
                                                handle.GetOsHandle(),
                                                handle.IsManaged(),
                                                clearMode);

    // ハンドルの管理権を放棄
    handle.Detach();

    NN_RESULT_SUCCESS;
}

::nn::Result SendConnectionTrigger(nn::bluetooth::Address address) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->SendConnectionTrigger(address));

    NN_RESULT_SUCCESS;
}

::nn::Result BindDeviceRegisteredEventForControllerSupport(nn::os::SystemEventType* pSystemEvent, nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    auto handle = ::nn::sf::NativeHandle();
    NN_RESULT_DO(pProxy->AcquireDeviceRegisteredEventForControllerSupport(&handle));

    ::nn::os::AttachReadableHandleToSystemEvent(pSystemEvent,
                                                handle.GetOsHandle(),
                                                handle.IsManaged(),
                                                clearMode);

    // ハンドルの管理権を放棄
    handle.Detach();

    NN_RESULT_SUCCESS;
}

int GetAllowedBluetoothLinksCount() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidSystemServer>();
    CreateHidSystemServerProxy(&pProxy);

    int64_t outCount;
    if (pProxy->GetAllowedBluetoothLinksCount(&outCount).IsFailure())
    {
        return 0;
    }
    return static_cast<int>(outCount);
}

::nn::Result AddRegisteredDevice(const system::RegisteredDevice& device) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidDebugServer>();
    CreateHidDebugServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->AddRegisteredDevice(device));
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
