﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/sf/sf_Types.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/xcd/xcd_BleDevice.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_ActivationCount.h"
#include "hid_AppletResourceManager.h"
#include "hid_PalmaManager.h"

namespace nn { namespace hid { namespace detail {

//!< PalmaManager の配列の定義です
typedef PalmaManager PalmaManagerArray[::nn::xcd::BleDeviceCountMax];

//!< Palma に関するリソースを管理するクラスです。
class PalmaResourceManager final
{
private:
    //!< Palma の Activation された回数
    ActivationCount m_ActivationCount;

    //!< Palma 操作排他用のミューテックス
    ::nn::os::Mutex m_PalmaMutex;

    //!< Palma 毎のリソースを管理するマネージャの配列
    PalmaManagerArray m_PalmaManagers;

    //!< アプレットリソースマネージャ
    AppletResourceManager* m_pAppletResourceManager;

    //!< アプレットリソースマネージャのミューテックス
    ::nn::os::SdkRecursiveMutex* m_pAppletResourceManagerMutex;

public:
    PalmaResourceManager() NN_NOEXCEPT;

    ~PalmaResourceManager() NN_NOEXCEPT;

    //!< PalmaResourceManager をアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< PalmaResourceManager を非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< アプレットリソースマネージャを設定します。
    void SetAppletResourceManager(
        AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
    ) NN_NOEXCEPT;

    //!< 新たに接続されたデバイスをアタッチします。
    void Attach(const nn::hid::NpadIdType& npadId,
                nn::xcd::BleConnectionHandle handle) NN_NOEXCEPT;

    //!< 切断されたデバイスを処理します。
    void Detach(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT;

    //!< Focus されている Aruid に変更があった際に必要な処理を行います。
    void UpdateOnAruidSwitch() NN_NOEXCEPT;

    //!< Aruid の登録の解除があった際に必要な処理を行います。
    void UnregisterAppletResourceUserId(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< TransferMemoryType を取得します。
    ::nn::Result GetTransferMemoryType(::nn::os::TransferMemoryType** pTransferMemory, PalmaConnectionHandle handle) NN_NOEXCEPT;

    void SetTransferMemory(PalmaConnectionHandle handle, void* pMappedAddress, uint64_t size) NN_NOEXCEPT;

    //!< NpadIdType から PalmaConnectionHandle を取得します。
    ::nn::Result GetPalmaConnectionHandle(PalmaConnectionHandle* pOutValue, NpadIdType npadId, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Palma の固有機能を初期化します。
    ::nn::Result Initialize(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< イベントをバインドします。
    ::nn::Result AcquireOperationCompleteEventHandle(::nn::os::NativeHandle* pOutValue, PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma デバイスに対しての各操作に対する実行結果を取得します。
    ::nn::Result GetPalmaOperationInfo(PalmaOperationType* pOutValue, const nn::sf::OutBuffer& outBuffer, PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma デバイスに対しての各操作に対する実行結果の Result を返します。
    ::nn::Result GetPalmaOperationResult(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma デバイス演出の再生を指示します。
    ::nn::Result PlayPalmaActivity(PalmaConnectionHandle handle, uint16_t index) NN_NOEXCEPT;

    //!< Palma デバイスの FrModeType (Friend Buddy) を設定します。
    ::nn::Result SetFrModeType(PalmaConnectionHandle handle, uint64_t type) NN_NOEXCEPT;

    //!< Palma デバイスの StepCount を読み出します。
    ::nn::Result ReadStep(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma デバイスの StepCount 機能の有効/無効を設定します。
    ::nn::Result EnableStepCounter(PalmaConnectionHandle handle, bool isEnabled) NN_NOEXCEPT;

    //!< Palma デバイスの StepCount をゼロクリアします。
    ::nn::Result ResetStep(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma デバイスの ApplicationSection の指定の領域からデータを読み出します。
    ::nn::Result ReadApplicationSection(PalmaConnectionHandle handle, uint64_t address, uint64_t size) NN_NOEXCEPT;

    //!< Palma デバイスの ApplicationSection の指定の領域へデータを書き込みます。
    ::nn::Result WriteApplicationSection(PalmaConnectionHandle handle, uint64_t address, uint64_t size, PalmaApplicationSectionAccessBuffer buffer) NN_NOEXCEPT;

    //!< Palma ユニークコードを読み出します。
    ::nn::Result ReadPalmaUniqueCode(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma ユニークコードを無効化します。以降、無効化した Palma デバイスからは読み出しできなくなります。
    ::nn::Result SetPalmaUniqueCodeInvalid(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Activity Entry を Palma の指定した Index に書き込みます。
    ::nn::Result WritePalmaActivityEntry(PalmaConnectionHandle handle, uint64_t index, uint64_t ledIndex, uint64_t waveSet, uint64_t waveIndex) NN_NOEXCEPT;

    //!< RGB LED パターンを Palma の指定した Index に書き込みます。
    ::nn::Result WritePalmaRgbLedPatternEntry(PalmaConnectionHandle handle, uint64_t index, const char* buffer, size_t size) NN_NOEXCEPT;

    //!< 音源データを Palma の指定した Index に書き込みます。
    ::nn::Result WritePalmaWaveEntry(PalmaConnectionHandle handle, uint64_t waveSet, uint64_t index, size_t size) NN_NOEXCEPT;

    //!< Palma の ActivityDataBase のデータベース識別バージョンを設定します。
    ::nn::Result SetPalmaDataBaseIdentificationVersion(PalmaConnectionHandle handle, int32_t version) NN_NOEXCEPT;

    //!< Palma の ActivityDataBase のデータベース識別バージョンを取得します。
    ::nn::Result GetPalmaDataBaseIdentificationVersion(PalmaConnectionHandle handle) NN_NOEXCEPT;

    //!< Palma の Feature Set の機能のサスペンド状態を制御します。
    ::nn::Result SuspendPalmaFeature(PalmaConnectionHandle handle, PalmaFeatureSet suspendFeatureSet) NN_NOEXCEPT;

    //!< Palma の Play Log を読み出します。
    ::nn::Result ReadPalmaPlayLog(PalmaConnectionHandle handle, uint16_t index) NN_NOEXCEPT;

    //!< Palma の Play Log をゼロクリアします。
    ::nn::Result ResetPalmaPlayLog(PalmaConnectionHandle handle, uint16_t index) NN_NOEXCEPT;

    //!< 全ての Palma のスキャンの有効無効を切り替える
    void SetIsPalmaAllConnectable(const ::nn::applet::AppletResourceUserId& aruid, bool connectable) NN_NOEXCEPT;

    //!< 登録済みの Palma のスキャンの有効無効を切り替える
    void SetIsPalmaPairedConnectable(bool connectable) NN_NOEXCEPT;

    //!< 指定されたデバイスとペアリングする
    ::nn::Result PairPalma(const PalmaConnectionHandle& handle) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
