﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_SixAxisSensor.h>

#include "hid_ActivationCount.h"
#include "hid_ConsoleSixAxisSensorManager.h"
#include "hid_NpadAbstractedPadHolder.h"
#include "hid_NpadCommonResourceHolder.h"
#include "hid_NpadDeviceManager.h"
#include "hid_SixAxisSensorAppletSettingManager.h"
#include "hid_SixAxisSensorXcdDriver.h"


namespace nn { namespace hid { namespace detail {

//!< Npad に対して対応づけられる SixAxisSensor のハンドル数の Max
const int SixAxisSensorHandlePerNpadCountMax = 6;

//!< Xcd デバイスの設定値です。
struct XcdDeviceConfiguration
{
    IAbstractedPad* pPad; // AbstractedPad へのポインタ
    bool isAvailable; // 利用可能か

    void Reset() NN_NOEXCEPT
    {
        pPad = nullptr;
        isAvailable = false;
    }
};

//!< 6軸センサーの制御/状態更新を担うクラスです。
class NpadSixAxisSensorController final
{
    NN_DISALLOW_COPY(NpadSixAxisSensorController);
    NN_DISALLOW_MOVE(NpadSixAxisSensorController);

private:
    //!< Npad が共通で扱う hid のサービスを保持するクラス
    NpadCommonResourceHolder* m_pCommonResourceHolder;

    //!< Npad 毎に割り当てられる AbstractedPad を保持するクラス
    NpadAbstractedPadHolder* m_pAbstractedPadHolder;

    //!< Npad の状態管理を担うマネージャー
    NpadDeviceManager* m_pDeviceManager;

    //!< SixAxisSensor-Xcd ドライバ
    SixAxisSensorXcdDriver* m_pSixAxisSensorXcdDriver[SixAxisSensorIdxCountMax];

    //!< SixAxisSensor の設定値マネージャ
    SixAxisSensorAppletSettingManager* m_pSixAxisSensorAppletSettingManager;

    //!< ConsoleSixAxisSensor マネージャ
    ConsoleSixAxisSensorManager* m_pConsoleSixAxisSensorManager;

    //!< アクティブ化された回数
    ActivationCount m_NpadActivationCount;

    //!< Xcd のデバイスの設定値です。
    XcdDeviceConfiguration m_XcdDeviceConfigurations[SixAxisSensorIdxCountMax];

    //!< 1回のReadで更新される SixAxisSensor のサンプルの最大数
    static const int SixAxisSensorSamplePerReadCountMax = 3;

    //!< SixAxisSensor の入力状態の Index
    enum SixAxisSensorStateIndex
    {
        SixAxisSensorStateIndex_FullKey = 0,
        SixAxisSensorStateIndex_Handheld,
        SixAxisSensorStateIndex_JoyDualLeft,
        SixAxisSensorStateIndex_JoyDualRight,
        SixAxisSensorStateIndex_JoyLeft,
        SixAxisSensorStateIndex_JoyRight,
    };

    //!< FullKey 操作形態が参照する 6軸センサー識別子番号
    SixAxisSensorIdx m_FullKeySixAxisSensorIdx;

    //!< Handheld 操作形態が参照する 6軸センサー識別子番号
    SixAxisSensorIdx m_HandheldSixAxisSensorIdx;

    //!< Npad Handheld Activation Mode
    NpadHandheldActivationMode m_NpadHandheldActivationMode;

    //!< 最新の 6軸センサーの入力状態
    SixAxisSensorState m_SixAxisSensorState[SixAxisSensorHandlePerNpadCountMax][SixAxisSensorSamplePerReadCountMax];

    //!< 最新の 6軸センサーのサンプル数
    int m_SixAxisSensorCount[SixAxisSensorHandlePerNpadCountMax];

    //!< 前回の xcd のサンプリング番号
    int64_t m_SixAxisSensorLastSamplingNumbers[SixAxisSensorHandlePerNpadCountMax];

    //!< 6軸センサーのサンプリング番号
    int64_t m_SixAxisSensorSamplingNumbers[SixAxisSensorHandlePerNpadCountMax];

public:
    NpadSixAxisSensorController() NN_NOEXCEPT;

    ~NpadSixAxisSensorController() NN_NOEXCEPT;

    //!< NpadCommonResourceHolder を設定します。
    void SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT;

    //!< NpadAbstractedPadHolder を設定します。
    void SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT;

    //!< NpadDeviceManager を設定します。
    void SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT;

    //!< SixAxisSensor-Xcd ドライバを設定します。
    void SetSixAxisSensorXcdDriver(SixAxisSensorXcdDriver* pDriver, SixAxisSensorIdx idx) NN_NOEXCEPT;

    //!< SixAxisSensor の設定値マネージャを設定します。
    void SetSixAxisSensorAppletSettingManager(SixAxisSensorAppletSettingManager* pManager) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor マネージャを設定します。
    void SetConsoleSixAxisSensorManager(ConsoleSixAxisSensorManager* pManager) NN_NOEXCEPT;

    //!< アクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< 非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< Npad のアプレットリソースの状態を保証します。
    ::nn::Result EnsureNpadAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< デバイスの内部状態を更新します
    void UpdateDeviceStates() NN_NOEXCEPT;

    //!< 入力状態を更新します。
    void UpdateInputStates() NN_NOEXCEPT;

    //!< 共有メモリの状態を更新します。
    void UpdateSharedMemory() NN_NOEXCEPT;

    //!< 特定の Aruid に対して共有メモリ内部の状態を更新します
    void UpdateSharedMemoryForTarget(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 設定値をアタッチします。
    void AttachSixAxisSensorSetting(const nn::hid::SixAxisSensorHandle& handle) NN_NOEXCEPT;

    //!< 切断時に内部ステートをクリアします。
    void ResetSixAxisSensorInternalState(const system::NpadDeviceTypeSet& disconnectedDeviceType) NN_NOEXCEPT;

    //!< 本体センサー入力が Handheld 操作形態として利用されているか。
    bool IsConsoleSixAxisSensorUsed() const NN_NOEXCEPT;

    //!< SixAxisSensorProcessor を返します。
    const SixAxisSensorProcessor* GetSixAxisSensorProcessor(const SixAxisSensorHandle& handle) const NN_NOEXCEPT;

private:
    //!< 6軸センサー の搭載状態を変更します。
    void UpdateSixAxisSensorAvailability() NN_NOEXCEPT;

    //!< コントローラーの割り当て変更に伴って、6軸センサー設定を更新する
    void UpdateSixAxisSensorSetting() NN_NOEXCEPT;

    //!< 入力状態を 6軸センサーの状態を更新します。
    void UpdateSixAxisSensorState() NN_NOEXCEPT;

    //!< 操作形態ごとに 6 軸センサーの入力状態を更新します
    void UpdateSixAxisSensorStatePerStyle(const SixAxisSensorState* const pStates,
                                          const SixAxisSensorStateIndex& index,
                                          const int& count) NN_NOEXCEPT;

    //!< 共有メモリを処理します。
    void ProcessSharedMemory(
        void (*processor)(NpadSixAxisSensorController* that,
                          NpadSharedMemoryEntry* address,
                          ::nn::applet::AppletResourceUserId aruid,
                          bool enablesInput) NN_NOEXCEPT) NN_NOEXCEPT;

    //!< 指定した Aruid のものについてのみ共有メモリを処理します。
    void ProcessSharedMemoryForTargetAruid(
        ::nn::applet::AppletResourceUserId aruid,
        void (*processor)(NpadSixAxisSensorController* that,
                          NpadSharedMemoryEntry* address,
                          ::nn::applet::AppletResourceUserId aruid,
                          bool enablesInput) NN_NOEXCEPT) NN_NOEXCEPT;

    //!< 共有メモリの内容を更新します
    void UpdateSharedMemoryForInputs(NpadSharedMemoryEntry* address,
                            ::nn::applet::AppletResourceUserId aruid,
                            bool enablesInput,
                            bool updateOnlyWithEmpty) NN_NOEXCEPT;

    //!< 6軸センサーの共有メモリをスタイル毎に更新します。
    template <typename styleT, typename lifoT>
    void UpdateSharedMemoryForSixAxisSensor(NpadStyleSet style,
                            lifoT* pLifo,
                            SixAxisSensorState state[],
                            int count,
                            int64_t samplingNumberPerStyle,
                            bool enablesInput,
                            bool updateOnlyWithEmpty) NN_NOEXCEPT;

    //!< 6軸センサーハンドルが参照する識別番号を取得します。
    SixAxisSensorIdx GetSixAxisSensorIdx(const nn::hid::SixAxisSensorHandle& handle) const NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
