﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_AbstractedPadXcd.h"
#include "hid_CommonStateUtility.h"
#include "hid_NpadId.h"
#include "hid_NpadIndicatorPattern.h"
#include "hid_NpadInternalState.h"
#include "hid_NpadSixAxisSensorController.h"
#include "hid_NpadStateUtility.h"
#include "hid_Settings.h"

namespace nn { namespace hid { namespace detail {

NpadSixAxisSensorController::NpadSixAxisSensorController() NN_NOEXCEPT
    : m_pCommonResourceHolder(nullptr)
    , m_pAbstractedPadHolder(nullptr)
    , m_pDeviceManager(nullptr)
    , m_pSixAxisSensorAppletSettingManager(nullptr)
    , m_pConsoleSixAxisSensorManager(nullptr)
    , m_NpadActivationCount()
    , m_FullKeySixAxisSensorIdx(SixAxisSensorIdx_Center)
    , m_HandheldSixAxisSensorIdx(SixAxisSensorIdx_Left)
    , m_NpadHandheldActivationMode(NpadHandheldActivationMode_Dual)
{
    for (auto& sixAxisSensorStatePerStyle : m_SixAxisSensorState)
    {
        for(auto& sixAxisSensorState : sixAxisSensorStatePerStyle)
        sixAxisSensorState = SixAxisSensorState();
    }

    for (int i = 0; i < SixAxisSensorHandlePerNpadCountMax; i++)
    {
        m_SixAxisSensorLastSamplingNumbers[i] = 0;
        m_SixAxisSensorSamplingNumbers[i] = 0;
    }

    for (auto& configuration : m_XcdDeviceConfigurations)
    {
        configuration.Reset();
    }
}

NpadSixAxisSensorController::~NpadSixAxisSensorController() NN_NOEXCEPT { /* 何もしない */ }

void NpadSixAxisSensorController::SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pAbstractedPadHolder = pHolder;
}

void NpadSixAxisSensorController::SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(pHolder->IsInitialized());
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pCommonResourceHolder = pHolder;
}

void NpadSixAxisSensorController::SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);

    m_pDeviceManager = pHolder;
}

void NpadSixAxisSensorController::SetSixAxisSensorXcdDriver(SixAxisSensorXcdDriver* pDriver, SixAxisSensorIdx idx) NN_NOEXCEPT
{
    const int PositionIndex = static_cast<int>(idx);
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(PositionIndex < SixAxisSensorIdxCountMax);
    m_pSixAxisSensorXcdDriver[PositionIndex] = pDriver;
}

void NpadSixAxisSensorController::SetSixAxisSensorAppletSettingManager(SixAxisSensorAppletSettingManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pSixAxisSensorAppletSettingManager = pManager;
}

void NpadSixAxisSensorController::SetConsoleSixAxisSensorManager(ConsoleSixAxisSensorManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pConsoleSixAxisSensorManager = pManager;
}

::nn::Result NpadSixAxisSensorController::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pDeviceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pSixAxisSensorXcdDriver[0]);
    NN_SDK_REQUIRES_NOT_NULL(m_pSixAxisSensorAppletSettingManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pConsoleSixAxisSensorManager);

    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsMax(),
                           ResultNpadActivationUpperLimitOver());

    if (m_NpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
        // 6軸センサーの入力状態を更新
        UpdateSixAxisSensorState();
    }

    ++m_NpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadSixAxisSensorController::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsZero(),
                           ResultNpadDeactivationLowerLimitOver());

    --m_NpadActivationCount;

    if (m_NpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
    }

    NN_RESULT_SUCCESS;
}

::nn::Result NpadSixAxisSensorController::EnsureNpadAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    this->ProcessSharedMemoryForTargetAruid(
                              aruid,
                              [](NpadSixAxisSensorController* that,
                                 NpadSharedMemoryEntry* address,
                                 ::nn::applet::AppletResourceUserId haystack,
                                 bool enablesInput) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        that->UpdateSharedMemoryForInputs(
            address,
            haystack,
            enablesInput,
            true);
    });

    NN_RESULT_SUCCESS;
}

void NpadSixAxisSensorController::AttachSixAxisSensorSetting(const nn::hid::SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    SixAxisSensorSetting* pSetting = nullptr;
    m_pSixAxisSensorAppletSettingManager->GetSixAxisSensorSetting(&pSetting,
                                                                  m_pCommonResourceHolder->GetAppletResourceManager().GetAppletResourceUserId(),
                                                                  handle);

    const int Position = GetSixAxisSensorIdx(handle);
    m_pSixAxisSensorXcdDriver[Position]->SetSixAxisSensorSetting(pSetting);
}

void NpadSixAxisSensorController::ResetSixAxisSensorInternalState(const system::NpadDeviceTypeSet& disconnectedDeviceType) NN_NOEXCEPT
{
    SixAxisSensorIdx sixAxisSensorIdx;

    // 切断されたデバイスタイプに対応した識別子を取得
    if (disconnectedDeviceType.Test<system::NpadDeviceType::JoyConLeft>() == true)
    {
        sixAxisSensorIdx = SixAxisSensorIdx_Left;
    }
    else if (disconnectedDeviceType.Test<system::NpadDeviceType::JoyConRight>() == true)
    {
        sixAxisSensorIdx = SixAxisSensorIdx_Right;
    }
    else if (disconnectedDeviceType.Test<system::NpadDeviceType::FullKeyController>() == true)
    {
        sixAxisSensorIdx = m_FullKeySixAxisSensorIdx;
    }
    else if (disconnectedDeviceType.Test<system::NpadDeviceType::HandheldJoyLeft>() == true)
    {
        sixAxisSensorIdx = SixAxisSensorIdx_Left;
    }
    else if (disconnectedDeviceType.Test<system::NpadDeviceType::HandheldJoyRight>() == true)
    {
        sixAxisSensorIdx = SixAxisSensorIdx_Right;
    }
    else
    {
        return;
    }

    // 該当するサンプリング番号をクリア
    m_SixAxisSensorLastSamplingNumbers[sixAxisSensorIdx] = 0;
}

bool NpadSixAxisSensorController::IsConsoleSixAxisSensorUsed() const NN_NOEXCEPT
{
    auto handheldActivationMode = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy().GetNpadHandheldActivationMode();
    const auto NpadId = m_pDeviceManager->GetNpadIdType();

    return (handheldActivationMode == NpadHandheldActivationMode_None &&
            m_HandheldSixAxisSensorIdx == SixAxisSensorIdx_Center &&
            NpadId == ::nn::hid::NpadId::Handheld);
}

const SixAxisSensorProcessor* NpadSixAxisSensorController::GetSixAxisSensorProcessor(const SixAxisSensorHandle& handle) const NN_NOEXCEPT
{
    int sixAxisSensorIdx = GetSixAxisSensorHandleSixAxisSensorIdx(handle);

    switch (GetSixAxisSensorHandleNpadStyleIndex(handle))
    {
    case NpadStyleFullKey::Index:
        sixAxisSensorIdx = m_FullKeySixAxisSensorIdx;
        break;
    case NpadStyleHandheld::Index:
        sixAxisSensorIdx = m_HandheldSixAxisSensorIdx;

        if (IsConsoleSixAxisSensorUsed())
        {
            return &(m_pConsoleSixAxisSensorManager->GetSixAxisSensorProcessor());
        }
        break;
    default:
        // 何もしない
        break;
    }

    return &(m_pSixAxisSensorXcdDriver[sixAxisSensorIdx]->GetSixAxisSensorProcessorInstance());
}

void NpadSixAxisSensorController::UpdateDeviceStates() NN_NOEXCEPT
{
    this->UpdateSixAxisSensorAvailability();
    this->UpdateSixAxisSensorSetting();
}

void NpadSixAxisSensorController::UpdateInputStates() NN_NOEXCEPT
{
    if (!m_NpadActivationCount.IsZero())
    {
        this->UpdateSixAxisSensorState();
    }
}

void NpadSixAxisSensorController::UpdateSharedMemory() NN_NOEXCEPT
{
    this->ProcessSharedMemory([](NpadSixAxisSensorController* that,
                                 NpadSharedMemoryEntry* address,
                                 ::nn::applet::AppletResourceUserId aruid,
                                 bool enablesInput) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);

        that->UpdateSharedMemoryForInputs(
            address,
            aruid,
            enablesInput,
            false);
    });

    // サンプル数を初期化
    for (auto& count : m_SixAxisSensorCount)
    {
        count = 0;
    }
}

void NpadSixAxisSensorController::UpdateSharedMemoryForTarget(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    this->ProcessSharedMemoryForTargetAruid(
                              aruid,
                              [](NpadSixAxisSensorController* that,
                                 NpadSharedMemoryEntry* address,
                                 ::nn::applet::AppletResourceUserId aruid,
                                 bool enablesInput) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(that);
        NN_SDK_REQUIRES_NOT_NULL(address);
        NN_UNUSED(enablesInput);

        that->UpdateSharedMemoryForInputs(
            address,
            aruid,
            enablesInput,
            false);
    });

    // サンプル数を初期化
    for (auto& count : m_SixAxisSensorCount)
    {
        count = 0;
    }
}

void NpadSixAxisSensorController::ProcessSharedMemory(
    void (*processor)(NpadSixAxisSensorController* that,
                      NpadSharedMemoryEntry* address,
                      ::nn::applet::AppletResourceUserId aruid,
                      bool enablesInput) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);

    const auto index =
        static_cast<size_t>(GetIndexFromNpadIdType(m_pDeviceManager->GetNpadIdType()));

    for (const AppletResourceEntry& entry :
             m_pCommonResourceHolder->GetAppletResourceManager().GetAppletResourceEntries())
    {
        if (entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);
            NN_SDK_ASSERT_LESS(
                index,
                ::std::extent<
                    decltype(entry.address->npad.entries)>::value);
            NN_SDK_ASSERT_NOT_NULL(&entry.address->npad.entries[index]);
            processor(this,
                &entry.address->npad.entries[index],
                entry.aruid,
                entry.flags.Test<AppletResourceFlag::EnablesSixAxisSensor>());
        }
    }
}

void NpadSixAxisSensorController::ProcessSharedMemoryForTargetAruid(
    ::nn::applet::AppletResourceUserId aruid,
    void (*processor)(NpadSixAxisSensorController* that,
                      NpadSharedMemoryEntry* address,
                      ::nn::applet::AppletResourceUserId aruid,
                      bool enablesInput) NN_NOEXCEPT) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(processor);
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);

    const auto index =
        static_cast<size_t>(GetIndexFromNpadIdType(m_pDeviceManager->GetNpadIdType()));

    for (const AppletResourceEntry& entry :
             m_pCommonResourceHolder->GetAppletResourceManager().GetAppletResourceEntries())
    {
        if (entry.flags.Test<AppletResourceFlag::IsAvailable>() &&
            entry.aruid == aruid)
        {
            NN_SDK_ASSERT_NOT_NULL(entry.address);
            NN_SDK_ASSERT_LESS(
                index,
                ::std::extent<
                    decltype(entry.address->npad.entries)>::value);
            NN_SDK_ASSERT_NOT_NULL(&entry.address->npad.entries[index]);
            processor(this,
                &entry.address->npad.entries[index],
                entry.aruid,
                entry.flags.Test<AppletResourceFlag::EnablesSixAxisSensor>());

            return;
        }
    }
}

void NpadSixAxisSensorController::UpdateSharedMemoryForInputs(NpadSharedMemoryEntry* address,
                                     ::nn::applet::AppletResourceUserId aruid,
                                     bool enablesInput,
                                     bool updateOnlyWithEmpty) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(address);

    // スタイルを更新
    NpadInternalState& lifo = ::nn::util::Get(address->internalState);
    auto style = m_pDeviceManager->GetAvailableNpadStyleSet(aruid);

    if (style.Test<NpadStylePalma>() == true)
    {
        UpdateSharedMemoryForSixAxisSensor<NpadStylePalma>(style,
                                                             lifo.GetFullKeySixAxisSensorLifo(),
                                                             m_SixAxisSensorState[SixAxisSensorStateIndex_FullKey],
                                                             m_SixAxisSensorCount[SixAxisSensorStateIndex_FullKey],
                                                             m_SixAxisSensorSamplingNumbers[SixAxisSensorStateIndex_FullKey],
                                                             enablesInput, updateOnlyWithEmpty);
    }
    else
    {
        UpdateSharedMemoryForSixAxisSensor<NpadStyleFullKey>(style,
                                                             lifo.GetFullKeySixAxisSensorLifo(),
                                                             m_SixAxisSensorState[SixAxisSensorStateIndex_FullKey],
                                                             m_SixAxisSensorCount[SixAxisSensorStateIndex_FullKey],
                                                             m_SixAxisSensorSamplingNumbers[SixAxisSensorStateIndex_FullKey],
                                                             enablesInput, updateOnlyWithEmpty);
    }
    UpdateSharedMemoryForSixAxisSensor<NpadStyleHandheld>(style,
                                                       lifo.GetHandheldSixAxisSensorLifo(),
                                                       m_SixAxisSensorState[SixAxisSensorStateIndex_Handheld],
                                                       m_SixAxisSensorCount[SixAxisSensorStateIndex_Handheld],
                                                       m_SixAxisSensorSamplingNumbers[SixAxisSensorStateIndex_Handheld],
                                                       enablesInput, updateOnlyWithEmpty);
    UpdateSharedMemoryForSixAxisSensor<NpadStyleJoyDual>(style,
                                                      lifo.GetJoyDualLeftSixAxisSensorLifo(),
                                                      m_SixAxisSensorState[SixAxisSensorStateIndex_JoyDualLeft],
                                                      m_SixAxisSensorCount[SixAxisSensorStateIndex_JoyDualLeft],
                                                      m_SixAxisSensorSamplingNumbers[SixAxisSensorStateIndex_JoyDualLeft],
                                                      enablesInput, updateOnlyWithEmpty);
    UpdateSharedMemoryForSixAxisSensor<NpadStyleJoyDual>(style,
                                                      lifo.GetJoyDualRightSixAxisSensorLifo(),
                                                      m_SixAxisSensorState[SixAxisSensorStateIndex_JoyDualRight],
                                                      m_SixAxisSensorCount[SixAxisSensorStateIndex_JoyDualRight],
                                                      m_SixAxisSensorSamplingNumbers[SixAxisSensorStateIndex_JoyDualRight],
                                                      enablesInput, updateOnlyWithEmpty);
    UpdateSharedMemoryForSixAxisSensor<NpadStyleJoyLeft>(style,
                                                      lifo.GetJoyLeftSixAxisSensorLifo(),
                                                      m_SixAxisSensorState[SixAxisSensorStateIndex_JoyLeft],
                                                      m_SixAxisSensorCount[SixAxisSensorStateIndex_JoyLeft],
                                                      m_SixAxisSensorSamplingNumbers[SixAxisSensorStateIndex_JoyLeft],
                                                      enablesInput, updateOnlyWithEmpty);
    UpdateSharedMemoryForSixAxisSensor<NpadStyleJoyRight>(style,
                                                      lifo.GetJoyRightSixAxisSensorLifo(),
                                                      m_SixAxisSensorState[SixAxisSensorStateIndex_JoyRight],
                                                      m_SixAxisSensorCount[SixAxisSensorStateIndex_JoyRight],
                                                      m_SixAxisSensorSamplingNumbers[SixAxisSensorStateIndex_JoyRight],
                                                      enablesInput, updateOnlyWithEmpty);
}


template <typename styleT, typename lifoT>
void NpadSixAxisSensorController::UpdateSharedMemoryForSixAxisSensor(NpadStyleSet style,
                                                  lifoT* pLifo,
                                                  SixAxisSensorState state[],
                                                  int count,
                                                  int64_t samplingNumberPerStyle,
                                                  bool enablesInput,
                                                  bool updateOnlyWithEmpty) NN_NOEXCEPT
{
    if (updateOnlyWithEmpty == false || pLifo->IsEmpty())
    {
        if (count == 0 && pLifo->IsEmpty() && updateOnlyWithEmpty == true)
        {
            // サンプリング処理が全く行われていない場合は無入力状態を1つセットする
            auto nonInputState = SixAxisSensorState();
            nonInputState.samplingNumber = samplingNumberPerStyle;
            pLifo->Append(nonInputState);
            return;
        }
        for (int i = count - 1; i >= 0; i--)
        {
            auto samplingNumber = state[i].samplingNumber;
            if (style.Test<styleT>() == true)
            {
                if (enablesInput)
                {
                    pLifo->Append(state[i]);
                }
                else
                {
                    // 接続状態だけは同期しつつ、入力は無入力とする
                    auto nonInputState = SixAxisSensorState();
                    nonInputState.samplingNumber = samplingNumber;
                    nonInputState.attributes = state[i].attributes;
                    pLifo->Append(nonInputState);
                }
            }
            else
            {
                // スタイルが有効になっていない場合は完全無入力状態
                auto nonInputState = SixAxisSensorState();
                nonInputState.samplingNumber = samplingNumber;
                pLifo->Append(nonInputState);
            }
        }
    }
}

void NpadSixAxisSensorController::UpdateSixAxisSensorAvailability() NN_NOEXCEPT
{
    XcdDeviceConfiguration deviceConfigurations[NN_ARRAY_SIZE(m_pSixAxisSensorXcdDriver)];

    // deviceConfigurations の初期化
    for (auto& configuration : deviceConfigurations)
    {
        configuration.Reset();
    }

    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    // deviceConfigurations の状態取得
    for (int i = 0; i < count; i++)
    {
        if (pPads[i]->IsConnected() &&
            (pPads[i]->GetType() == AbstractedPadType_Xcd || pPads[i]->GetType() == AbstractedPadType_Palma))
        {
            if (pPads[i]->GetFeatureSet().Test<AbstractedPadFeature::SixAxisSensorLeft>())
            {
                if (m_pDeviceManager->IsStyleActiveForDevice(pPads[i]->GetDeviceType()))
                {
                    deviceConfigurations[SixAxisSensorIdx_Left].isAvailable = true;
                    deviceConfigurations[SixAxisSensorIdx_Left].pPad = pPads[i];
                }
            }
            if (pPads[i]->GetFeatureSet().Test<AbstractedPadFeature::SixAxisSensorRight>())
            {
                if (m_pDeviceManager->IsStyleActiveForDevice(pPads[i]->GetDeviceType()))
                {
                    deviceConfigurations[SixAxisSensorIdx_Right].isAvailable = true;
                    deviceConfigurations[SixAxisSensorIdx_Right].pPad = pPads[i];
                }
            }
            if (pPads[i]->GetFeatureSet().Test<AbstractedPadFeature::SixAxisSensorCenter>())
            {
                if (m_pDeviceManager->IsStyleActiveForDevice(pPads[i]->GetDeviceType()))
                {
                    deviceConfigurations[SixAxisSensorIdx_Center].isAvailable = true;
                    deviceConfigurations[SixAxisSensorIdx_Center].pPad = pPads[i];
                }
            }
        }
    }

    // deviceConfigurations の状態遷移に応じた更新処理
    for (int i = 0; i < NN_ARRAY_SIZE(m_pSixAxisSensorXcdDriver); i++)
    {
        if(deviceConfigurations[i].pPad != m_XcdDeviceConfigurations[i].pPad)
        {
            if (deviceConfigurations[i].isAvailable == true)
            {
                SixAxisSensorXcdConfig sixAxisSensorConfig;
                // SixAxisSensorXcdDriver を有効にする
                sixAxisSensorConfig.pPad = deviceConfigurations[i].pPad;

                m_pSixAxisSensorXcdDriver[i]->ActivateSixAxisSensor(sixAxisSensorConfig);
            }
            else
            {
                m_pSixAxisSensorXcdDriver[i]->DeactivateSixAxisSensor();
            }

            m_XcdDeviceConfigurations[i] = deviceConfigurations[i];
        }
    }
}

void NpadSixAxisSensorController::UpdateSixAxisSensorSetting() NN_NOEXCEPT
{
    // FullKey 操作形態に追加する入力状態を選択
    auto lastFullKeySixAxisSensorIdx = m_FullKeySixAxisSensorIdx;
    auto deviceType = m_pDeviceManager->GetNpadDeviceTypeSet();

    if (IsJoyDual(deviceType))
    {
        m_FullKeySixAxisSensorIdx = SixAxisSensorIdx_Right;
    }
    else
    {
        m_FullKeySixAxisSensorIdx = SixAxisSensorIdx_Center;
    }

    // 参照先のジョイコンが切り替わった時は SamplingNumber の連続性を担保する
    if (lastFullKeySixAxisSensorIdx != m_FullKeySixAxisSensorIdx)
    {
        SixAxisSensorState state;
        m_pSixAxisSensorXcdDriver[m_FullKeySixAxisSensorIdx]->GetSixAxisSensorStates(&state, 1);
    }

    // Handheld 操作形態に追加する入力状態を選択
    auto lastHandheldSixAxisSensorIdx = m_HandheldSixAxisSensorIdx;

    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto abstractedPadCount = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));
    system::DeviceTypeSet sensorAvailableDeviceType;
    sensorAvailableDeviceType.Reset();

    // deviceConfigurations の状態取得
    for (int i = 0; i < abstractedPadCount; i++)
    {
        if ((pPads[i]->GetFeatureSet() & AbstractedPadFeature::GetSixAxisSensorMask()).IsAnyOn())
        {
            sensorAvailableDeviceType |= pPads[i]->GetDeviceType();
        }

    }

    if (sensorAvailableDeviceType.Test<system::NpadDeviceType::HandheldJoyLeft>() == false &&
        sensorAvailableDeviceType.Test<system::NpadDeviceType::HandheldJoyRight>() == true)
    {
        m_HandheldSixAxisSensorIdx = SixAxisSensorIdx_Right;
    }
    else if (sensorAvailableDeviceType.Test<system::NpadDeviceType::HandheldJoyLeft>() == true &&
             sensorAvailableDeviceType.Test<system::NpadDeviceType::HandheldJoyRight>() == false)
    {
        m_HandheldSixAxisSensorIdx = SixAxisSensorIdx_Left;
    }
    else if(sensorAvailableDeviceType.Test<system::NpadDeviceType::HandheldJoyLeft>() == false &&
            sensorAvailableDeviceType.Test<system::NpadDeviceType::HandheldJoyRight>() == false)
    {
        m_HandheldSixAxisSensorIdx = SixAxisSensorIdx_Center;
    }
    else
    {

        // デフォルトは左コン
        m_HandheldSixAxisSensorIdx = SixAxisSensorIdx_Left;
    }

    // 参照先のジョイコンが切り替わった時は SamplingNumber の連続性を担保する
    if (lastHandheldSixAxisSensorIdx != m_HandheldSixAxisSensorIdx)
    {
        SixAxisSensorState state;
        m_pSixAxisSensorXcdDriver[m_HandheldSixAxisSensorIdx]->GetSixAxisSensorStates(&state, 1);
    }

    // 前回サンプル番号のクリア
    for (auto& samplingNumber : m_SixAxisSensorLastSamplingNumbers)
    {
        samplingNumber = 0;
    }

    // サンプリング番号のクリア
    m_SixAxisSensorLastSamplingNumbers[SixAxisSensorStateIndex_Handheld] = 0;

    auto style = m_pDeviceManager->GetAvailableNpadStyleSet(m_pCommonResourceHolder->GetAppletResourceManager().GetAppletResourceUserId());
    auto id = m_pDeviceManager->GetNpadIdType();

    SixAxisSensorHandle handles[SixAxisSensorHandlePerNpadCountMax];
    int count = 0;
    GetSixAxisSensorHandle(handles,
                           &count,
                           SixAxisSensorHandlePerNpadCountMax,
                           id,
                           style);

    // handle と対応する操作形態の設定値を SixAxisSensorProcessor にアタッチ
    for (int i = 0; i < count; i++)
    {
        AttachSixAxisSensorSetting(handles[i]);
    }
}

void NpadSixAxisSensorController::UpdateSixAxisSensorState() NN_NOEXCEPT
{
    // キャッシュされた入力状態をクリア
    for (int i = 0; i < SixAxisSensorHandlePerNpadCountMax; i++)
    {
        for (int j = 0; j < SixAxisSensorSamplePerReadCountMax; j++)
        {
            m_SixAxisSensorState[i][j] = ::nn::hid::SixAxisSensorState();
        }
    }

    // SixAxisSensorDriver から入力状態を取得
    SixAxisSensorState states[SixAxisSensorIdxCountMax][SixAxisSensorSamplePerReadCountMax] = {};
    int counts[SixAxisSensorIdxCountMax] = {};

    for (int i = 0; i < SixAxisSensorIdxCountMax; i++)
    {
        if (m_pSixAxisSensorXcdDriver[i] != nullptr)
        {
            m_pSixAxisSensorXcdDriver[i]->Sample();
            counts[i] = m_pSixAxisSensorXcdDriver[i]->GetSixAxisSensorStates(states[i],
                SixAxisSensorSamplePerReadCountMax);
        }
    }

    // 6軸センサーのデータを更新
    if (IsConsoleSixAxisSensorUsed())
    {
        const auto ConsoleSixAxisSensorIdx = SixAxisSensorIdx_Center;
        const auto StateCount = static_cast<int>(5) * 200 / 1000; // 200Hz にダウンサンプリング
        counts[ConsoleSixAxisSensorIdx] = m_pConsoleSixAxisSensorManager->GetSixAxisSensorState(states[ConsoleSixAxisSensorIdx],
                                                                                                StateCount);
    }

    UpdateSixAxisSensorStatePerStyle(states[SixAxisSensorIdx_Right],
                                     SixAxisSensorStateIndex_JoyDualRight,
                                     counts[SixAxisSensorIdx_Right]);

    UpdateSixAxisSensorStatePerStyle(states[SixAxisSensorIdx_Right],
                                     SixAxisSensorStateIndex_JoyRight,
                                     counts[SixAxisSensorIdx_Right]);

    UpdateSixAxisSensorStatePerStyle(states[SixAxisSensorIdx_Left],
                                     SixAxisSensorStateIndex_JoyDualLeft,
                                     counts[SixAxisSensorIdx_Left]);

    UpdateSixAxisSensorStatePerStyle(states[SixAxisSensorIdx_Left],
                                     SixAxisSensorStateIndex_JoyLeft,
                                     counts[SixAxisSensorIdx_Left]);

    UpdateSixAxisSensorStatePerStyle(states[m_HandheldSixAxisSensorIdx],
                                     SixAxisSensorStateIndex_Handheld,
                                     counts[m_HandheldSixAxisSensorIdx]);

    UpdateSixAxisSensorStatePerStyle(states[m_FullKeySixAxisSensorIdx],
                                     SixAxisSensorStateIndex_FullKey,
                                     counts[m_FullKeySixAxisSensorIdx]);

    for (int i = 0; i < SixAxisSensorHandlePerNpadCountMax; i++)
    {
        if (IsSixAxisSensorStateChanged(m_SixAxisSensorState[i][0]))
        {
            ::std::lock_guard<decltype(m_pCommonResourceHolder->GetInputDetectorManagerMutex())
                              > locker(m_pCommonResourceHolder->GetInputDetectorManagerMutex());
            m_pCommonResourceHolder->GetInputDetectorManager().Notify(::nn::hid::system::InputSourceId::Sensor::Mask);
            break;
        }
    }

    // NpadHandheldActivationMode の変化を ConsoleSixAxisSensorManager に通知
    const auto HandheldActivationMode = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy()
                                                                                             .GetNpadHandheldActivationMode();

    if (HandheldActivationMode != m_NpadHandheldActivationMode)
    {
        m_pConsoleSixAxisSensorManager->EnableNpadHandheldSampling(HandheldActivationMode == NpadHandheldActivationMode_None);
        m_NpadHandheldActivationMode = HandheldActivationMode;
    }
}

void NpadSixAxisSensorController::UpdateSixAxisSensorStatePerStyle(const SixAxisSensorState* const pStates,
                                                                   const SixAxisSensorStateIndex& index,
                                                                   const int& count) NN_NOEXCEPT
{
    if (m_SixAxisSensorCount[index] > 0)
    {
        // 共有メモリ書き込み前のデータがあるためスキップ
        return;
    }

    const int BufferCountMax = (SixAxisSensorSamplePerReadCountMax < count) ? SixAxisSensorSamplePerReadCountMax
                                                                            : count;

    // 更新されたサンプル数を計算
    int appendSampleCount = 0;
    for (int i = BufferCountMax - 1; i >= 0; i--)
    {
        // サンプル数の増加チェック
        if (pStates[i].samplingNumber <= m_SixAxisSensorLastSamplingNumbers[index])
        {
            continue;
        }

        m_SixAxisSensorLastSamplingNumbers[index] = pStates[i].samplingNumber;
        appendSampleCount++;
    }

    // 更新されたサンプル数だけ入力状態を更新
    for (int i = 0; i < appendSampleCount; i++)
    {
        m_SixAxisSensorState[index][appendSampleCount - 1 - i] = pStates[BufferCountMax - 1 - i];
        m_SixAxisSensorState[index][appendSampleCount - 1 - i].samplingNumber = ++m_SixAxisSensorSamplingNumbers[index];
        m_SixAxisSensorCount[index]++;
    }
}

SixAxisSensorIdx NpadSixAxisSensorController::GetSixAxisSensorIdx(const nn::hid::SixAxisSensorHandle& handle) const NN_NOEXCEPT
{
    const auto StyleIndex = GetSixAxisSensorHandleNpadStyleIndex(handle);
    SixAxisSensorIdx idx;

    switch (StyleIndex)
    {
    case NpadStyleFullKey::Index:
        idx = m_FullKeySixAxisSensorIdx;
        break;
    case NpadStyleHandheld::Index:
        idx = m_HandheldSixAxisSensorIdx;
        break;
    default:
        idx = static_cast<SixAxisSensorIdx>(GetSixAxisSensorHandleSixAxisSensorIdx(handle));
        break;
    }
    return idx;
}

}}} // namespace nn::hid::detail
