﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_Npad.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AbstractedPalmaPad.h"
#include "hid_NpadPalmaController.h"

namespace nn { namespace hid { namespace detail {

NpadPalmaController::NpadPalmaController() NN_NOEXCEPT
    : m_pAbstractedPadHolder(nullptr)
    , m_pDeviceManager(nullptr)
    , m_pPalmaResourceManager(nullptr)
    , m_NpadActivationCount()
    , m_XcdBleConnectionHandle(nn::xcd::BleInvalidConnectionHandle)
{
}

NpadPalmaController::~NpadPalmaController() NN_NOEXCEPT { /* 何もしない */ }

void NpadPalmaController::SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pAbstractedPadHolder = pHolder;
}

void NpadPalmaController::SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);

    m_pDeviceManager = pHolder;
}

void NpadPalmaController::SetPalmaResourceManager(PalmaResourceManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pPalmaResourceManager = pManager;
}

::nn::Result NpadPalmaController::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pDeviceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pPalmaResourceManager);

    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsMax(),
                           ResultNpadActivationUpperLimitOver());

    if (m_NpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
    }

    ++m_NpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadPalmaController::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsZero(),
                           ResultNpadDeactivationLowerLimitOver());

    --m_NpadActivationCount;

    if (m_NpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
    }

    NN_RESULT_SUCCESS;
}

void NpadPalmaController::UpdateDeviceStates() NN_NOEXCEPT
{
    auto oldHandle = m_XcdBleConnectionHandle;

    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    m_XcdBleConnectionHandle = nn::xcd::BleInvalidConnectionHandle;

    if (count == 0)
    {
        // 何も接続されていない
    }
    else
    {
        for (int i = 0; i < count; i++)
        {
            if (pPads[i]->IsConnected() &&
                pPads[i]->GetType() == AbstractedPadType_Palma)
            {
                if (m_pDeviceManager->IsStyleActiveForDevice(pPads[i]->GetDeviceType()))
                {
                    m_XcdBleConnectionHandle = reinterpret_cast<AbstractedPalmaPad*>(pPads[i])->GetXcdDeviceHandle();
                }
            }
        }
    }

    // Palma の状態に変化があった場合は通知
    if (oldHandle != m_XcdBleConnectionHandle)
    {
        if (m_XcdBleConnectionHandle == nn::xcd::BleInvalidConnectionHandle)
        {
            // 切断された
            m_pPalmaResourceManager->Detach(m_pDeviceManager->GetNpadIdType());
        }
        else
        {
            // 接続された
            if (m_XcdBleConnectionHandle != nn::xcd::BleInvalidConnectionHandle)
            {
                // 前回も別のデバイスが接続状態にあった場合は一度 Detach を挟む
                m_pPalmaResourceManager->Detach(m_pDeviceManager->GetNpadIdType());
            }
            m_pPalmaResourceManager->Attach(m_pDeviceManager->GetNpadIdType(), m_XcdBleConnectionHandle);
        }
    }
}

}}} // namespace nn::hid::detail
