﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>

#include "hid_ActivationCount.h"
#include "hid_CaptureButtonManager.h"
#include "hid_ExternalBusXcdDriver.h"
#include "hid_HomeButtonManager.h"
#include "hid_IrSensorXcdDriver.h"
#include "hid_NpadAbstractedPadHolder.h"
#include "hid_NpadDeviceManager.h"
#include "hid_NpadIndicatorController.h"
#include "hid_NpadInputStateUpdater.h"
#include "hid_NpadIrSensorController.h"
#include "hid_NpadNfcController.h"
#include "hid_NpadPalmaController.h"
#include "hid_NpadPowerInfoUpdater.h"
#include "hid_NpadRailAttachmentController.h"
#include "hid_NpadSixAxisSensorController.h"
#include "hid_NpadVibrationController.h"
#include "hid_SixAxisSensorXcdDriver.h"
#include "hid_VibratorGcDriver.h"
#include "hid_VibratorXcdDriver.h"

namespace nn { namespace hid { namespace detail {

//!< NpadIdType 単位のリソース管理を担うマネージャーです。
class NpadManager final
{
    NN_DISALLOW_COPY(NpadManager);
    NN_DISALLOW_MOVE(NpadManager);

private:
    //!< Npad が共通で扱う hid のサービスを保持するクラス
    NpadCommonResourceHolder* m_pCommonResourceHolder;

    //!< Npad 毎に割り当てられる AbstractedPad を保持するクラス
    NpadAbstractedPadHolder m_AbstractedPadHolder;

    //!< Npad の状態管理を担うマネージャー
    NpadDeviceManager m_DeviceManager;

    //!< インジケーターの制御を担うクラス
    NpadIndicatorController m_Indicator;

    //!< Npad の IrSensor 制御
    NpadIrSensorController m_IrSensor;

    //!< IrSensor の状態の制御を担うクラス
    NpadNfcController m_Nfc;

    //!< レール拡張デバイスの状態の制御を担うクラス
    NpadRailAttachmentController m_RailAttachment;

    //!< 振動デバイスの管理を担うクラス
    NpadVibrationController m_Vibration;

    //!< 6軸センサーの制御/状態更新を担うクラス
    NpadSixAxisSensorController m_SixAxisSensor;

    //!< ボタンやアナログスティックなどの入力状態の更新を担うクラス
    NpadInputStateUpdater m_InputStateUpdater;

    //!< 電源状態の更新を担うクラス
    NpadPowerInfoUpdater m_PowerInfoUpdater;

    //!< Palma の状態の制御を行うクラス
    NpadPalmaController m_Palma;

    //!< VibratorXcd ドライバ
    VibratorXcdDriver m_VibratorXcdDrivers[NpadVibratorCountPerPlayerMax];

    //!< VibratorGc ドライバ
    VibratorGcDriver m_VibratorGcDriver;

    //!< SixAxisSensorXcdDriver ドライバ
    SixAxisSensorXcdDriver m_SixAxisSensorXcdDrivers[SixAxisSensorHandlePerNpadCountMax];

    //!< IrSensor Xcd ドライバ
    IrSensorXcdDriver m_IrSensorXcdDriver;

    //!< ExternalBus Xcd ドライバ
    ExternalBusXcdDriver m_ExternalBusXcdDriver[JoyConRailAttachmentCountMaxPerNpad];

    //!< Npad がアクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< Switch Pro Controller のインタフェース種別
    system::InterfaceType m_InterfaceTypeOfFullKey;

public:
    NpadManager() NN_NOEXCEPT;

    ~NpadManager() NN_NOEXCEPT;

    //!< NpadManager を初期化します。
    void Initialize(NpadCommonResourceHolder* pHolder,
                    CaptureButtonManager* pCaptureButtonManager,
                    HomeButtonManager* pHomeButtonManager,
                    SixAxisSensorAppletSettingManager* pSixAxisSensorSettingsManager,
                    ConsoleSixAxisSensorManager* pConsoleSixAxisSensor,
                    GcAdapterDriver* pGcAdapterDriver,
                    PalmaResourceManager* pPalma) NN_NOEXCEPT;

    //!< NpadIdType を設定します。
    void SetNpadIdType(NpadIdType id) NN_NOEXCEPT;

    //!< マネージャをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< マネージャを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< Npad のアプレットリソースの状態を保証します。
    ::nn::Result EnsureAppletResource(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< NpadAbstractedPadHolder を取得します。
    NpadAbstractedPadHolder& GetAbstractedPadHolder() NN_NOEXCEPT;

    //!< NpadDeviceManager を取得します。
    NpadDeviceManager& GetDeviceManager() NN_NOEXCEPT;

    //!< ExternalBusXcdDriver を返します。
    ExternalBusXcdDriver* GetExternalBusXcdDriver(const int index) NN_NOEXCEPT;

    //!< NpadIndicatorController を取得します。
    NpadIndicatorController& GetIndicator() NN_NOEXCEPT;

    //!< NpadIrSensorController を取得します。
    NpadIrSensorController& GetIrSensor() NN_NOEXCEPT;

    //!< NpadNfcController を取得します。
    NpadNfcController& GetNfc() NN_NOEXCEPT;

    //!< NpadRailAttachmentController を取得します。
    NpadRailAttachmentController& GetRailAttachment() NN_NOEXCEPT;

    //!< NpadVibrationController を取得します。
    NpadVibrationController& GetVibration() NN_NOEXCEPT;

    //!< NpadSixAxisSensorController を取得します。
    NpadSixAxisSensorController& GetSixAxisSensor() NN_NOEXCEPT;

    //!< NpadInputStateUpdater を取得します。
    NpadInputStateUpdater& GetInputStateUpdater() NN_NOEXCEPT;

    //!< NpadPowerInfoUpdater を取得します。
    NpadPowerInfoUpdater& GetPowerInfoUpdater() NN_NOEXCEPT;

    //!< 指定されたハンドルに対応する VibratorXcdDriver を返します。
    VibratorXcdDriver* GetVibratorXcdDriver(const int index) NN_NOEXCEPT;

    //!< VibratorXcdDriver のリストを返します。
    void GetVibratorXcdDrivers(int* pOutCount, VibratorXcdDriver** outDrivers) NN_NOEXCEPT;

    //!< VibratorGcDriver を返します。
    VibratorGcDriver* GetVibratorGcDriver() NN_NOEXCEPT;

    //!< SixAxisSensorProcessor を返します。
    const SixAxisSensorProcessor* GetSixAxisSensorProcessor(const SixAxisSensorHandle& handle) const NN_NOEXCEPT;

    //!< IrSensorXcdDriver を返します。
    IrSensorXcdDriver* GetIrSensorXcdDriver() NN_NOEXCEPT;

    //!< NpadId を取得する
    NpadIdType GetNpadIdType() NN_NOEXCEPT;

    //!< 定常的に行う処理を行う
    void ProceedPeriodicEvent() NN_NOEXCEPT;

    //!< 定常的に更新が必要な、共有メモリの状態の更新を行う
    void UpdateSharedMemoryForPeriodicalStates() NN_NOEXCEPT;

    //!< 特定の Aruid に対して共有メモリ内部の状態を更新します
    void UpdateSharedMemoryForTarget(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 現在のデバイスの割り当て状態と有効な操作形態から、有効な操作形態とアプリ向けデバイス状態を更新する
    void UpdateNpad() NN_NOEXCEPT;

private:
};

//!< Npad の配列の定義です
typedef NpadManager NpadManagerArray[NpadIdCountMax];

//!< NpadManager を取得する
inline const NpadManager& GetNpadManager(const NpadManagerArray& array, const NpadIdType& id) NN_NOEXCEPT
{
    return array[GetIndexFromNpadIdType(id)];
}

inline NpadManager& GetNpadManager(NpadManagerArray& array, const NpadIdType& id) NN_NOEXCEPT
{
    return array[GetIndexFromNpadIdType(id)];
}

}}} // namespace nn::hid::detail
