﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_AppletResourceUserId.h"
#include "hid_HidDebugServer.h"
#include "hid_HidServer.h"
#include "hid_SharedMemoryAccessor.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_StaticObject.h"
#include "hid_NpadId.h"
#include "hid_NpadImpl.h"
#include "hid_NpadInternalState.h"
#include "hid_NpadRevisionTypes.h"
#include "hid_NpadSessionManager.h"
#include "hid_NpadStateUtility.h"
#include "hid_RingLifoHolder.h"
#include "hid_SixAxisSensor.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< Npad のセッションを扱うためのクラスです。
class NpadSharedMemoryAccessor final : public SharedMemoryAccessor
{
private:
    //!< Npad の LIFO
    ::std::atomic<NpadInternalState*> m_pInternalState[NpadIdCountMax];

    //!< ボタンの排他制御の方法
    ControlPadRestrictionType m_ControlPadRestrictionTypes[NpadStyleSet::StorageBitCount];

    //!< ボタンの排他制御の有無
    bool m_IsControlPadRestrictionEnabled[NpadStyleSet::StorageBitCount];

public:
    explicit NpadSharedMemoryAccessor() NN_NOEXCEPT
    {
        this->SetResultActivationUpperLimitOver(
            ::nn::hid::ResultNpadActivationUpperLimitOver());
        this->SetResultDeactivationLowerLimitOver(
            ::nn::hid::ResultNpadDeactivationLowerLimitOver());
        for (auto& pState : m_pInternalState)
        {
            pState = nullptr;
        }

        for (auto& type : m_ControlPadRestrictionTypes)
        {
            type = ControlPadRestrictionType_KeepUpAndLeft;
        }

        for (auto& enabled : m_IsControlPadRestrictionEnabled)
        {
            enabled = true;
        }
        // JoyDual, JoyLeft, NpadSystemExt はデフォルトで禁則処理なし
        m_IsControlPadRestrictionEnabled[NpadStyleJoyDual::Index] = false;
        m_IsControlPadRestrictionEnabled[NpadStyleJoyLeft::Index] = false;
        m_IsControlPadRestrictionEnabled[system::NpadStyleSystemExt::Index] = false;
    }

    virtual ~NpadSharedMemoryAccessor() NN_NOEXCEPT NN_OVERRIDE { /* 何もしない */ }

    //!< Npad の内部状態を返します。
    NpadInternalState* GetInternalState(NpadIdType id) const NN_NOEXCEPT
    {
        return m_pInternalState[GetIndexFromNpadIdType(id)];
    }

    void SetControlPadRestrictionType(NpadStyleSet style, ControlPadRestrictionType type) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(style.CountPopulation(), 1);
        for (int i = 0; i < style.StorageBitCount; i++)
        {
            if (style.Test(i) == true)
            {
                m_ControlPadRestrictionTypes[i] = type;
            }
        }
    }

    ControlPadRestrictionType GetControlPadRestrictionType(uint32_t index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS_EQUAL(index, NN_ARRAY_SIZE(m_ControlPadRestrictionTypes));
        return m_ControlPadRestrictionTypes[index];
    }

    void EnableControlPadRestriction(const NpadStyleSet& style, bool enable) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(style.CountPopulation(), 1);
        for (int i = 0; i < style.StorageBitCount; i++)
        {
            if (style.Test(i) == true)
            {
                m_IsControlPadRestrictionEnabled[i] = enable;
            }
        }
    }

    bool IsControlPadRestrictionEnabled(uint32_t index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_LESS_EQUAL(index, NN_ARRAY_SIZE(m_ControlPadRestrictionTypes));
        return m_IsControlPadRestrictionEnabled[index];
    }

    ControlPadRestrictionType GetControlPadRestrictionType(NpadStyleSet style) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_EQUAL(style.CountPopulation(), 1);
        for (int i = 0; i < style.GetCount(); i++)
        {
            if (style.Test(i) == true)
            {
                return m_ControlPadRestrictionTypes[i];
            }
        }
        return ControlPadRestrictionType_KeepUpAndLeft;
    }

protected:
    virtual ::nn::Result Attach(SharedMemoryFormat* ptr) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(ptr);
        ::nn::sf::SharedPointer<IHidServer> pHidServer;
        NN_RESULT_DO(CreateHidServerProxy(&pHidServer));
        NN_RESULT_DO(
            pHidServer->ActivateNpadWithRevision(GetAppletResourceUserId(), NpadApplicationRevision_1));
        for (int i = 0; i < NpadIdCountMax; i++)
        {
            m_pInternalState[i] = &::nn::util::Get(ptr->npad.entries[i].internalState);
        }
        NN_RESULT_SUCCESS;
    }

    virtual ::nn::Result Detach() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::sf::SharedPointer<IHidServer> pHidServer;
        NN_RESULT_DO(CreateHidServerProxy(&pHidServer));
        NN_RESULT_DO(pHidServer->DeactivateNpad(GetAppletResourceUserId()));
        for (auto& pState : m_pInternalState)
        {
            pState = nullptr;
        }
        NN_RESULT_SUCCESS;
    }
};


//!< 基本的な構成を持つ Xpad の共有メモリアクセッサを返します。
NpadSharedMemoryAccessor* GetNpadSharedMemoryAccessor() NN_NOEXCEPT
{
    return StaticObject<
        NpadSessionManager<
            NpadSharedMemoryAccessor>>::Get().GetSession();
}

//!< SystemExt のボタン状態を System にコンバートします
NpadButtonSet ConvertSystemButtonSet(const NpadButtonSet& input) NN_NOEXCEPT
{
    auto output = NpadButtonSet();
    output.Set<NpadButton::Down>(input.Test<NpadButton::Down>() ||
                                 input.Test<NpadButton::StickLDown>() ||
                                 input.Test<NpadButton::StickRDown>());
    output.Set<NpadButton::Up>(input.Test<NpadButton::Up>() ||
                               input.Test<NpadButton::StickLUp>() ||
                               input.Test<NpadButton::StickRUp>());
    output.Set<NpadButton::Left>(input.Test<NpadButton::Left>() ||
                                 input.Test<NpadButton::StickLLeft>() ||
                                 input.Test<NpadButton::StickRLeft>());
    output.Set<NpadButton::Right>(input.Test<NpadButton::Right>() ||
                                  input.Test<NpadButton::StickLRight>() ||
                                  input.Test<NpadButton::StickRRight>());

    output.Set<NpadButton::L>(input.Test<NpadButton::L>() ||
                              input.Test<NpadButton::ZL>());
    output.Set<NpadButton::R>(input.Test<NpadButton::R>() ||
                              input.Test<NpadButton::ZR>());

    output |= (input & (NpadButton::A::Mask |
                        NpadButton::B::Mask |
                        NpadButton::X::Mask |
                        NpadButton::Y::Mask));

    return output;
}

} // namespace


::nn::Result InitializeNpad() NN_NOEXCEPT
{
    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    NN_RESULT_DO(pAccessor->SetAppletResourceUserId(GetAppletResourceUserId()));
    NN_RESULT_DO(pAccessor->Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeNpad() NN_NOEXCEPT
{
    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    NN_RESULT_DO(pAccessor->Deactivate());
    NN_RESULT_SUCCESS;
}

::nn::Result BindNpadStyleSetUpdateEvent(const NpadIdType& id,
                             ::nn::os::SystemEventType* pEvent,
                             ::nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    auto handle = ::nn::sf::NativeHandle();
    auto clientId = reinterpret_cast<uint64_t>(pEvent);
    NN_RESULT_DO(pHidServer->AcquireNpadStyleSetUpdateEventHandle(GetAppletResourceUserId(), &handle, id, clientId));

    ::nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                                handle.GetOsHandle(),
                                                handle.IsManaged(),
                                                clearMode);

    // ハンドルの管理権を放棄
    handle.Detach();

    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadStyleSet(NpadStyleSet* pOutValue, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());
    *pOutValue = pInternalState->GetNpadStyleSet();
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_SUCCESS;
}

::nn::Result DisconnectNpad(const NpadIdType& id) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->DisconnectNpad(GetAppletResourceUserId(), id));
    NN_RESULT_SUCCESS;
}

::nn::Result GetPlayerLedPattern(Bit8* pOutValue, const NpadIdType& id) NN_NOEXCEPT
{
    uint64_t pattern;

    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->GetPlayerLedPattern(&pattern, id));

    *pOutValue = static_cast<Bit8>(pattern & 0xff);
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadJoyAssignment(NpadJoyAssignmentMode* pOutValue, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());
    *pOutValue = pInternalState->GetNpadJoyAssignmentMode();
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_RESULT_SUCCESS;
}

::nn::Result SetNpadJoyAssignmentModeSingle(const NpadIdType& id) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->SetNpadJoyAssignmentModeSingleByDefault(GetAppletResourceUserId(), id));
    NN_RESULT_SUCCESS;
}

::nn::Result SetNpadJoyAssignmentModeSingle(const NpadIdType& id, NpadJoyDeviceType type) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->SetNpadJoyAssignmentModeSingle(GetAppletResourceUserId(), id, type));
    NN_RESULT_SUCCESS;
}

::nn::Result SetNpadJoyAssignmentModeSingle(bool* pOutIsAssigned, NpadIdType* pOutId, const NpadIdType& id, NpadJoyDeviceType type) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->SetNpadJoyAssignmentModeSingleWithDestination(pOutIsAssigned, pOutId, GetAppletResourceUserId(), id, type));
    NN_RESULT_SUCCESS;
}

::nn::Result SetNpadJoyAssignmentModeDual(const NpadIdType& id) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->SetNpadJoyAssignmentModeDual(GetAppletResourceUserId(), id));
    NN_RESULT_SUCCESS;
}

::nn::Result MergeSingleJoyAsDualJoy(const NpadIdType& id1, const NpadIdType& id2) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->MergeSingleJoyAsDualJoy(GetAppletResourceUserId(), id1, id2));
    NN_RESULT_SUCCESS;
}

::nn::Result SwapNpadAssignment(const NpadIdType& id1, const NpadIdType& id2) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->SwapNpadAssignment(GetAppletResourceUserId(), id1, id2));
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadControllerColor(NpadControllerColor* pOutValue, const NpadIdType& id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutValue);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());
    NN_RESULT_THROW(pInternalState->GetNpadControllerColor(pOutValue));
}

::nn::Result GetNpadControllerColor(NpadControllerColor* pOutLeftColor, NpadControllerColor* pOutRightColor, const NpadIdType& id) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutLeftColor);
    NN_ABORT_UNLESS_NOT_NULL(pOutRightColor);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());
    NN_RESULT_THROW(pInternalState->GetNpadControllerColor(pOutLeftColor, pOutRightColor));
}

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadFullKeyState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGotCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    const auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());
    const NpadFullKeyLifo* pFullKeyLifo = pInternalState->GetFullKeyLifo();
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    *pOutGotCount = pFullKeyLifo->Read(pOutStates, count);
    NN_SDK_REQUIRES(!pFullKeyLifo->IsEmpty());

    for (int i = 0; i < *pOutGotCount; i++)
    {
        pOutStates[i].buttons = RestrictNpadButtons(pOutStates[i].buttons, pAccessor->GetControlPadRestrictionType(NpadStyleFullKey::Index), pAccessor->IsControlPadRestrictionEnabled(NpadStyleFullKey::Index));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadHandheldState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGotCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    const auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());
    const NpadHandheldLifo* pHandheldLifo = pInternalState->GetHandheldLifo();
    NN_RESULT_THROW_UNLESS(pHandheldLifo != nullptr, ResultNpadNotInitialized());
    *pOutGotCount = pHandheldLifo->Read(pOutStates, count);
    NN_SDK_REQUIRES(!pHandheldLifo->IsEmpty());

    for (int i = 0; i < *pOutGotCount; i++)
    {
        pOutStates[i].buttons = RestrictNpadButtons(pOutStates[i].buttons, pAccessor->GetControlPadRestrictionType(NpadStyleHandheld::Index), pAccessor->IsControlPadRestrictionEnabled(NpadStyleHandheld::Index));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadJoyDualState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGotCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    const auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());
    const NpadJoyDualLifo* pJoyDualLifo = pInternalState->GetJoyDualLifo();
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    *pOutGotCount = pJoyDualLifo->Read(pOutStates, count);
    NN_SDK_REQUIRES(!pJoyDualLifo->IsEmpty());

    for (int i = 0; i < *pOutGotCount; i++)
    {
        pOutStates[i].buttons = RestrictNpadButtons(pOutStates[i].buttons, pAccessor->GetControlPadRestrictionType(NpadStyleJoyDual::Index), pAccessor->IsControlPadRestrictionEnabled(NpadStyleJoyDual::Index));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadJoyLeftState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGotCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    const auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());
    const NpadJoyLeftLifo* pJoyLeftLifo = pInternalState->GetJoyLeftLifo();
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    *pOutGotCount = pJoyLeftLifo->Read(pOutStates, count);
    NN_SDK_REQUIRES(!pJoyLeftLifo->IsEmpty());

    for (int i = 0; i < *pOutGotCount; i++)
    {
        pOutStates[i].buttons = RestrictNpadButtons(pOutStates[i].buttons, pAccessor->GetControlPadRestrictionType(NpadStyleJoyLeft::Index), pAccessor->IsControlPadRestrictionEnabled(NpadStyleJoyLeft::Index));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadJoyRightState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGotCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    const auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());
    const NpadJoyRightLifo* pJoyRightLifo = pInternalState->GetJoyRightLifo();
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    *pOutGotCount = pJoyRightLifo->Read(pOutStates, count);
    NN_SDK_REQUIRES(!pJoyRightLifo->IsEmpty());

    for (int i = 0; i < *pOutGotCount; i++)
    {
        pOutStates[i].buttons = RestrictNpadButtons(pOutStates[i].buttons, pAccessor->GetControlPadRestrictionType(NpadStyleJoyRight::Index), pAccessor->IsControlPadRestrictionEnabled(NpadStyleJoyRight::Index));
    }
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadGcState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGotCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    NpadFullKeyState fullKeyState[NpadStateCountMax];
    NpadGcTriggerState gcTriggerState[NpadStateCountMax];

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    const auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());
    const NpadFullKeyLifo* pFullKeyLifo = pInternalState->GetFullKeyLifo();
    NN_SDK_REQUIRES_NOT_NULL(pFullKeyLifo);
    NN_SDK_REQUIRES(!pFullKeyLifo->IsEmpty());
    *pOutGotCount = pFullKeyLifo->Read(fullKeyState, count);

    const NpadGcTriggerLifo* pGcTriggerLifo = pInternalState->GetGcTriggerLifo();
    NN_SDK_REQUIRES_NOT_NULL(pGcTriggerLifo);
    NN_SDK_REQUIRES(!pGcTriggerLifo->IsEmpty());
    auto outGotCountTrigger = pGcTriggerLifo->Read(gcTriggerState, count);

    *pOutGotCount = std::min(*pOutGotCount, outGotCountTrigger);

    for (int i = 0; i < *pOutGotCount; i++)
    {
        pOutStates[i].buttons = RestrictNpadButtons(
            fullKeyState[i].buttons,
            pAccessor->GetControlPadRestrictionType(NpadStyleFullKey::Index),
            pAccessor->IsControlPadRestrictionEnabled(NpadStyleFullKey::Index));
        pOutStates[i].analogStickL = fullKeyState->analogStickL;
        pOutStates[i].analogStickR = fullKeyState->analogStickR;
        pOutStates[i].samplingNumber = fullKeyState[i].samplingNumber;
        pOutStates[i].attributes = fullKeyState[i].attributes;
        pOutStates[i].triggerL = gcTriggerState[i].triggerL;
        pOutStates[i].triggerR = gcTriggerState[i].triggerR;
    }
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadStates(int* pOutGotCount,
                           NpadPalmaState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGotCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    const auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());
    const NpadPalmaLifo* pPalmaLifo = pInternalState->GetPalmaLifo();
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    *pOutGotCount = pPalmaLifo->Read(pOutStates, count);
    NN_SDK_REQUIRES(!pPalmaLifo->IsEmpty());

    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadStates(int* pOutGotCount,
                           system::NpadSystemState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGotCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    system::NpadSystemExtState extState[NpadStateCountMax];

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());

    const NpadSystemExtLifo* pLifo = pInternalState->GetSystemExtLifo();

    *pOutGotCount = pLifo->Read(extState, count);
    NN_SDK_REQUIRES(!pLifo->IsEmpty());

    for (int i = 0; i < *pOutGotCount; i++)
    {
        pOutStates[i].buttons = RestrictNpadButtons(
            ConvertSystemButtonSet(extState[i].buttons),
            pAccessor->GetControlPadRestrictionType(system::NpadStyleSystem::Index), pAccessor->IsControlPadRestrictionEnabled(system::NpadStyleSystem::Index));
        pOutStates[i].analogStickL = AnalogStickState();
        pOutStates[i].analogStickR = AnalogStickState();
        pOutStates[i].attributes = extState[i].attributes;
    }
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadStates(int* pOutGotCount,
                           system::NpadSystemExtState* pOutStates,
                           int count,
                           const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGotCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());

    const NpadSystemExtLifo* pLifo = pInternalState->GetSystemExtLifo();

    *pOutGotCount = pLifo->Read(pOutStates, count);
    NN_SDK_REQUIRES(!pLifo->IsEmpty());

    for (int i = 0; i < *pOutGotCount; i++)
    {
        pOutStates[i].buttons = RestrictNpadButtons(pOutStates[i].buttons, pAccessor->GetControlPadRestrictionType(system::NpadStyleSystemExt::Index), pAccessor->IsControlPadRestrictionEnabled(system::NpadStyleSystemExt::Index));
    }
    NN_RESULT_SUCCESS;
}

// ------------------------------------------------
// 6軸センサー関連
// ------------------------------------------------

::nn::Result StartSixAxisSensor(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->StartSixAxisSensor(GetAppletResourceUserId(),
                                            handle));
    NN_RESULT_SUCCESS;
}

::nn::Result StopSixAxisSensor(const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->StopSixAxisSensor(GetAppletResourceUserId(),
                                           handle));
    NN_RESULT_SUCCESS;
}

::nn::Result IsSixAxisSensorFusionEnabled(bool* pOutIsEnabled,
                                          const nn::hid::SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutIsEnabled);

    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto outIsEnabled = bool();

    NN_RESULT_DO(pProxy->IsSixAxisSensorFusionEnabled(::nn::sf::Out<bool>(&outIsEnabled),
                                                      GetAppletResourceUserId(),
                                                      handle));
    *pOutIsEnabled = static_cast<bool>(outIsEnabled);
    NN_RESULT_SUCCESS;
}

::nn::Result EnableSixAxisSensorFusion(const nn::hid::SixAxisSensorHandle& handle, bool enable) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->EnableSixAxisSensorFusion(GetAppletResourceUserId(),
                                                   handle,
                                                   enable));
    NN_RESULT_SUCCESS;
}

::nn::Result SetSixAxisSensorFusionParameters(const nn::hid::SixAxisSensorHandle& handle,
                                              float revisePower,
                                              float reviseRange) NN_NOEXCEPT
{
    const bool IsValidRevisePower = (revisePower >= 0.0f && revisePower <= 1.0f);

    NN_SDK_REQUIRES(IsValidRevisePower, "revisePower is out of range.");

    NN_RESULT_THROW_UNLESS(IsValidRevisePower,
                           ResultSixAxisSensorFusionRevisePowerOutOfRange());

    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->SetSixAxisSensorFusionParameters(GetAppletResourceUserId(),
                                                          handle,
                                                          revisePower,
                                                          reviseRange));
    NN_RESULT_SUCCESS;
}

::nn::Result GetSixAxisSensorFusionParameters(float* pOutRevisePower,
                                              float* pOutReviseRange,
                                              const nn::hid::SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutRevisePower);
    NN_ABORT_UNLESS_NOT_NULL(pOutReviseRange);

    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto outRevisePower = float();
    auto outReviseRange = float();

    NN_RESULT_DO(pProxy->GetSixAxisSensorFusionParameters(::nn::sf::Out<float>(&outRevisePower),
                                                          ::nn::sf::Out<float>(&outReviseRange),
                                                          GetAppletResourceUserId(),
                                                          handle));
    *pOutRevisePower = static_cast<float>(outRevisePower);
    *pOutReviseRange = static_cast<float>(outReviseRange);

    NN_RESULT_SUCCESS;
}

::nn::Result ResetSixAxisSensorFusionParameters(const nn::hid::SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ResetSixAxisSensorFusionParameters(GetAppletResourceUserId(),
                                                            handle));
    NN_RESULT_SUCCESS;
}

::nn::Result SetAccelerometerParameters(nn::hid::SixAxisSensorHandle handle, float playRadius, float sensitivity) NN_NOEXCEPT
{
    const bool IsValidPlayRadius = (playRadius >= 0.0f);
    const bool IsValidSensitivity = (sensitivity >= 0.0f && sensitivity <= 1.0f);

    NN_SDK_REQUIRES(IsValidPlayRadius, "playRadius is out of range.");
    NN_SDK_REQUIRES(IsValidSensitivity, "sensitivity is out of range.");

    NN_RESULT_THROW_UNLESS(IsValidPlayRadius,
                           ResultAccelerometerPlayRadiusOutOfRange());
    NN_RESULT_THROW_UNLESS(IsValidSensitivity,
                           ResultAccelerometerSensitivityOutOfRange());

    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->SetAccelerometerParameters(GetAppletResourceUserId(),
                                                    handle,
                                                    playRadius,
                                                    sensitivity));
    NN_RESULT_SUCCESS;
}

::nn::Result GetAccelerometerParameters(float* pOutPlayRadius, float* pOutSensitivity, nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutPlayRadius);
    NN_ABORT_UNLESS_NOT_NULL(pOutSensitivity);

    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto outPlayRadius = float();
    auto outSensitivity = float();

    NN_RESULT_DO(pProxy->GetAccelerometerParameters(::nn::sf::Out<float>(&outPlayRadius),
                                                    ::nn::sf::Out<float>(&outSensitivity),
                                                    GetAppletResourceUserId(),
                                                    handle));
    *pOutPlayRadius  = static_cast<float>(outPlayRadius);
    *pOutSensitivity = static_cast<float>(outSensitivity);

    NN_RESULT_SUCCESS;
}

::nn::Result ResetAccelerometerParameters(nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ResetAccelerometerParameters(GetAppletResourceUserId(), handle));
    NN_RESULT_SUCCESS;
}

::nn::Result SetAccelerometerPlayMode(nn::hid::SixAxisSensorHandle handle,
                                      nn::hid::AccelerometerPlayMode mode) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->SetAccelerometerPlayMode(GetAppletResourceUserId(),
                                                  handle,
                                                  mode));
    NN_RESULT_SUCCESS;
}

::nn::Result GetAccelerometerPlayMode(nn::hid::AccelerometerPlayMode* pOutMode,
                                      nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutMode);

    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto outMode = uint32_t();

    NN_RESULT_DO(pProxy->GetAccelerometerPlayMode(::nn::sf::Out<uint32_t>(&outMode),
                                                  GetAppletResourceUserId(),
                                                  handle));
    *pOutMode  = static_cast<nn::hid::AccelerometerPlayMode>(outMode);

    NN_RESULT_SUCCESS;
}

::nn::Result ResetAccelerometerPlayMode(nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ResetAccelerometerPlayMode(GetAppletResourceUserId(), handle));
    NN_RESULT_SUCCESS;
}

::nn::Result SetGyroscopeZeroDriftMode(nn::hid::SixAxisSensorHandle handle, nn::hid::GyroscopeZeroDriftMode mode) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->SetGyroscopeZeroDriftMode(GetAppletResourceUserId(),
                                                   handle,
                                                   mode));
    NN_RESULT_SUCCESS;
}

::nn::Result GetGyroscopeZeroDriftMode(nn::hid::GyroscopeZeroDriftMode* pOutMode, nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutMode);

    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto outMode = uint32_t();

    NN_RESULT_DO(pProxy->GetGyroscopeZeroDriftMode(::nn::sf::Out<uint32_t>(&outMode),
                                                   GetAppletResourceUserId(),
                                                   handle));
    *pOutMode  = static_cast<nn::hid::GyroscopeZeroDriftMode>(outMode);

    NN_RESULT_SUCCESS;
}

::nn::Result ResetGyroscopeZeroDriftMode(nn::hid::SixAxisSensorHandle handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ResetGyroscopeZeroDriftMode(GetAppletResourceUserId(), handle));
    NN_RESULT_SUCCESS;
}

::nn::Result IsSixAxisSensorAtRest(bool* pOutIsAtRest,
    const nn::hid::SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pOutIsAtRest);

    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto outIsAtRest = bool();

    NN_RESULT_DO(pProxy->IsSixAxisSensorAtRest(::nn::sf::Out<bool>(&outIsAtRest),
                                               GetAppletResourceUserId(),
                                               handle));
    *pOutIsAtRest = static_cast<bool>(outIsAtRest);
    NN_RESULT_SUCCESS;
}

::nn::Result GetSixAxisSensorStates(int* pOutGotCount,
                                    SixAxisSensorState* pOutStates,
                                    int count,
                                    const NpadIdType& id,
                                    NpadStyleSet style) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGotCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    const NpadSixAxisSensorLifo* pSixAxisSensorLifo = nullptr;

    const auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());

    if (style.Test<NpadStyleFullKey>())
    {
        pSixAxisSensorLifo = pInternalState->GetFullKeySixAxisSensorLifo();
    }
    else if (style.Test<NpadStyleHandheld>())
    {
        pSixAxisSensorLifo = pInternalState->GetHandheldSixAxisSensorLifo();
    }
    else if (style.Test<NpadStyleJoyDual>())
    {
        NN_ABORT("Not allowed to set NpadStyleJoyDual");
    }
    else if (style.Test<NpadStyleJoyLeft>())
    {
        pSixAxisSensorLifo = pInternalState->GetJoyLeftSixAxisSensorLifo();
    }
    else if (style.Test<NpadStyleJoyRight>())
    {
        pSixAxisSensorLifo = pInternalState->GetJoyRightSixAxisSensorLifo();
    }
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);
    *pOutGotCount = pSixAxisSensorLifo->Read(pOutStates, count);

    NN_SDK_REQUIRES(!pSixAxisSensorLifo->IsEmpty()); //janos: it can happen with the proper pipelining
    NN_RESULT_SUCCESS;
}

::nn::Result GetSixAxisSensorRightStates(int* pOutGotCount,
                                         SixAxisSensorState* pOutStates,
                                         int count,
                                         const NpadIdType& id,
                                         NpadStyleSet style) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGotCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    NN_ABORT_UNLESS(style.Test<NpadStyleJoyDual>() == true);

    const auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());

    const NpadSixAxisSensorLifo* pSixAxisSensorLifo = pInternalState->GetJoyDualRightSixAxisSensorLifo();
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    *pOutGotCount = pSixAxisSensorLifo->Read(pOutStates, count);
    NN_SDK_REQUIRES(!pSixAxisSensorLifo->IsEmpty());
    NN_RESULT_SUCCESS;
}

::nn::Result GetSixAxisSensorLeftStates(int* pOutGotCount,
                                    SixAxisSensorState* pOutStates,
                                    int count,
                                    const NpadIdType& id,
                                    NpadStyleSet style) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutGotCount);
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    NN_ABORT_UNLESS(style.Test<NpadStyleJoyDual>() == true);

    const auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());
    const NpadSixAxisSensorLifo* pSixAxisSensorLifo = pInternalState->GetJoyDualLeftSixAxisSensorLifo();
    NN_SDK_REQUIRES_NOT_NULL(pOutStates);

    *pOutGotCount = pSixAxisSensorLifo->Read(pOutStates, count);
    NN_SDK_REQUIRES(!pSixAxisSensorLifo->IsEmpty());
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadDeviceType(system::NpadDeviceTypeSet* pOutValue, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());

    *pOutValue = pInternalState->GetNpadDeviceTypeSet();
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadPowerInfo(system::NpadPowerInfo* pOutValue, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());

    *pOutValue = pInternalState->GetNpadPowerInfo(NpadInternalState::BatteryIndex_Standard);
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadPowerInfo(system::NpadPowerInfo* pOutLeftJoy, system::NpadPowerInfo* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLeftJoy);
    NN_SDK_REQUIRES_NOT_NULL(pOutRightJoy);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());

    *pOutLeftJoy = pInternalState->GetNpadPowerInfo(NpadInternalState::BatteryIndex_Left);
    *pOutRightJoy = pInternalState->GetNpadPowerInfo(NpadInternalState::BatteryIndex_Right);
    NN_RESULT_SUCCESS;
}

::nn::Result IsUnsupportedButtonPressed(bool* pOutValue, const NpadIdType& id, const NpadStyleSet& style) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());

    if (style == system::NpadStyleSystem::Mask)
    {
        *pOutValue = pInternalState->IsUnsuportedButtonPressedOnNpadSystem();
    }
    else if (style == system::NpadStyleSystemExt::Mask)
    {
        *pOutValue = pInternalState->IsUnsuportedButtonPressedOnNpadSystemExt();
    }
    else
    {
        NN_ABORT("Unsupported NpadStyleSet");
    }

    NN_RESULT_SUCCESS;
}

::nn::Result IsAbxyButtonOriented(bool* pOutValue, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());

    *pOutValue = pInternalState->IsAbxyButtonOriented();
    NN_RESULT_SUCCESS;
}

::nn::Result IsSlSrButtonOriented(bool* pOutValue, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());

    *pOutValue = pInternalState->IsSlSrButtonOriented();
    NN_RESULT_SUCCESS;
}

::nn::Result GetPlusMinusButtonCapability(bool* pOutPlusButton, bool* pOutMinusButton, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPlusButton);
    NN_SDK_REQUIRES_NOT_NULL(pOutMinusButton);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr, ResultNpadNotInitialized());

    *pOutPlusButton = pInternalState->IsPlusAvailable();
    *pOutMinusButton = pInternalState->IsMinusAvailable();
    NN_RESULT_SUCCESS;

}

::nn::Result SetControlPadRestrictionType(NpadStyleSet style, ControlPadRestrictionType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(style.CountPopulation(), 1);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());

    pAccessor->SetControlPadRestrictionType(style, type);

    NN_RESULT_SUCCESS;
}

::nn::Result GetControlPadRestrictionType(NpadStyleSet style, ControlPadRestrictionType* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(style.CountPopulation(), 1);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());

    *pOutValue = pAccessor->GetControlPadRestrictionType(style);

    NN_RESULT_SUCCESS;
}

::nn::Result EnableControlPadRestriction(const NpadStyleSet& style, bool enable) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(style.CountPopulation(), 1);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());

    pAccessor->EnableControlPadRestriction(style, enable);

    NN_RESULT_SUCCESS;
}

::nn::Result IsUnintendedHomeButtonInputProtectionEnabled(bool* pOutEnabled, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEnabled);

    NpadSharedMemoryAccessor* const
        pAccessor = GetNpadSharedMemoryAccessor();
    NN_RESULT_THROW_UNLESS(pAccessor != nullptr,
                           ResultNpadInvalidId());
    auto pInternalState = pAccessor->GetInternalState(id);
    NN_RESULT_THROW_UNLESS(pInternalState != nullptr,
                           ResultNpadNotInitialized());

    *pOutEnabled = !(pInternalState->IsHomeButtonEnabled());

    NN_RESULT_SUCCESS;
}

::nn::Result SetShiftAccelerometerCalibrationValue(const SixAxisSensorHandle& handle,
                                                   float shiftOrigin,
                                                   float shiftSensitivity) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->SetShiftAccelerometerCalibrationValue(handle,
                                                                        GetAppletResourceUserId(),
                                                                        shiftOrigin,
                                                                        shiftSensitivity));
    NN_RESULT_SUCCESS;
}

::nn::Result GetShiftAccelerometerCalibrationValue(float* pOutShiftOrigin,
                                                   float* pOutShiftSensitivity,
                                                   const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->GetShiftAccelerometerCalibrationValue(pOutShiftOrigin,
                                                                        pOutShiftSensitivity,
                                                                        handle,
                                                                        GetAppletResourceUserId()));
    NN_RESULT_SUCCESS;
}

::nn::Result SetShiftGyroscopeCalibrationValue(const SixAxisSensorHandle& handle,
                                               float shiftOrigin,
                                               float shiftSensitivity) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->SetShiftGyroscopeCalibrationValue(handle,
                                                                    GetAppletResourceUserId(),
                                                                    shiftOrigin,
                                                                    shiftSensitivity));
    NN_RESULT_SUCCESS;
}

::nn::Result GetShiftGyroscopeCalibrationValue(float* pOutShiftOrigin,
                                               float* pOutShiftSensitivity,
                                               const SixAxisSensorHandle& handle) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->GetShiftGyroscopeCalibrationValue(pOutShiftOrigin,
                                                                    pOutShiftSensitivity,
                                                                    handle,
                                                                    GetAppletResourceUserId()));
    NN_RESULT_SUCCESS;
}

::nn::Result HasBattery(bool* pOutValue, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->HasBattery(pOutValue, id));
    NN_RESULT_SUCCESS;
}

::nn::Result HasBattery(bool* pOutLeftJoy, bool* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLeftJoy);
    NN_SDK_REQUIRES_NOT_NULL(pOutRightJoy);

    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->HasLeftRightBattery(pOutLeftJoy, pOutRightJoy, id));
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadInterfaceType(system::InterfaceType* pOutValue, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->GetNpadInterfaceType(pOutValue, id));
    NN_RESULT_SUCCESS;
}

::nn::Result GetNpadInterfaceType(system::InterfaceType* pOutLeftJoy, system::InterfaceType* pOutRightJoy, const NpadIdType& id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutLeftJoy);
    NN_SDK_REQUIRES_NOT_NULL(pOutRightJoy);

    ::nn::sf::SharedPointer<IHidServer> pHidServer;
    NN_RESULT_DO(CreateHidServerProxy(&pHidServer));

    NN_RESULT_DO(pHidServer->GetNpadLeftRightInterfaceType(pOutLeftJoy, pOutRightJoy, id));
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
