﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>

#include "hid_AppletResourceManager.h"
#include "hid_HandheldManager.h"
#include "hid_InputDetectorManager.h"
#include "hid_InterruptSceneNotifier.h"
#include "hid_NpadAppletPolicyManager.h"
#include "hid_OvlnSenderManager.h"
#include "hid_PlayReportManager.h"

namespace nn { namespace hid { namespace detail {

//!< Npad が共通で扱う hid のサービスを保持するクラスです
class NpadCommonResourceHolder final
{
    NN_DISALLOW_COPY(NpadCommonResourceHolder);
    NN_DISALLOW_MOVE(NpadCommonResourceHolder);

private:
    //!< アプレットリソースマネージャ
    AppletResourceManager* m_pAppletResourceManager;

    //!< アプレットリソースマネージャのミューテックス
    ::nn::os::SdkRecursiveMutex* m_pAppletResourceManagerMutex;

    //!< Npad のポリシーマネージャ
    NpadAppletPolicyManager* m_pNpadAppletPolicyManager;

    //!< Handheld マネージャ
    HandheldManager* m_pHandheldManager;

    //!< InputDetector マネージャ
    InputDetectorManager* m_pInputDetectorManager;

    //!< InputDetector マネージャのミューテックス
    ::nn::os::SdkMutex* m_pInputDetectorManagerMutex;

    //!< オーバーレイ通知を送信するための型
    OvlnSenderManager* m_pOvlnSender;

    //!< 割り込みシーンの表示を通知クラス
    InterruptSceneNotifier* m_pInterruptSceneNotifier;

    //!< PlayReport 用の情報を送信するためのマネージャー
    PlayReportManager* m_pPlayReportManager;

public:
    NpadCommonResourceHolder() NN_NOEXCEPT
        : m_pAppletResourceManager(nullptr)
        , m_pAppletResourceManagerMutex(nullptr)
        , m_pNpadAppletPolicyManager(nullptr)
        , m_pHandheldManager(nullptr)
        , m_pInputDetectorManager(nullptr)
        , m_pInputDetectorManagerMutex(nullptr)
        , m_pOvlnSender(nullptr)
        , m_pInterruptSceneNotifier(nullptr)
        , m_pPlayReportManager(nullptr)
    {
        // 何もしない
    }

    //!< アプレットリソースマネージャを設定します。
    void SetAppletResourceManager(
        AppletResourceManager* pManager, ::nn::os::SdkRecursiveMutex* pMutex
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pManager);
        NN_SDK_REQUIRES_NOT_NULL(pMutex);
        m_pAppletResourceManager = pManager;
        m_pAppletResourceManagerMutex = pMutex;
    }

    //!< Npad のポリシーマネージャを設定します。
    void SetNpadAppletPolicyManager(NpadAppletPolicyManager* pManager) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pManager);
        m_pNpadAppletPolicyManager = pManager;
    }

    //!< Handheld マネージャを設定します。
    void SetHandheldManager(HandheldManager* pManager) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pManager);
        m_pHandheldManager = pManager;
    }

    //!< InputDetector マネージャを設定します。
    void SetInputDetectorManager(InputDetectorManager* pManager,
                                 ::nn::os::SdkMutex* pMutex) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pManager);
        NN_SDK_REQUIRES_NOT_NULL(pMutex);
        m_pInputDetectorManager = pManager;
        m_pInputDetectorManagerMutex = pMutex;
    }

    //!< オーバーレイ通知用のマネージャーを設定します。
    void SetOvlnSenderManager(OvlnSenderManager* pSender) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pSender);
        m_pOvlnSender = pSender;
    }

    //!< 割り込みシーン通知用のマネージャーを設定します。
    void SetInterruptSceneNotifier(InterruptSceneNotifier* pNotifier) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pNotifier);
        m_pInterruptSceneNotifier = pNotifier;
    }

    //!< プレイレポート送信用のマネージャーを設定します。
    void SetPlayReportManager(PlayReportManager* pManager) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pManager);
        m_pPlayReportManager = pManager;
    }

    //!< 共通リソースがすべて初期化済みかどうか検証します
    bool IsInitialized() NN_NOEXCEPT
    {
        return(
            m_pAppletResourceManager != nullptr &&
            m_pAppletResourceManagerMutex != nullptr &&
            m_pNpadAppletPolicyManager != nullptr &&
            m_pHandheldManager != nullptr &&
            m_pInputDetectorManager != nullptr &&
            m_pInputDetectorManagerMutex != nullptr &&
            m_pOvlnSender != nullptr &&
            m_pInterruptSceneNotifier != nullptr &&
            m_pPlayReportManager != nullptr
            );
    }

    //!< アプレットリソースマネージャを取得します。
    AppletResourceManager& GetAppletResourceManager() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManager);
        return *m_pAppletResourceManager;
    }

    //!< アプレットリソースマネージャへのアクセスを排他するための Mutex を取得します。
    nn::os::SdkRecursiveMutex& GetAppletResourceManagerMutex() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pAppletResourceManagerMutex);
        return *m_pAppletResourceManagerMutex;
    }

    //!< Npad のポリシーマネージャを取得します。
    NpadAppletPolicyManager& GetNpadAppletPolicyManager() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pNpadAppletPolicyManager);
        return *m_pNpadAppletPolicyManager;
    }

    //!< Handheld マネージャを取得します。
    HandheldManager& GetHandheldManager() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pHandheldManager);
        return *m_pHandheldManager;
    }

    //!< InputDetector マネージャを取得します。
    InputDetectorManager& GetInputDetectorManager() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManager);
        return *m_pInputDetectorManager;
    }

    //!< InputDetector マネージャへのアクセスを排他するための Mutex を取得します。
    nn::os::SdkMutex& GetInputDetectorManagerMutex() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pInputDetectorManagerMutex);
        return *m_pInputDetectorManagerMutex;
    }

    //!< オーバーレイ通知用のマネージャーを取得します。
    OvlnSenderManager& GetOvlnSenderManager() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pOvlnSender);
        return *m_pOvlnSender;
    }

    //!< 割り込みシーン通知用のマネージャーを取得します。
    InterruptSceneNotifier& GetInterruptSceneNotifier() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pInterruptSceneNotifier);
        return *m_pInterruptSceneNotifier;
    }

    //!< プレイレポート送信用のマネージャーを取得します。
    PlayReportManager& GetPlayReportManager() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pPlayReportManager);
        return *m_pPlayReportManager;
    }
};

}}} // namespace nn::hid::detail
