﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/xcd/xcd.h>

#include "hid_BluetoothMode.h"
#include "hid_NpadAssignmentManager.h"
#include "hid_NpadStateUtility.h"

namespace nn { namespace hid { namespace detail {

namespace
{
    const NpadJoyDeviceType DefaultAssignmentForSingle = NpadJoyDeviceType_Left;

    bool IsWirelessAcceptableNpadId(NpadIdType id)
    {
        switch (id)
        {
        case NpadId::No1:
        case NpadId::No2:
        case NpadId::No3:
        case NpadId::No4:
        case NpadId::No5:
        case NpadId::No6:
        case NpadId::No7:
        case NpadId::No8:
            return true;
        default:
            return false;
        }
    }
}

NpadAssignmentManager::NpadAssignmentManager() NN_NOEXCEPT
    : m_pCommonResourceHolder(nullptr)
    , m_pNpads(nullptr)
    , m_pLastActiveControllerMonitor(nullptr)
    , m_AbstractedPadCount(0)
    , m_ActivationCount()
{
    // 何もしない
}

NpadAssignmentManager::~NpadAssignmentManager() NN_NOEXCEPT
{
    // 何もしない
}

void NpadAssignmentManager::SetNpadCommonResourceHolder(NpadCommonResourceHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(pHolder->IsInitialized());
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pCommonResourceHolder = pHolder;
}

void NpadAssignmentManager::SetNpadManagers(NpadManagerArray* pNpads) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pNpads);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());

    m_pNpads = pNpads;
}

void NpadAssignmentManager::SetNpadLastActiveControllerMonitor(NpadLastActiveControllerMonitor* pMonitor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pMonitor);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pLastActiveControllerMonitor = pMonitor;
}

void NpadAssignmentManager::AddIAbstractedPad(IAbstractedPad* pPads) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPads);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pAbstractedPads[m_AbstractedPadCount] = pPads;
    m_AbstractedPadCount++;
}

::nn::Result NpadAssignmentManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pCommonResourceHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pNpads);
    NN_SDK_REQUIRES_NOT_NULL(m_pLastActiveControllerMonitor);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadAssignmentManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    NN_RESULT_SUCCESS;
}

void NpadAssignmentManager::UpdateDeviceMap() NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        return;
    }

    // 参照するポリシーをアップデート
    const auto policy = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy();

    CheckForDeviceDetachOnAllNpads(policy);
    CheckForDeviceAttach(policy);
    UpdateAssignment(policy);
}

void NpadAssignmentManager::UpdateOnAruidSwitch() NN_NOEXCEPT
{
    // 参照するポリシーをアップデート
    const auto& policy = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy();

    DisconnectUnsupporedNpadStyle(policy);
    UpdateJoyAssignmentMode(policy);
    DisconnectUnsupporedNpadIds(policy);
    UpdateNpadAll(policy);
}

void NpadAssignmentManager::Resume() NN_NOEXCEPT
{
    // ジョイコンの2本持ち状態をすべてリセット
    for (auto& npadId : SupportedNpadIdList)
    {
        if (npadId != NpadId::Handheld &&
            npadId != system::NpadSystemId::Other)
        {
            this->SetNpadJoyAssignmentModeDual(m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetFocusedAppletResourceUserId(), npadId);
        }
    }
    // デバイスマップを更新
    UpdateDeviceMap();
}

void NpadAssignmentManager::ProceedSupportedNpadStyleSetUpdate() NN_NOEXCEPT
{
    // 参照するポリシーをアップデート
    const auto policy = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy();

    // スタイルに対応していないコントローラーが存在する場合は切断
    DisconnectUnsupporedNpadStyle(policy);

    // スタイルに合わせて割り当てモードも更新
    UpdateJoyAssignmentMode(policy);

    // 各 Npad の内部状態の更新
    UpdateNpadAll(policy);
}

void NpadAssignmentManager::ProceedSupportedNpadIdTypeUpdate() NN_NOEXCEPT
{
    // 参照するポリシーをアップデート
    const auto policy = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy();

    // 非サポートになったコントローラーは切断
    DisconnectUnsupporedNpadIds(policy);

    // 各 Npad の内部状態の更新
    UpdateNpadAll(policy);
}

void NpadAssignmentManager::ProceedApplyNpadSystemCommonPolicy() NN_NOEXCEPT
{
    // 参照するポリシーをアップデート
    const auto policy = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy();
    UpdateNpadAll(policy);
}

void NpadAssignmentManager::ProceedEnableLrAssignmentMode() NN_NOEXCEPT
{
    // LR Assignment Mode を開始した際は未接続のスロットに対しては2本持ちをデフォルトにする
    for (auto& npad : *m_pNpads)
    {
        auto& holder = npad.GetAbstractedPadHolder();
        if (holder.GetDeviceType().IsAllOff())
        {
            holder.SetJoyAssignmentMode(NpadJoyAssignmentMode_Dual);
        }
    }
}

void NpadAssignmentManager::Disconnect(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    // フォーカスあたっている ARUID 以外からの要求は何もしない
    if (aruid != m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetFocusedAppletResourceUserId())
    {
        return;
    }

    this->DisconnectAllImpl(GetNpadManager(*m_pNpads, id).GetNpadIdType());

    // NpadStyleSet を更新
    GetNpadManager(*m_pNpads, id).UpdateNpad();
}

NpadJoyAssignmentMode NpadAssignmentManager::GetNpadJoyAssignmentMode(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    return GetNpadManager(*m_pNpads, id).GetAbstractedPadHolder().GetJoyAssignmentMode();
}

void NpadAssignmentManager::SetNpadJoyAssignmentModeSingleByDefault(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    // フォーカスあたっている ARUID 以外からの要求は何もしない
    if (aruid != m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetFocusedAppletResourceUserId())
    {
        return;
    }

    SetNpadJoyAssignmentModeSingle(aruid, id, DefaultAssignmentForSingle);
}

void NpadAssignmentManager::SetNpadJoyAssignmentModeSingle(::nn::applet::AppletResourceUserId aruid, NpadIdType id, NpadJoyDeviceType type) NN_NOEXCEPT
{
    // フォーカスあたっている ARUID 以外からの要求は何もしない
    if (aruid != m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetFocusedAppletResourceUserId())
    {
        return;
    }

    SetNpadJoyAssignmentModeSingleWithDestination(nullptr, aruid, id, type);
}

bool NpadAssignmentManager::SetNpadJoyAssignmentModeSingleWithDestination(NpadIdType* pOutValue,
                                                                          ::nn::applet::AppletResourceUserId aruid,
                                                                          NpadIdType id,
                                                                          NpadJoyDeviceType type) NN_NOEXCEPT
{
    // フォーカスあたっている ARUID 以外からの要求は何もしない
    if (aruid != m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetFocusedAppletResourceUserId())
    {
        return false;
    }

    const auto& policy = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy();

    // 変更予定の Npad についてコントローラーの接続状態が変化していないか確認
    CheckForDeviceDetach(id, policy);

    auto& holder = GetNpadManager(*m_pNpads, id).GetAbstractedPadHolder();
    auto deviceType = holder.GetDeviceType();

    // NpadStyleJoyLeft 及び JoyRight がともに無効の場合は 1本持ちに変更できないため無視する
    auto supportedStyle = policy.GetSupportedNpadStyleSet();
    if (supportedStyle.Test<NpadStyleJoyLeft>() == false &&
        supportedStyle.Test<NpadStyleJoyRight>() == false)
    {
        return false;
    }

    bool isAssigned = false;

    if (IsJoyDual(deviceType))
    {
        IAbstractedPad* pDetachedPad;

        // Type で指定されなかったほうのジョイコンをディアクティベート
        if (type == NpadJoyDeviceType_Left)
        {
            // 右コンをディアクティベート
            pDetachedPad = holder.DetachAbstractedPad(system::DeviceType::JoyConRight::Mask);
        }
        else
        {
            // 左コンをディアクティベート
            pDetachedPad = holder.DetachAbstractedPad(system::DeviceType::JoyConLeft::Mask);
        }

        // NpadManager をシングルモードに変更
        holder.SetJoyAssignmentMode(NpadJoyAssignmentMode_Single);

        // ディアクティベートしたジョイコンをあいている Npad に割り当て
        if (pDetachedPad != nullptr)
        {
            int index;
            if (AddAbstractedPadToNewNpad(&index, pDetachedPad, false, policy) == true)
            {
                (*m_pNpads)[index].UpdateNpad();
                isAssigned = true;
                if (pOutValue != nullptr)
                {
                    *pOutValue = SupportedNpadIdList[index];
                }
            }
        }
    }
    else
    {
        // NpadManager をシングルモードに変更
        holder.SetJoyAssignmentMode(NpadJoyAssignmentMode_Single);
    }

    // NpadStyleSet を更新
    GetNpadManager(*m_pNpads, id).UpdateNpad();
    return isAssigned;
}

void NpadAssignmentManager::SetNpadJoyAssignmentModeDual(::nn::applet::AppletResourceUserId aruid, NpadIdType id) NN_NOEXCEPT
{
    // フォーカスあたっている ARUID 以外からの要求は何もしない
    if (aruid != m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetFocusedAppletResourceUserId())
    {
        return;
    }


    // NpadStyleJoyDual 及び FullKey がともに無効の場合は 2本持ちに変更できないため無視する
    auto supportedStyle = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy().GetSupportedNpadStyleSet();
    if (supportedStyle.Test<NpadStyleJoyDual>() == false &&
        supportedStyle.Test<NpadStyleFullKey>() == false)
    {
        return;
    }

    const auto& policy = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy();

    // 変更予定の Npad についてコントローラーの接続状態が変化していないか確認
    CheckForDeviceDetach(id, policy);

    GetNpadManager(*m_pNpads, id).GetAbstractedPadHolder().SetJoyAssignmentMode(NpadJoyAssignmentMode_Dual);

    // NpadStyleSet を更新
    GetNpadManager(*m_pNpads, id).UpdateNpad();
}

::nn::Result NpadAssignmentManager::MergeSingleJoyAsDualJoy(::nn::applet::AppletResourceUserId aruid, NpadIdType id1, NpadIdType id2) NN_NOEXCEPT
{
    // フォーカスあたっている ARUID 以外からの要求は何もしない
    if (aruid != m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetFocusedAppletResourceUserId())
    {
        NN_RESULT_SUCCESS;
    }

    const auto& policy = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy();

    // 変更予定の Npad についてコントローラーの接続状態が変化していないか確認
    CheckForDeviceDetach(id1, policy);
    CheckForDeviceDetach(id2, policy);

    auto index1 = GetIndexFromNpadIdType(id1);
    auto& holder1 = (*m_pNpads)[index1].GetAbstractedPadHolder();
    auto deviceType1 = holder1.GetDeviceType();
    auto index2 = GetIndexFromNpadIdType(id2);
    auto& holder2 = (*m_pNpads)[index2].GetAbstractedPadHolder();
    auto deviceType2 = holder2.GetDeviceType();

    if (deviceType1 == system::DeviceType::JoyConLeft::Mask)
    {
        if (deviceType2 == system::DeviceType::JoyConRight::Mask)
        {
            // id2 の Npad から右コンをディアクティベート
            auto pPad = holder2.DetachAbstractedPad(system::DeviceType::JoyConRight::Mask);

            // 両方の Npad の割り当てモードを 2本セットに変更
            holder2.SetJoyAssignmentMode(NpadJoyAssignmentMode_Dual);
            holder1.SetJoyAssignmentMode(NpadJoyAssignmentMode_Dual);

            // ディアクティベートした右コンを id1 でアクティベート
            if (pPad != nullptr)
            {
                AddAbstractedPadToNpadImpl(index1, pPad, false);
            }
        }
        else if (deviceType2 == system::DeviceType::JoyConLeft::Mask)
        {
            // 両方とも左コンが接続されている
            NN_RESULT_DO(ResultNpadSameJoyTypeConnected());
        }
        else
        {
            // ジョイコン が2本接続されている
            // TODO: 現状エラーとしてジョイコン以外が接続されている場合, 接続されていない場合も含んでいる
            NN_RESULT_DO(ResultNpadDualConnected());
        }
    }
    else if (deviceType1 == system::DeviceType::JoyConRight::Mask)
    {
        if (deviceType2 == system::DeviceType::JoyConLeft::Mask)
        {
            // id2 の Npad から左コンをディアクティベート
            auto pPad = holder2.DetachAbstractedPad(system::DeviceType::JoyConLeft::Mask);

            // 両方の Npad の割り当てモードを 2本セットに変更
            holder2.SetJoyAssignmentMode(NpadJoyAssignmentMode_Dual);
            holder1.SetJoyAssignmentMode(NpadJoyAssignmentMode_Dual);

            // ディアクティベートした右コンを id1 でアクティベート
            if (pPad != nullptr)
            {
                AddAbstractedPadToNpadImpl(index1, pPad, false);
            }
        }
        else if (deviceType2 == system::DeviceType::JoyConLeft::Mask)
        {
            // 両方とも左コンが接続されている
            NN_RESULT_DO(ResultNpadSameJoyTypeConnected());
        }
        else
        {
            // ジョイコン が2本接続されている
            // TODO: 現状エラーとしてジョイコン以外が接続されている場合, 接続されていない場合も含んでいる
            NN_RESULT_DO(ResultNpadDualConnected());
        }
    }
    else
    {
        // ジョイコン が2本接続されている
        // TODO: 現状エラーとしてジョイコン以外が接続されている場合, 接続されていない場合も含んでいる
        NN_RESULT_DO(ResultNpadDualConnected());
    }


    // NpadStyleSet を更新
    (*m_pNpads)[index1].UpdateNpad();
    (*m_pNpads)[index2].UpdateNpad();

    NN_RESULT_SUCCESS;
}

::nn::Result NpadAssignmentManager::SwapNpadAssignment(::nn::applet::AppletResourceUserId aruid, NpadIdType id1, NpadIdType id2) NN_NOEXCEPT
{
    // フォーカスあたっている ARUID 以外からの要求は何もしない
    if (aruid != m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetFocusedAppletResourceUserId())
    {
        NN_RESULT_SUCCESS;
    }

    // いずれかの id が Handheld だった場合は無視する
    if (id1 == NpadId::Handheld || id2 == NpadId::Handheld)
    {
        NN_RESULT_SUCCESS;
    }

    const auto& policy = m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetCurrentPolicy();
    if (policy.IsNpadIdTypeSupported(id1) == false ||
        policy.IsNpadIdTypeSupported(id2) == false)
    {
        NN_RESULT_THROW(ResultNpadNotInitialized());
    }

    // 変更予定の Npad についてコントローラーの接続状態が変化していないか確認
    CheckForDeviceDetach(id1, policy);
    CheckForDeviceDetach(id2, policy);

    struct DeviceInfoImpl
    {
        int index;
        NpadManager* pManager;
        system::DeviceTypeSet deviceType;
        NpadJoyAssignmentMode assignment;
        int abstractedPadCount;
        IAbstractedPad* pAbstractedPads[AbstractedPadCountMaxPerNpad];
    };
    DeviceInfoImpl deviceImpl[2];
    deviceImpl[0].index = GetIndexFromNpadIdType(id1);
    deviceImpl[1].index = GetIndexFromNpadIdType(id2);

    // 両方とも接続されているデバイスをディアクティベート
    for (int i = 0; i < 2; i++)
    {
        deviceImpl[i].pManager = &((*m_pNpads)[deviceImpl[i].index]);
        auto& holder = deviceImpl[i].pManager->GetAbstractedPadHolder();
        deviceImpl[i].deviceType = holder.GetDeviceType();
        deviceImpl[i].assignment = holder.GetJoyAssignmentMode();
        deviceImpl[i].abstractedPadCount = holder.GetAbstractedPads(deviceImpl[i].pAbstractedPads, NN_ARRAY_SIZE(deviceImpl[i].pAbstractedPads));
        holder.DetachAbstractedPadAll();
    }

    // Xcd のハンドルを入れ替えつつ再度アクティベート
    for (int i = 0; i < 2; i++)
    {
        int invertedIndex = 1 - i;

        deviceImpl[i].pManager->GetAbstractedPadHolder().SetJoyAssignmentMode(deviceImpl[invertedIndex].assignment);

        for (int j = 0; j < deviceImpl[invertedIndex].abstractedPadCount; j++)
        {
            AddAbstractedPadToNpadImpl(deviceImpl[i].index, deviceImpl[invertedIndex].pAbstractedPads[j], false);
        }

        // NpadStyleSet を更新
        deviceImpl[i].pManager->UpdateNpad();
    }

    NN_RESULT_SUCCESS;
}

void NpadAssignmentManager::CheckForDeviceDetach(NpadIdType id, const NpadAppletPolicy& policy) NN_NOEXCEPT
{
    NN_UNUSED(policy);

    auto& manager = GetNpadManager(*m_pNpads, id);

    auto& holder = manager.GetAbstractedPadHolder();
    auto disconnectedDeviceType = holder.GetDeviceType();

    // NpadManager に対してもともとデバイスが何も接続されていない場合は何もしない
    if (disconnectedDeviceType.IsAllOff() == true)
    {
        return;
    }

    // デバイスが切断されていれば、Holder から Detachする
    holder.DetachIfDisconnected();

    // 切断されたデバイスを更新
    disconnectedDeviceType = disconnectedDeviceType ^ holder.GetDeviceType();

    // ひとつも切断されていない場合は何もしない
    if (disconnectedDeviceType.IsAllOff())
    {
        return;
    }

    // 電池切れで切断された場合は、オーバーレイ通知を発行
    this->NotifyIfBatteryNoneDetach(manager, disconnectedDeviceType);

    // Bluetooth オフ中のジョイコン取り外しを検出
    if ((disconnectedDeviceType.Test<system::DeviceType::HandheldJoyLeft>() &&
        ::nn::xcd::IsDeviceAttachedOnRail(::nn::xcd::RailType_Left) == false) ||
        (disconnectedDeviceType.Test < system::DeviceType::HandheldJoyRight>() &&
            ::nn::xcd::IsDeviceAttachedOnRail(::nn::xcd::RailType_Right) == false))
    {
        if (IsBluetoothEnabled() == false)
        {
            m_pCommonResourceHolder->GetInterruptSceneNotifier().NotifyJoyDetachOnBluetoothOffEvent();
        }
    }

    // InputDetector にデバイスの切断を通知
    m_pCommonResourceHolder->GetInputDetectorManager().Notify(::nn::hid::system::InputSourceId::PadConnection::Mask);

    // NpadStyleSet を更新
    manager.UpdateNpad();
}

void NpadAssignmentManager::CheckForDeviceDetachOnAllNpads(const NpadAppletPolicy& policy) NN_NOEXCEPT
{
    // 切断・更新されたデバイスを探索
    for (int index = 0; index < NpadIdCountMax; index++)
    {
        CheckForDeviceDetach(SupportedNpadIdList[index], policy);
    }
}

void NpadAssignmentManager::CheckForDeviceAttach(const NpadAppletPolicy& policy) NN_NOEXCEPT
{
    // 新たに接続されたデバイスを処理
    for (int i = 0; i < m_AbstractedPadCount; i++)
    {
        auto pPad = m_pAbstractedPads[i];
        if (pPad->IsConnected() == false)
        {
            continue;
        }

        // NpadManager を探索して接続済みかどうかをチェック
        bool isConnected = false;
        for (auto& manager : *m_pNpads)
        {
            if (manager.GetAbstractedPadHolder().IsConnectedDevice(pPad->GetId()) == true)
            {
                // 接続済みのデバイス
                isConnected = true;
                continue;
            }
        }

        // 接続済みデバイスは処理しない
        if (isConnected == true)
        {
            continue;
        }

        // 新規で接続されたデバイスか確認
        if (pPad->GetInterfaceType() == system::InterfaceType_Rail)
        {
            auto npadManagerIndex = GetIndexFromNpadIdType(NpadId::Handheld);
            AddAbstractedPadToNpadImpl(npadManagerIndex, pPad, true);
            // NpadMangaer の内部状態の更新
            (*m_pNpads)[npadManagerIndex].UpdateNpad();
            m_pLastActiveControllerMonitor->SetLastActiveNpad(NpadId::Handheld);
        }
        else
        {
            if (IsNewAbstractedPadConnectionAllowed(pPad->GetId()) == false)
            {
                pPad->Detach();
            }
            else
            {
                NpadIdType id;
                int index;
                if (pPad->IsFixedNpadAssignmentEnabled(&id) == true)
                {
                    AddAbstractedPadToFixedNpad(id, pPad, true, policy);
                    index = GetIndexFromNpadIdType(id);
                    (*m_pNpads)[index].UpdateNpad();
                    m_pLastActiveControllerMonitor->SetLastActiveNpad((*m_pNpads)[index].GetNpadIdType());
                }
                else if (AddAbstractedPadToNewNpad(&index, pPad, true, policy) == true)
                {
                    // NpadMangaer の内部状態の更新
                    (*m_pNpads)[index].UpdateNpad();
                    m_pLastActiveControllerMonitor->SetLastActiveNpad((*m_pNpads)[index].GetNpadIdType());
                }
            }
        }
    }
}

void NpadAssignmentManager::UpdateAssignment(const NpadAppletPolicy& policy) NN_NOEXCEPT
{
    auto supportedStyle = policy.GetSupportedNpadStyleSet();

    // LR ボタン押しモードが有効
    if (policy.IsLrAssignmentModeEnabled() == true)
    {
        for (int i = 0; i < m_AbstractedPadCount; i++)
        {
            if (m_pAbstractedPads[i]->IsConnected() == false)
            {
                continue;
            }
            // SL, SR 押しによる割り当て変更のハンドリング
            if (supportedStyle.Test<NpadStyleJoyLeft>() || supportedStyle.Test<NpadStyleJoyRight>())
            {
                HandleSlSrAssignment(policy, i);
            }
            // L, R, ZL, ZR 押しによる割り当て変更のハンドリング
            if (supportedStyle.Test<NpadStyleFullKey>() || supportedStyle.Test<NpadStyleJoyDual>())
            {
                HandleLRAssignment(policy, i);
            }
        }
    }
}

void NpadAssignmentManager::HandleSlSrAssignment(const NpadAppletPolicy& policy,
                                              const int index) NN_NOEXCEPT
{
    NN_UNUSED(policy);

    auto state = m_pAbstractedPads[index]->GetPadState();

    // SL/SR の同時押しを検出
    if (state.buttons.Test<AbstractedPadButton::SL>() && state.buttons.Test<AbstractedPadButton::SR>())
    {
        NpadManager* pManager;
        if (GetNpadManagerFromAbstractedPadId(&pManager, m_pAbstractedPads[index]->GetId()).IsSuccess())
        {
            auto& holder = pManager->GetAbstractedPadHolder();
            auto deviceTypeOnSlSr = m_pAbstractedPads[index]->GetDeviceType();
            auto deviceTypeOnNpad = holder.GetDeviceType();
            holder.SetJoyAssignmentMode(NpadJoyAssignmentMode_Single);

            if (deviceTypeOnSlSr.Test<system::DeviceType::JoyConLeft>() &&
                deviceTypeOnNpad.Test<system::DeviceType::JoyConRight>() == true)
            {
                this->DisconnectImpl(pManager->GetNpadIdType(), system::DeviceType::JoyConRight::Mask);
            }
            else if (deviceTypeOnSlSr.Test<system::DeviceType::JoyConRight>() &&
                     deviceTypeOnNpad.Test<system::DeviceType::JoyConLeft>() == true)
            {
                this->DisconnectImpl(pManager->GetNpadIdType(),system::DeviceType::JoyConLeft::Mask);
            }
            // StyleSet を更新
            pManager->UpdateNpad();
        }
    }
}

void NpadAssignmentManager::HandleLRAssignment(const NpadAppletPolicy& policy,
                                              const int index) NN_NOEXCEPT
{
    NN_UNUSED(policy);

    auto state = m_pAbstractedPads[index]->GetPadState();

    // L/R/ZL/ZR の同時押しを検出
    bool isLorZlPressed = state.buttons.Test<AbstractedPadButton::L>() ||
                          state.buttons.Test<AbstractedPadButton::ZL>();
    bool isRorZrPressed = state.buttons.Test<AbstractedPadButton::R>() ||
                          state.buttons.Test<AbstractedPadButton::ZR>();
    if (isLorZlPressed || isRorZrPressed)
    {
        NpadManager* pFirstNpadManager;
        NpadManager* pSecondNpadManager;

        if (GetNpadManagerFromAbstractedPadId(&pFirstNpadManager, m_pAbstractedPads[index]->GetId()).IsSuccess())
        {
            auto deviceType = pFirstNpadManager->GetAbstractedPadHolder().GetDeviceType();
            // 1本持ちであることのチェック
            if ((isLorZlPressed && deviceType.Test<system::DeviceType::JoyConLeft>()  == true &&
                                   deviceType.Test<system::DeviceType::JoyConRight>() == false) ||
                (isRorZrPressed && deviceType.Test<system::DeviceType::JoyConRight>() == true &&
                                   deviceType.Test<system::DeviceType::JoyConLeft>()  == false))
            {
                // 対となるコントローラーを探索
                for (int j = index + 1; j < m_AbstractedPadCount; j++)
                {
                    if (m_pAbstractedPads[j]->IsConnected() == false)
                    {
                        continue;
                    }
                    auto statePair = m_pAbstractedPads[j]->GetPadState();
                    bool isLorZlPressedOnPair = statePair.buttons.Test<AbstractedPadButton::L>() ||
                                                statePair.buttons.Test<AbstractedPadButton::ZL>();
                    bool isRorZrPressedOnPair = statePair.buttons.Test<AbstractedPadButton::R>() ||
                                                statePair.buttons.Test<AbstractedPadButton::ZR>();
                    if ((isLorZlPressed && isRorZrPressedOnPair) ||
                        (isRorZrPressed && isLorZlPressedOnPair))
                    {
                        if (GetNpadManagerFromAbstractedPadId(&pSecondNpadManager, m_pAbstractedPads[j]->GetId()).IsSuccess())
                        {
                            auto deviceTypePair = pSecondNpadManager->GetAbstractedPadHolder().GetDeviceType();
                            // 1本持ちであることのチェック
                            if ((isLorZlPressedOnPair && deviceTypePair.Test<system::DeviceType::JoyConLeft>()  == true &&
                                                         deviceTypePair.Test<system::DeviceType::JoyConRight>() == false) ||
                                (isRorZrPressedOnPair && deviceTypePair.Test<system::DeviceType::JoyConRight>() == true &&
                                                         deviceTypePair.Test<system::DeviceType::JoyConLeft>()  == false))
                            {
                                // 見つかった2つの Npad をマージしてペアにする
                                auto firstNpadId = pFirstNpadManager->GetNpadIdType();
                                auto secondNpadId = pSecondNpadManager->GetNpadIdType();
                                if (firstNpadId < secondNpadId)
                                {
                                    MergeSingleJoyAsDualJoy(m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetFocusedAppletResourceUserId(), firstNpadId, secondNpadId);
                                }
                                else if (secondNpadId < firstNpadId)
                                {
                                    MergeSingleJoyAsDualJoy(m_pCommonResourceHolder->GetNpadAppletPolicyManager().GetFocusedAppletResourceUserId(), secondNpadId, firstNpadId);
                                }
                                break;
                            }
                        }
                    }
                }
            }
        }
    }
}

Result NpadAssignmentManager::GetNpadManagerFromAbstractedPadId(NpadManager** pOutManager, const AbstractedPadId& id) NN_NOEXCEPT
{
    for (auto& manager : *m_pNpads)
    {
        if (manager.GetAbstractedPadHolder().IsConnectedDevice(id))
        {
            *pOutManager = &manager;
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultNoNpadAssignedForXcdHandle());
}

void NpadAssignmentManager::AddAbstractedPadToFixedNpad(NpadIdType id,
                                                        IAbstractedPad* pPad,
                                                        bool isPhysicallyConnected,
                                                        const NpadAppletPolicy& policy) NN_NOEXCEPT
{
    NN_UNUSED(policy);

    auto connectedDevice = static_cast<system::DeviceTypeSet>(pPad->GetDeviceType());
    auto index = GetIndexFromNpadIdType(id);

    if (IsDeviceAcceptable(index, connectedDevice, policy))
    {
        this->DisconnectAllImpl(id);
        (*m_pNpads)[index].UpdateNpad();
        this->AddAbstractedPadToNpadImpl(index, pPad, isPhysicallyConnected);
    }
    else
    {
        pPad->Detach();
    }
}

bool NpadAssignmentManager::AddAbstractedPadToNewNpad(int* pOutIndex,
                                                   IAbstractedPad* pPad,
                                                   bool isPhysicallyConnected,
                                                   const NpadAppletPolicy& policy) NN_NOEXCEPT
{
    auto attached = false;
    int index = 0;

    // 新たに接続されたデバイスを空いているところに追加する
    for (index = 0; index < NpadIdCountMax; index++)
    {
        // Handheld と System::Other に対しては無線デバイスの割り当ては行わない
        if (IsWirelessAcceptableNpadId((*m_pNpads)[index].GetNpadIdType()) == false)
        {
            continue;
        }

        auto connectedDevice = static_cast<system::DeviceTypeSet>(pPad->GetDeviceType());

        if (IsDeviceAcceptable(index, connectedDevice, policy))
        {
            auto deviceType = (*m_pNpads)[index].GetAbstractedPadHolder().GetDeviceType();

            // LR 割り当てモード または SlSr1本接続モードが有効 かつ
            // 1本持ちが有効かつ
            // ジョイコンが有効の場合
            // かつ新規の物理接続の場合
            if ((policy.IsLrAssignmentModeEnabled() == true || policy.IsAssigningSingleOnSlSrPressEnabled() == true) &&
                (policy.GetSupportedNpadStyleSet() & (NpadStyleJoyLeft::Mask | NpadStyleJoyRight::Mask)).IsAnyOn() &&
                (connectedDevice & DeviceTypeMask_JoyCons).IsAnyOn() &&
                isPhysicallyConnected == true)
            {
                // 縦持ちが有効なら強制的に1本持ちに
                if (policy.GetNpadJoyHoldType() == NpadJoyHoldType_Vertical)
                {
                    if (deviceType.IsAnyOn())
                    {
                        // 別のジョイコンが接続済みの場合は2本持ちになってしまうのでスキップする
                        continue;
                    }
                    (*m_pNpads)[index].GetAbstractedPadHolder().SetJoyAssignmentMode(NpadJoyAssignmentMode_Single);
                    attached = true;
                    break;
                }
                // 横持ちが有効ならSL/SR が押されている場合に1本持ちに
                else if (policy.GetNpadJoyHoldType() == NpadJoyHoldType_Horizontal)
                {
                    // Sl ボタンと Sr ボタンが押された時間差を計算
                    ::nn::os::Tick slTick;
                    ::nn::os::Tick srTick;
                    auto resultSl = pPad->GetButtonTriggerElapsedTime(&slTick, AbstractedPadButton::SL::Mask);
                    auto resultSr = pPad->GetButtonTriggerElapsedTime(&srTick, AbstractedPadButton::SR::Mask);
                    if (resultSl && resultSr)
                    {
                        auto gap = (slTick > srTick) ? slTick - srTick : srTick - slTick;
                        if (gap.ToTimeSpan() < ::nn::TimeSpan::FromMilliSeconds(200))
                        {
                            if (deviceType.IsAnyOn())
                            {
                                // 別のジョイコンが接続済みの場合は2本持ちになってしまうのでスキップする
                                continue;
                            }
                            (*m_pNpads)[index].GetAbstractedPadHolder().SetJoyAssignmentMode(NpadJoyAssignmentMode_Single);
                            attached = true;
                            break;
                        }
                    }
                }
            }

            if (connectedDevice.Test<system::DeviceType::JoyConLeft>())
            {
                // 「何もつながっていない」または「2本セット割り当て かつ 右コンだけつながっている」、場合に追加できる
                if (deviceType.IsAllOff() == true ||
                    (deviceType == system::DeviceType::JoyConRight::Mask &&
                        (*m_pNpads)[index].GetAbstractedPadHolder().GetJoyAssignmentMode() == NpadJoyAssignmentMode_Dual)
                    )
                {
                    attached = true;
                    break;
                }
            }
            else if (connectedDevice.Test<system::DeviceType::JoyConRight>())
            {
                // 「何もつながっていない」または「2本セット割り当て かつ 左コンだけつながっている」、場合に追加できる
                if (deviceType.IsAllOff() == true ||
                    (deviceType == system::DeviceType::JoyConLeft::Mask &&
                        (*m_pNpads)[index].GetAbstractedPadHolder().GetJoyAssignmentMode() == NpadJoyAssignmentMode_Dual)
                    )
                {
                    attached = true;
                    break;
                }
            }
            else if (connectedDevice.Test<system::DeviceType::FullKeyController>() ||
                     connectedDevice.Test<system::DeviceType::Palma>())
            {
                // 何もつながっていない場合に追加できる
                if (deviceType.IsAllOff() == true)
                {
                    attached = true;
                    break;
                }
            }
            else
            {
                // 割り当てない
                break;
            }
        }
    }

    if (attached == false ||
        index >= NpadIdCountMax)
    {
        // 割り当てができなかった場合は Detach する
        pPad->Detach();
        return false;
    }

    // Npad に Xcd デバイスを追加
    AddAbstractedPadToNpadImpl(index, pPad, isPhysicallyConnected);
    if (pOutIndex != nullptr)
    {
        *pOutIndex = index;
    }
    return true;
}

void NpadAssignmentManager::AddAbstractedPadToNpadImpl(int index,
                                                IAbstractedPad* pPad,
                                                bool isPhysicallyConnected) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS(index, NpadIdCountMax);

    // Npad に対してデバイスを割り当てる
    (*m_pNpads)[index].GetAbstractedPadHolder().AttachAbstractedPad(pPad);
    // AbstractedPad にコントローラー番号を保持する
    switch (SupportedNpadIdList[index])
    {
    case NpadId::No1:
    case NpadId::No2:
    case NpadId::No3:
    case NpadId::No4:
    case NpadId::No5:
    case NpadId::No6:
    case NpadId::No7:
    case NpadId::No8:
        pPad->SetControllerNumber(SupportedNpadIdList[index] + 1);
        break;
    default:
        pPad->SetControllerNumber(0);
    }

    if (isPhysicallyConnected == true)
    {
        m_pCommonResourceHolder->GetInputDetectorManager().Notify(::nn::hid::system::InputSourceId::PadConnection::Mask);
        if (pPad->GetInterfaceType() == system::InterfaceType_Bluetooth || pPad->GetInterfaceType() == system::InterfaceType_Usb)
        {
            pPad->VibrationOnConnect();
        }
    }

    // 疑似接続またはレール接続の場合は、内部デバイスをリセットする
    if(isPhysicallyConnected == false ||
       (pPad->GetDeviceType() & DeviceTypeMask_HandheldJoyCons).IsAnyOn())
    {
        pPad->ResetInternalDeviceState();
    }
}

bool NpadAssignmentManager::IsDeviceAcceptable(int index, system::DeviceTypeSet deviceType, const NpadAppletPolicy& policy) NN_NOEXCEPT
{
    auto& manager = (*m_pNpads)[index];
    auto npadId = manager.GetNpadIdType();

    if (policy.IsAnySystemCommonPolicyEnabled() == true && policy.IsSystemCommonPolicyInFullMode() == false)
    {
        // 本体機能中は、No1 から No4, Handheld, system::other のみ新規接続を待ち受ける
        if (npadId != NpadId::No1 &&
            npadId != NpadId::No2 &&
            npadId != NpadId::No3 &&
            npadId != NpadId::No4 &&
            npadId != NpadId::Handheld &&
            npadId != system::NpadSystemId::Other)
        {
            // 接続を受けつけない
            return false;
        }
    }

    // 本体機能共通ポリシー以外は、Npad が有効になっているかどうかをチェック
    if (policy.IsNpadIdTypeSupported(npadId) == false)
    {
        // 接続をうけつけない
        return false;
    }

    // 有効な操作形態を確認
    {
        // フルコンは NpadStyleFullKey が有効な場合のみ接続
        if (deviceType.Test<system::DeviceType::FullKeyController>() == true)
        {
            if ((policy.GetSupportedNpadStyleSet() & StyleSetMask_FullKeySupportedStyles).IsAnyOn() == true)
            {
                return true;
            }
        }
        // 左コンは NpadStyleFullKey, JoyDual, JoyLeft のいずれかが有効な場合のみ接続
        else if (deviceType.Test<system::DeviceType::JoyConLeft>() == true)
        {
            if ((policy.GetSupportedNpadStyleSet() & StyleSetMask_JoyLeftSupportedStyles).IsAnyOn() == true)
            {
                return true;
            }
        }
        // 右コンは NpadStyleFullKey, JoyDual, JoyRight のいずれかが有効な場合のみ接続
        else if (deviceType.Test<system::DeviceType::JoyConRight>() == true)
        {
            if ((policy.GetSupportedNpadStyleSet() & StyleSetMask_JoyRightSupportedStyles).IsAnyOn() == true)
            {
                return true;
            }
        }
        // 携帯機左/右コン
        else if ((deviceType & DeviceTypeMask_HandheldJoyCons).IsAnyOn() == true)
        {
            // 携帯機操作が有効、かつ NpadId:Handheld
            if ((policy.GetSupportedNpadStyleSet() & StyleSetMask_HandheldLeftSupportedStyles).IsAnyOn() == true &&
                manager.GetNpadIdType() == NpadId::Handheld)
            {
                return true;
            }
        }
        // Palma は NpadStylePalma が有効な場合のみ接続
        else if (deviceType.Test<system::DeviceType::Palma>() == true)
        {
            if ((policy.GetSupportedNpadStyleSet() & StyleSetMask_PalmaSupportedStyles).IsAnyOn() == true)
            {
                return true;
            }
        }
    }
    return false;
}

void NpadAssignmentManager::DisconnectUnsupporedNpadIds(const NpadAppletPolicy& policy) NN_NOEXCEPT
{
    for (int i = 0; i < NpadIdCountMax; i++)
    {
        auto& manager = (*m_pNpads)[i];

        // 対応しないデバイスは切断
        if (policy.IsNpadIdTypeSupported(manager.GetNpadIdType()) == false)
        {
            this->DisconnectAllImpl(manager.GetNpadIdType());
        }
    }
}

void NpadAssignmentManager::DisconnectUnsupporedNpadStyle(const NpadAppletPolicy& policy) NN_NOEXCEPT
{
    auto supportedStyle = policy.GetSupportedNpadStyleSet();
    for (auto& manager : *m_pNpads)
    {
        // 有効な Style がひとつもない場合は切断
        if (supportedStyle.IsAllOff())
        {
            this->DisconnectAllImpl(manager.GetNpadIdType());
        }

        // FullKeyController
        if (manager.GetAbstractedPadHolder().GetDeviceType().Test<system::DeviceType::FullKeyController>())
        {
            // FullKeyStyle のみサポート
            if (supportedStyle.Test<NpadStyleFullKey>() == false)
            {
                this->DisconnectAllImpl(manager.GetNpadIdType());
            }
        }

        // Palma
        if (manager.GetAbstractedPadHolder().GetDeviceType().Test<system::DeviceType::Palma>())
        {
            // Palma のみサポート
            if (supportedStyle.Test<NpadStylePalma>() == false)
            {
                this->DisconnectAllImpl(manager.GetNpadIdType());
            }
        }
    }
}

void NpadAssignmentManager::UpdateJoyAssignmentMode(const NpadAppletPolicy& policy) NN_NOEXCEPT
{
    for (auto& manager : *m_pNpads)
    {
        if (manager.GetNpadIdType() == NpadId::Handheld || manager.GetNpadIdType() == system::NpadSystemId::Other)
        {
            continue;
        }
        auto& holder = manager.GetAbstractedPadHolder();
        if (policy.GetSupportedNpadStyleSet().Test<NpadStyleJoyLeft>() == false &&
            policy.GetSupportedNpadStyleSet().Test<NpadStyleJoyRight>() == false)
        {
            if (holder.GetJoyAssignmentMode() == NpadJoyAssignmentMode_Single)
            {
                // 割り当てモードが一致しない場合は切断
                this->DisconnectAllImpl(manager.GetNpadIdType());
                holder.SetJoyAssignmentMode(NpadJoyAssignmentMode_Dual);
            }
        }
        else if (policy.GetSupportedNpadStyleSet().Test<NpadStyleFullKey>() == false &&
            policy.GetSupportedNpadStyleSet().Test<NpadStyleJoyDual>() == false)
        {
            if (holder.GetJoyAssignmentMode() == NpadJoyAssignmentMode_Dual)
            {
                // 割り当てモードが一致しない場合は切断
                this->DisconnectAllImpl(manager.GetNpadIdType());
                holder.SetJoyAssignmentMode(NpadJoyAssignmentMode_Single);
            }
        }
    }
}

void NpadAssignmentManager::UpdateNpadAll(const NpadAppletPolicy& policy) NN_NOEXCEPT
{
    // 更新前に切断されたデバイスがないか検証
    CheckForDeviceDetachOnAllNpads(policy);

    for (auto& manager : *m_pNpads)
    {
        // NpadStyleSet を更新
        manager.UpdateNpad();
    }
}

// 新規の AbsttracatedPadId を接続できるかどうかをチェックする
bool NpadAssignmentManager::IsNewAbstractedPadConnectionAllowed(AbstractedPadId padId) NN_NOEXCEPT
{
    int connectedPhysicalDeviceCount = 0;

    // 接続済みのデバイスを処理
    for (int i = 0; i < m_AbstractedPadCount; i++)
    {
        auto pPads = m_pAbstractedPads[i];
        if (pPads->IsConnected() == true &&
            pPads->GetInterfaceType() != system::InterfaceType_Rail &&
            pPads->GetId() !=  padId)
        {
            connectedPhysicalDeviceCount++;
            // 9台以上接続されてしまう場合は、追加の割り当てはできない
            if (connectedPhysicalDeviceCount >= 8)
            {
                return false;
            }
        }
    }
    return true;
}

void NpadAssignmentManager::DisconnectImpl(const NpadIdType& id, const system::DeviceTypeSet& deviceType) NN_NOEXCEPT
{
    // 外部接続デバイスじゃない場合は、処理を行わない
    if (IsWirelessAcceptableNpadId(id) == false)
    {
        return;
    }

    auto& holder = GetNpadManager(*m_pNpads, id).GetAbstractedPadHolder();
    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = holder.GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    for (int i = 0; i < count; i++)
    {
        if ((pPads[i]->GetDeviceType() & deviceType).IsAnyOn())
        {
            auto abstractedPadId = pPads[i]->GetId();
            pPads[i]->Detach();
            holder.DetachAbstractedPad(abstractedPadId);
        }
    }
}

void NpadAssignmentManager::DisconnectAllImpl(const NpadIdType& id) NN_NOEXCEPT
{
    auto deviceType = system::DeviceType::FullKeyController::Mask |
                      system::DeviceType::Palma::Mask |
                      DeviceTypeMask_JoyCons;
    DisconnectImpl(id, deviceType);
}

void NpadAssignmentManager::NotifyIfBatteryNoneDetach(NpadManager& manager, const system::DeviceTypeSet& deviceType) NN_NOEXCEPT
{
    nn::ovln::format::ControllerDisconnectionData data;

    PowerInfoIndex powerIndex;
    if ((deviceType & DeviceTypeMask_JoyLefts).IsAnyOn())
    {
        data.controllerInfo.joyType = system::DeviceType::JoyConLeft::Mask;
        powerIndex = PowerInfoIndex_Left;
    }
    else if ((deviceType & DeviceTypeMask_JoyRights).IsAnyOn())
    {
        data.controllerInfo.joyType = system::DeviceType::JoyConRight::Mask;
        powerIndex = PowerInfoIndex_Right;
    }
    else if (deviceType.Test<system::DeviceType::FullKeyController>())
    {
        powerIndex = PowerInfoIndex_Standard;
    }
    else
    {
        return;
    }

    // 切断前の電池残量が Critical Low ならば、オーバーレイ通知を発行する
    auto powerInfo = manager.GetPowerInfoUpdater().GetPowerInfo(static_cast<int>(powerIndex));
    if (powerInfo.batteryLevel == system::BatteryLevel_None &&
        powerInfo.isCharging == false &&
        powerInfo.isPowered == false)
    {
        // 切断理由の生成
        data.reason = ::nn::ovln::format::ControllerDisconnectionReason_Battery;
        manager.GetDeviceManager().CreateNotifyControllerInfo(&data.controllerInfo);

        m_pCommonResourceHolder->GetOvlnSenderManager().Send(data);
    }
}

}}} // namespace nn::hid::detail
