﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadPalma.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_NpadAppletPolicy.h"
#include "hid_NpadUtilTypes.h"

namespace nn { namespace hid { namespace detail {

void NpadAppletPolicy::Activate() NN_NOEXCEPT
{
    for (auto& systemEvent : m_StyleUpdateEvents)
    {
        ::nn::os::CreateSystemEvent(&systemEvent, ::nn::os::EventClearMode_ManualClear, true);
    }
}

void NpadAppletPolicy::Deactivate() NN_NOEXCEPT
{
    for (auto& systemEvent : m_StyleUpdateEvents)
    {
        ::nn::os::DestroySystemEvent(&systemEvent);
    }
}

void NpadAppletPolicy::Reset() NN_NOEXCEPT
{
    m_JoyHoldType = NpadJoyHoldType_Vertical;
    m_IsLrAssignmentModeEnabled = false;
    m_IsSystemCommonPolicyEnabled = false;
    m_IsSystemCommonPolicyFullEnabled = false;
    m_HandheldActivationMode = NpadHandheldActivationMode_Dual;

    // 初期化直後はすべての Style は有効にしておく
    m_IsNpadStyleInitialized = false;
    m_SupportedStyles.Set();
    m_IsNpadJoyHoldTypeInitialized = false;

    // 初期化直後はすべての Id を有効にしておく
    this->SetSupportedNpadIdType(SupportedNpadIdList, NpadIdCountMax);

    for (auto& isEnabled : m_IsHomeButtonEnabled)
    {
        isEnabled = true;
    }
}

void NpadAppletPolicy::SetSystemCommonPolicy(bool isFull) NN_NOEXCEPT
{
    m_SupportedStyles = StyleSetMask_SystemCommonStyles;
    m_IsNpadStyleInitialized = true;
    m_IsNpadJoyHoldTypeInitialized = true;

    NN_ABORT_UNLESS_EQUAL(sizeof(m_SupportedId), sizeof(SupportedNpadIdList));
    std::memcpy(m_SupportedId, SupportedNpadIdList, sizeof(m_SupportedId));
    m_SupportedIdCount = NpadIdCountMax;

    m_IsLrAssignmentModeEnabled = false;
    if (isFull == true)
    {
        m_IsSystemCommonPolicyFullEnabled = true;
    }
    else
    {
        m_IsSystemCommonPolicyEnabled = true;
    }
    m_HandheldActivationMode = NpadHandheldActivationMode_Dual;

    for (auto& isEnabled : m_IsHomeButtonEnabled)
    {
        isEnabled = true;
    }
}

void NpadAppletPolicy::SetNpadJoyHoldType(NpadJoyHoldType holdType) NN_NOEXCEPT
{
    m_IsNpadJoyHoldTypeInitialized = true;
    m_JoyHoldType = holdType;
}

void NpadAppletPolicy::SetSupportedNpadStyleSet(NpadStyleSet style) NN_NOEXCEPT
{
    m_SupportedStyles = style;
    m_IsNpadStyleInitialized = true;
    // Style が初期化されたときに持ち方も初期化されたとみなす
    m_IsNpadJoyHoldTypeInitialized = true;
}

Result NpadAppletPolicy::SetSupportedNpadIdType(const NpadIdType* pIds, int count) NN_NOEXCEPT
{
    if (count > 0)
    {
        NN_SDK_REQUIRES_NOT_NULL(pIds);
    }

    NN_RESULT_THROW_UNLESS(count <= NpadIdCountMax, ResultNpadInvalidNpadIdTypeCount());

    for (int i = 0; i < count; i++)
    {
        m_SupportedId[i] = pIds[i];
    }
    m_SupportedIdCount = count;

    NN_RESULT_SUCCESS;
}

bool NpadAppletPolicy::IsNpadIdTypeSupported(NpadIdType id) const NN_NOEXCEPT
{
    for (int i = 0; i < m_SupportedIdCount; i++)
    {
        if (m_SupportedId[i] == id)
        {
            return true;
        }
    }

    return false;
}

void NpadAppletPolicy::SetIsLrAssignmentModeEnabled(bool enabled) NN_NOEXCEPT
{
    m_IsLrAssignmentModeEnabled = enabled;
}

bool NpadAppletPolicy::IsLrAssignmentModeEnabled() const NN_NOEXCEPT
{
    return m_IsLrAssignmentModeEnabled;
}

void NpadAppletPolicy::SetAssigningSingleOnSlSrPressMode(bool enabled) NN_NOEXCEPT
{
    m_IsAssigningSingleOnSlSrPressEnabled = enabled;
}

bool NpadAppletPolicy::IsAssigningSingleOnSlSrPressEnabled() const NN_NOEXCEPT
{
    return m_IsAssigningSingleOnSlSrPressEnabled;
}

void NpadAppletPolicy::SetHomeButtonEnabled(bool enabled, NpadIdType id) NN_NOEXCEPT
{
    const auto Index = GetIndexFromNpadIdType(id);
    m_IsHomeButtonEnabled[Index] = enabled;
}

bool NpadAppletPolicy::IsHomeButtonEnabled(NpadIdType id) const NN_NOEXCEPT
{
    const auto Index = GetIndexFromNpadIdType(id);
    return m_IsHomeButtonEnabled[Index];
}

Result NpadAppletPolicy::AcquireStyleUpdateEventHandle(::nn::os::NativeHandle* pOutValue, NpadIdType id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    *pOutValue = ::nn::os::GetReadableHandleOfSystemEvent(&m_StyleUpdateEvents[GetIndexFromNpadIdType(id)]);
    ::nn::os::SignalSystemEvent(&m_StyleUpdateEvents[GetIndexFromNpadIdType(id)]);

    NN_RESULT_SUCCESS;
}

void NpadAppletPolicy::SignalStyleUpdateEvent(NpadIdType id) NN_NOEXCEPT
{
    ::nn::os::SignalSystemEvent(&m_StyleUpdateEvents[GetIndexFromNpadIdType(id)]);
}

}}} // namespace nn::hid::detail
