﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_NpadAbstractedPadHolder.h"

namespace nn { namespace hid { namespace detail {

NpadAbstractedPadHolder::NpadAbstractedPadHolder() NN_NOEXCEPT
    : m_AbstractedPadCount(0)
    , m_JoyAssignmentMode(NpadJoyAssignmentMode_Dual)
{
    // 何もしない
}

Result NpadAbstractedPadHolder::AttachAbstractedPad(IAbstractedPad* pPad) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPad);
    // TODO : Add Internal Result
    NN_RESULT_THROW_UNLESS(m_AbstractedPadCount < AbstractedPadCountMaxPerNpad, ResultNpadControllerNotConnected());

    for (int i = 0; i < m_AbstractedPadCount; i++)
    {
        if (m_pAbstractedPads[i]->GetDeviceType() == pPad->GetDeviceType())
        {
            // TODO : Add Internal Result
            NN_RESULT_THROW(ResultNpadControllerNotConnected());
        }
    }
    m_pAbstractedPads[m_AbstractedPadCount] = pPad;
    m_AbstractedPadCount++;

    NN_RESULT_SUCCESS;
}

void NpadAbstractedPadHolder::DetachAbstractedPad(const AbstractedPadId& id) NN_NOEXCEPT
{
    if (m_AbstractedPadCount == 0 ||
        id == AbstractedPadId())
    {
        return;
    }

    for (int i = 0; i < m_AbstractedPadCount; i++)
    {
        if (m_pAbstractedPads[i]->GetId() == id)
        {
            PopAbstractedPad(i);
            return;
        }
    }
}

void NpadAbstractedPadHolder::DetachAbstractedPadAll() NN_NOEXCEPT
{
    if (m_AbstractedPadCount == 0)
    {
        return;
    }

    while(m_AbstractedPadCount > 0)
    {
        PopAbstractedPad(0);
    }
}

IAbstractedPad* NpadAbstractedPadHolder::DetachAbstractedPad(const system::DeviceTypeSet& deviceType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(deviceType.CountPopulation() == 1);
    if (m_AbstractedPadCount == 0)
    {
        return nullptr;
    }

    for (int i = 0; i < m_AbstractedPadCount; i++)
    {
        if ((m_pAbstractedPads[i]->GetDeviceType() & deviceType).IsAnyOn())
        {
            auto pPad = m_pAbstractedPads[i];
            PopAbstractedPad(i);
            return pPad;
        }
    }
    return nullptr;
}

void NpadAbstractedPadHolder::DetachIfDisconnected() NN_NOEXCEPT
{
    int index = 0;
    while(index < m_AbstractedPadCount)
    {
        if (m_pAbstractedPads[index]->IsConnected() == false)
        {
            PopAbstractedPad(index);
        }
        else
        {
            ++index;
        }
    }
}

bool NpadAbstractedPadHolder::IsConnectedDevice(const AbstractedPadId& id) NN_NOEXCEPT
{
    if (m_AbstractedPadCount == 0 ||
        id == AbstractedPadId())
    {
        return false;
    }

    for (int i = 0; i < m_AbstractedPadCount; i++)
    {
        if (m_pAbstractedPads[i]->GetId() == id)
        {
            return true;
        }
    }

    return false;
}

int NpadAbstractedPadHolder::GetAbstractedPads(IAbstractedPad** pOutPad, int size) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutPad);

    if (pOutPad == nullptr)
    {
        return m_AbstractedPadCount;
    }

    int outSize = std::min(size, m_AbstractedPadCount);
    for (int i = 0; i < outSize; i++)
    {
        pOutPad[i] = m_pAbstractedPads[i];
    }
    return outSize;
}

void NpadAbstractedPadHolder::SetJoyAssignmentMode(const NpadJoyAssignmentMode& mode) NN_NOEXCEPT
{
    m_JoyAssignmentMode = mode;
}

NpadJoyAssignmentMode NpadAbstractedPadHolder::GetJoyAssignmentMode() NN_NOEXCEPT
{
    return m_JoyAssignmentMode;
}

system::DeviceTypeSet NpadAbstractedPadHolder::GetDeviceType() NN_NOEXCEPT
{
    auto deviceType = system::DeviceTypeSet();
    for (int i = 0; i < m_AbstractedPadCount; i++)
    {
        deviceType |= m_pAbstractedPads[i]->GetDeviceType();
    }
    return deviceType;
}

void NpadAbstractedPadHolder::PopAbstractedPad(int index) NN_NOEXCEPT
{
    if (index < 0 ||
        index >= m_AbstractedPadCount ||
        m_AbstractedPadCount <= 0)
    {
        return;
    }

    // 前づめする
    for (int i = index + 1; i < m_AbstractedPadCount; i++)
    {
        m_pAbstractedPads[i - 1] = m_pAbstractedPads[i];
    }

    --m_AbstractedPadCount;
    m_pAbstractedPads[m_AbstractedPadCount] = nullptr;
}

}}} // namespace nn::hid::detail
