﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceUserId.h"
#include "hid_HidDebugServer.h"
#include "hid_HidServer.h"
#include "hid_MouseImpl.h"
#include "hid_MouseLifo.h"
#include "hid_SharedMemoryAccessor.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_StaticObject.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< Mouse の共有メモリへのアクセスを扱うクラスです。
class MouseSharedMemoryAccessor final : public SharedMemoryAccessor
{
private:
    //!< Mouse の LIFO
    ::std::atomic<MouseLifo*> m_pLifo;

public:
    MouseSharedMemoryAccessor() NN_NOEXCEPT
        : m_pLifo(nullptr)
    {
        this->SetResultActivationUpperLimitOver(
            ::nn::hid::ResultMouseActivationUpperLimitOver());
        this->SetResultDeactivationLowerLimitOver(
            ::nn::hid::ResultMouseDeactivationLowerLimitOver());
    }

    virtual ~MouseSharedMemoryAccessor() NN_NOEXCEPT NN_OVERRIDE
    {
        // 何もしない
    }

    //!< Mouse の LIFO を返します。
    const MouseLifo* GetLifo() const NN_NOEXCEPT
    {
        return m_pLifo;
    }

protected:
    virtual ::nn::Result Attach(SharedMemoryFormat* ptr) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(ptr);
        ::nn::sf::SharedPointer<IHidServer> pHidServer;
        NN_RESULT_DO(CreateHidServerProxy(&pHidServer));
        NN_RESULT_DO(pHidServer->ActivateMouse(GetAppletResourceUserId()));
        m_pLifo = &::nn::util::Get(ptr->mouse.lifo);
        NN_RESULT_SUCCESS;
    }

    virtual ::nn::Result Detach() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
        NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
        NN_RESULT_DO(pHidDebugServer->DeactivateMouse());
        m_pLifo = nullptr;
        NN_RESULT_SUCCESS;
    }
};

//!< Mouse の共有メモリアクセッサを返します。
MouseSharedMemoryAccessor& GetMouseSharedMemoryAccessor() NN_NOEXCEPT
{
    return StaticObject<MouseSharedMemoryAccessor>::Get();
}

} // namespace

::nn::Result InitializeMouse() NN_NOEXCEPT
{
    MouseSharedMemoryAccessor& accessor = GetMouseSharedMemoryAccessor();
    NN_RESULT_DO(accessor.SetAppletResourceUserId(GetAppletResourceUserId()));
    NN_RESULT_DO(accessor.Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeMouse() NN_NOEXCEPT
{
    NN_RESULT_DO(GetMouseSharedMemoryAccessor().Deactivate());
    NN_RESULT_SUCCESS;
}

::nn::Result GetMouseState(MouseState* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultMouseNullStateBuffer());
    const MouseLifo* pLifo = GetMouseSharedMemoryAccessor().GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultMouseNotInitialized());
    int count = pLifo->Read(pOutValue, 1);
    NN_SDK_REQUIRES_EQUAL(count, 1);
    NN_UNUSED(count);
    NN_RESULT_SUCCESS;
}

::nn::Result GetMouseStates(
    int* pOutCount, MouseState outStates[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr, ResultMouseNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultMouseInvalidStateCount());
    const MouseLifo* pLifo = GetMouseSharedMemoryAccessor().GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultMouseNotInitialized());
    NN_SDK_REQUIRES(!pLifo->IsEmpty());
    *pOutCount = pLifo->Read(outStates, count);
    NN_RESULT_SUCCESS;
}

::nn::Result SetMouseAutoPilotState(
    const ::nn::hid::debug::MouseAutoPilotState& value) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->SetMouseAutoPilotState(value));
    NN_RESULT_SUCCESS;
}

::nn::Result UnsetMouseAutoPilotState() NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->UnsetMouseAutoPilotState());
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
