﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/util/util_BitFlagSet.h>

#include "hid_ActivationCount.h"
#include "hid_IMouseDriver.h"
#include "hid_Point.h"

namespace nn { namespace hid { namespace detail {

struct MouseReport final
{
    ::nn::util::BitFlagSet<32> buttons; //!< ボタンの状態です。
    int32_t deltaX;                     //!< カーソルの x 座標の移動差分です。
    int32_t deltaY;                     //!< カーソルの y 座標の移動差分です。
    int32_t wheelDelta;                 //!< ホイールの回転差分です。
};

class MouseDriver final : public IMouseDriver
{
    NN_DISALLOW_COPY(MouseDriver);
    NN_DISALLOW_MOVE(MouseDriver);

public:
    //!< ポートの数
    static const size_t PortCount = 8;

private:
    //!< このインスタンスからドライバがアクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< サンプリング番号
    int64_t m_SamplingNumber;

    //!< マウスの座標
    Point m_Position;

    //!< アタッチ状態にあるポートの数
    ::std::atomic<int> m_Count;

    //!< ボタンの押下状態
    ::std::atomic<uint32_t> m_Buttons[PortCount];

    //!< X 軸方向の変位
    ::std::atomic<int> m_DisplacementX;

    //!< Y 軸方向の変位
    ::std::atomic<int> m_DisplacementY;

    //!< ホイールの回転差分
    ::std::atomic<int> m_WheelDelta;

public:
    MouseDriver() NN_NOEXCEPT;

    virtual ~MouseDriver() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Activate() NN_NOEXCEPT NN_OVERRIDE;

    virtual ::nn::Result Deactivate() NN_NOEXCEPT NN_OVERRIDE;

    virtual void GetState(MouseState* pOutValue) NN_NOEXCEPT NN_OVERRIDE;

    //!< アタッチ処理を行います。
    void Attach(size_t port);

    //!< デタッチ処理を行います。
    void Detach(size_t port);

    //!< レポートを設定します。
    void SetReport(size_t port, const MouseReport& report) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
