﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitPack.h>

#include "hid_MouseDriver-os.horizon.h"
#include "hid_MouseStateUtility.h"
#include "hid_Rectangle.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< ボタンのフィールド定義です。
struct ButtonField final
{
    typedef ::nn::util::BitPack<uint32_t, ButtonField>::Field<0, 1, bool>
            Left;
    typedef ::nn::util::BitPack<uint32_t, ButtonField>::Field<1, 1, bool>
            Right;
    typedef ::nn::util::BitPack<uint32_t, ButtonField>::Field<2, 1, bool>
            Middle;
    typedef ::nn::util::BitPack<uint32_t, ButtonField>::Field<3, 1, bool>
            Back;
    typedef ::nn::util::BitPack<uint32_t, ButtonField>::Field<4, 1, bool>
            Forward;
};

//!< ボタンの集合を扱う型です。
typedef ::nn::util::BitPack<uint32_t, ButtonField> ButtonPack;

//!< ホイールの回転の重みです。
const int WheelWeight = 120;

} // namespace

MouseDriver::MouseDriver() NN_NOEXCEPT
    : m_ActivationCount()
    , m_SamplingNumber(0)
    , m_Position()
    , m_Count()
    , m_Buttons()
    , m_DisplacementX()
    , m_DisplacementY()
    , m_WheelDelta()
{
    // 何もしない
}

MouseDriver::~MouseDriver() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result MouseDriver::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultMouseDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        m_Position.x = MouseRectangle.width  / 2 + MouseRectangle.x;
        m_Position.y = MouseRectangle.height / 2 + MouseRectangle.y;

        m_Count = 0;

        for (size_t i = 0; i < PortCount; ++i)
        {
            m_Buttons[i] = 0;
        }

        m_DisplacementX = 0;

        m_DisplacementY = 0;
    }

    // このインスタンスからアクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result MouseDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultMouseDriverDeactivationLowerLimitOver());

    // このインスタンスからアクティブ化した回数をデクリメント
    --m_ActivationCount;

    NN_RESULT_SUCCESS;
}

void MouseDriver::GetState(MouseState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    // マウスの入力状態を初期化
    *pOutValue = MouseState();

    pOutValue->samplingNumber = m_SamplingNumber++;

    if (m_Count == 0)
    {
        // マウスの状態を取得できない限りは上位システムへ移譲
        pOutValue->attributes.Set<MouseAttribute::Transferable>();

        m_Position.x = MouseRectangle.width  / 2 + MouseRectangle.x;
        m_Position.y = MouseRectangle.height / 2 + MouseRectangle.y;
    }
    else
    {
        const int deltaX = m_DisplacementX.exchange(0);
        const int deltaY = m_DisplacementY.exchange(0);
        const int wheelDelta = m_WheelDelta.exchange(0) * WheelWeight;

        m_Position.x += deltaX;
        m_Position.y += deltaY;
        m_Position = MouseRectangle.Clamp(m_Position);

        ButtonPack buttonPack = {};

        for (int8_t buttons : m_Buttons)
        {
            buttonPack.storage |= buttons;
        }

        pOutValue->attributes.Set<MouseAttribute::IsConnected>();

        pOutValue->buttons.Set<
            MouseButton::Left>(buttonPack.Get<ButtonField::Left>());
        pOutValue->buttons.Set<
            MouseButton::Right>(buttonPack.Get<ButtonField::Right>());
        pOutValue->buttons.Set<
            MouseButton::Middle>(buttonPack.Get<ButtonField::Middle>());
        pOutValue->buttons.Set<
            MouseButton::Forward>(buttonPack.Get<ButtonField::Forward>());
        pOutValue->buttons.Set<
            MouseButton::Back>(buttonPack.Get<ButtonField::Back>());

        pOutValue->x = static_cast<int32_t>(m_Position.x);
        pOutValue->y = static_cast<int32_t>(m_Position.y);
        pOutValue->deltaX = static_cast<int32_t>(deltaX);
        pOutValue->deltaY = static_cast<int32_t>(deltaY);
        pOutValue->wheelDelta = static_cast<int32_t>(wheelDelta);
    }
}

void MouseDriver::Attach(size_t port)
{
    NN_SDK_REQUIRES_RANGE(port, 0u, static_cast<size_t>(PortCount));
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    NN_UNUSED(port);

    const int count = ++m_Count;

    NN_SDK_ASSERT_MINMAX(count, 1, static_cast<int>(PortCount));

    NN_UNUSED(count);
}

void MouseDriver::Detach(size_t port)
{
    NN_SDK_REQUIRES_RANGE(port, 0u, static_cast<size_t>(PortCount));

    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    const int count = --m_Count;

    NN_SDK_ASSERT_MINMAX(count, 0, static_cast<int>(PortCount - 1));

    NN_UNUSED(count);

    m_Buttons[port] = 0;
}

void MouseDriver::SetReport(size_t port, const MouseReport& report) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(port, 0u, static_cast<size_t>(PortCount));
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    m_Buttons[port] = report.buttons._storage[0];

    m_DisplacementX += report.deltaX;

    m_DisplacementY += report.deltaY;

    m_WheelDelta += report.wheelDelta;
}


}}} // namespace nn::hid::detail
