﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_IHidSystemServer.sfdl.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/debug/hid_Keyboard.h>
#include <nn/hid/system/hid_Keyboard.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceUserId.h"
#include "hid_KeyboardImpl.h"
#include "hid_KeyboardLifo.h"
#include "hid_HidDebugServer.h"
#include "hid_HidServer.h"
#include "hid_HidSystemServer.h"
#include "hid_SharedMemoryAccessor.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_StaticObject.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< Keyboard の共有メモリへのアクセスを扱うクラスです。
class KeyboardSharedMemoryAccessor final : public SharedMemoryAccessor
{
private:
    //!< Keyboard の LIFO
    ::std::atomic<KeyboardLifo*> m_pLifo;

public:
    KeyboardSharedMemoryAccessor() NN_NOEXCEPT
        : m_pLifo(nullptr)
    {
        this->SetResultActivationUpperLimitOver(
            ::nn::hid::ResultKeyboardActivationUpperLimitOver());
        this->SetResultDeactivationLowerLimitOver(
            ::nn::hid::ResultKeyboardDeactivationLowerLimitOver());
    }

    virtual ~KeyboardSharedMemoryAccessor() NN_NOEXCEPT NN_OVERRIDE
    {
        // 何もしない
    }

    //!< Keyboard の LIFO を返します。
    const KeyboardLifo* GetLifo() const NN_NOEXCEPT
    {
        return m_pLifo;
    }

protected:
    virtual ::nn::Result Attach(SharedMemoryFormat* ptr) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(ptr);
        ::nn::sf::SharedPointer<IHidServer> pHidServer;
        NN_RESULT_DO(CreateHidServerProxy(&pHidServer));
        NN_RESULT_DO(pHidServer->ActivateKeyboard(GetAppletResourceUserId()));
        m_pLifo = &::nn::util::Get(ptr->keyboard.lifo);
        NN_RESULT_SUCCESS;
    }

    virtual ::nn::Result Detach() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
        NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
        NN_RESULT_DO(pHidDebugServer->DeactivateKeyboard());
        m_pLifo = nullptr;
        NN_RESULT_SUCCESS;
    }
};

//!< Keyboard の共有メモリアクセッサを返します。
KeyboardSharedMemoryAccessor& GetKeyboardSharedMemoryAccessor() NN_NOEXCEPT
{
    return StaticObject<KeyboardSharedMemoryAccessor>::Get();
}

} // namespace

::nn::Result InitializeKeyboard() NN_NOEXCEPT
{
    KeyboardSharedMemoryAccessor& accessor = GetKeyboardSharedMemoryAccessor();
    NN_RESULT_DO(accessor.SetAppletResourceUserId(GetAppletResourceUserId()));
    NN_RESULT_DO(accessor.Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeKeyboard() NN_NOEXCEPT
{
    NN_RESULT_DO(GetKeyboardSharedMemoryAccessor().Deactivate());
    NN_RESULT_SUCCESS;
}

::nn::Result GetKeyboardState(KeyboardState* pOutValue) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(pOutValue != nullptr,
                           ResultKeyboardNullStateBuffer());
    const KeyboardLifo* pLifo = GetKeyboardSharedMemoryAccessor().GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultKeyboardNotInitialized());
    int count = pLifo->Read(pOutValue, 1);
    NN_SDK_REQUIRES_EQUAL(count, 1);
    NN_UNUSED(count);
    NN_RESULT_SUCCESS;
}

::nn::Result GetKeyboardStates(
    int* pOutCount, KeyboardState outStates[], int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr,
                           ResultKeyboardNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultKeyboardInvalidStateCount());
    const KeyboardLifo* pLifo = GetKeyboardSharedMemoryAccessor().GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultKeyboardNotInitialized());
    NN_SDK_REQUIRES(!pLifo->IsEmpty());
    *pOutCount = pLifo->Read(outStates, count);
    NN_RESULT_SUCCESS;
}

::nn::Result SetKeyboardAutoPilotState(
    const ::nn::hid::debug::KeyboardAutoPilotState& value) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->SetKeyboardAutoPilotState(value));
    NN_RESULT_SUCCESS;
}

::nn::Result UnsetKeyboardAutoPilotState() NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->UnsetKeyboardAutoPilotState());
    NN_RESULT_SUCCESS;
}

::nn::Result SendKeyboardLockKeyEvent(
    ::nn::hid::system::KeyboardLockKeyEventSet value) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));
    NN_RESULT_DO(pHidSystemServer->SendKeyboardLockKeyEvent(value));
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
