﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/hid_JoyXpad.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_HidServer.h"
#include "hid_JoyXpadLifo.h"
#include "hid_JoyXpadSixAxisSensorImpl.h"
#include "hid_RingLifoHolder.h"
#include "hid_Session.h"
#include "hid_StaticObject.h"
#include "hid_XpadId.h"
#include "hid_XpadSessionManager.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< 基本的な構成を持つ Xpad のセッションを扱うためのクラスです。
class JoyXpadSixAxisSensorSession final : public Session
{
    NN_DISALLOW_COPY(JoyXpadSixAxisSensorSession);
    NN_DISALLOW_MOVE(JoyXpadSixAxisSensorSession);

private:
    //!< hid サービスのセッション
    ::nn::sf::SharedPointer<IHidServer> m_Session;

    //!< 基本的な構成を持つ Xpad の ID
    JoyXpadId m_XpadId;

    //!< 基本的な構成を持つ Xpad の LIFO
    RingLifoHolder<JoyXpadSixAxisSensorLifoPair> m_LifoHolder;

public:
    explicit JoyXpadSixAxisSensorSession(const JoyXpadId& xpadId) NN_NOEXCEPT
        : m_XpadId(xpadId)
    {
        // 何もしない
    }

    virtual ~JoyXpadSixAxisSensorSession() NN_NOEXCEPT NN_OVERRIDE { /* 何もしない */ }

    //!< Xpad ID の型
    typedef JoyXpadId XpadIdType;

    //!< 基本的な構成を持つ Xpad の LIFO を返します。
    const JoyXpadSixAxisSensorLifo* GetRightLifo() const NN_NOEXCEPT
    {
        return &m_LifoHolder.GetLifo()->GetRightLifo();
    }
    const JoyXpadSixAxisSensorLifo* GetLeftLifo() const NN_NOEXCEPT
    {
        return &m_LifoHolder.GetLifo()->GetLeftLifo();
    }

    void AttachSharedMemoryIfNotMapped() NN_NOEXCEPT
    {
        if(!IsSharedMemoryMapped())
        {
            Activate();
        }
    }

protected:
    virtual ::nn::Result ThrowResultActivationUpperLimitOver(
        ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(ResultSixAxisSensorActivationUpperLimitOver());
    }

    virtual ::nn::Result ThrowResultDeactivationLowerLimitOver(
        ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(ResultJoyXpadDeactivationLowerLimitOver());
    }

    virtual ::nn::Result ThrowResultAllocationFailed() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(ResultJoyXpadLifoAllocationFailed());
    }

    virtual ::nn::Result CreateProxy() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(CreateHidServerProxy(&m_Session));
    }

    virtual void DestroyProxy() NN_NOEXCEPT NN_OVERRIDE
    {
        m_Session.Reset();
    }

    // TORIAEZU: 左右の Start/Stop をまとめて設定。本来は独立に設定したい。
    virtual ::nn::Result ActivateService() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(m_Session->ActivateJoySixAxisSensor(m_XpadId));
    }

    virtual ::nn::Result DeactivateService() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_THROW(m_Session->DeactivateJoySixAxisSensor(m_XpadId));
    }

    virtual ::nn::Result GetSharedMemoryHandle(::nn::sf::NativeHandle* outValue
                                               ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);
        NN_RESULT_THROW(m_Session->GetJoySixAxisSensorLifoHandle(outValue, m_XpadId));
    }

    virtual bool IsSharedMemoryMapped() NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_LifoHolder.GetLifo() != nullptr);
    }

    virtual void AttachSharedMemory(::nn::os::NativeHandle handle,
                                    bool managed) NN_NOEXCEPT NN_OVERRIDE
    {
        m_LifoHolder.Attach(handle, managed);
    }

    virtual void FinalizeSharedMemory() NN_NOEXCEPT NN_OVERRIDE
    {
        m_LifoHolder.Finalize();
    }
};

XpadSessionManager<JoyXpadSixAxisSensorSession>& GetXpadSessionManager() NN_NOEXCEPT
{
    return StaticObject<XpadSessionManager<JoyXpadSixAxisSensorSession>>::Get();
}

} // namespace

::nn::Result GetSixAxisSensorRightStates(int* outGotCount,
                                         SixAxisSensorState* outStates,
                                         int count,
                                         const JoyXpadId& xpadId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outGotCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr,
                           ResultSixAxisSensorNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultSixAxisSensorInvalidStateCount());
    JoyXpadSixAxisSensorSession *pSession = GetXpadSessionManager().GetSession(xpadId);
    NN_RESULT_THROW_UNLESS(pSession != nullptr,
                           ResultJoyXpadPlayerNumberOutOfRange());
    pSession->AttachSharedMemoryIfNotMapped();
    const JoyXpadSixAxisSensorLifo* pLifo = pSession->GetRightLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultSixAxisSensorNotInitialized());
    *outGotCount = pLifo->Read(outStates, count);
    NN_SDK_REQUIRES(!pLifo->IsEmpty());
    NN_RESULT_SUCCESS;
}

::nn::Result GetSixAxisSensorLeftStates(int* outGotCount,
                                        SixAxisSensorState* outStates,
                                        int count,
                                        const JoyXpadId& xpadId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outGotCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr,
                           ResultSixAxisSensorNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultSixAxisSensorInvalidStateCount());
    JoyXpadSixAxisSensorSession *pSession = GetXpadSessionManager().GetSession(xpadId);
    NN_RESULT_THROW_UNLESS(pSession != nullptr,
                           ResultJoyXpadPlayerNumberOutOfRange());
    pSession->AttachSharedMemoryIfNotMapped();
    const JoyXpadSixAxisSensorLifo* pLifo = pSession->GetLeftLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultSixAxisSensorNotInitialized());
    *outGotCount = pLifo->Read(outStates, count);
    NN_SDK_REQUIRES(!pLifo->IsEmpty());
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
