﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_JoyXpad.h>
#include <nn/os/os_NativeHandle.h>

#include "hid_ActivationCount.h"
#include "hid_IXpadDriver.h"
#include "hid_RingLifoHolder.h"
#include "hid_JoyXpadLifo.h"

namespace nn { namespace hid { namespace detail {

//!< JoyXpad の入力状態の解決を行う実装クラスです
class JoyXpadManager final
{
    NN_DISALLOW_COPY(JoyXpadManager);
    NN_DISALLOW_MOVE(JoyXpadManager);

private:
    //!< JoyXpad がアクティブ化された回数
    ActivationCount m_JoyXpadActivationCount;

    //!< JoyXpad のサンプリング番号
    int64_t m_JoyXpadRightSamplingNumber;
    int64_t m_JoyXpadLeftSamplingNumber;

    //!< JoyXpad の LIFO
    RingLifoHolder<JoyXpadLifoPair> m_JoyXpadLifoHolder;

    //!< JoyXpad の入力状態
    JoyXpadState m_JoyXpadRightState;
    JoyXpadState m_JoyXpadLeftState;

    //!< 6軸センサーがアクティブ化された回数
    ActivationCount m_SixAxisSensorActivationCount;

    //!< 6軸センサーのサンプリング番号
    int64_t m_RightSixAxisSensorSamplingNumber;
    int64_t m_LeftSixAxisSensorSamplingNumber;

    //!< 6軸センサー向け LIFO
    RingLifoHolder<JoyXpadSixAxisSensorLifoPair> m_SixAxisSensorLifoHolder;

    //!< 6軸センサーの入力状態
    SixAxisSensorState m_RightSixAxisSensorState;
    SixAxisSensorState m_LeftSixAxisSensorState;

public:
    JoyXpadManager() NN_NOEXCEPT;

    ~JoyXpadManager() NN_NOEXCEPT;

    //!< JoyXpad についてのみマネージャをアクティブ化します。
    ::nn::Result ActivateJoyXpad() NN_NOEXCEPT;

    //!< JoyXpad についてのみマネージャを非アクティブ化します。
    ::nn::Result DeactivateJoyXpad() NN_NOEXCEPT;

    //!< JoyXpad の LIFO のハンドルを取得します。
    ::nn::Result GetJoyXpadLifoHandle(::nn::os::NativeHandle* outValue
                                        ) NN_NOEXCEPT;

    //!< 6軸センサーについてのみマネージャをアクティブ化します。
    ::nn::Result ActivateSixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーについてのみマネージャを非アクティブ化します。
    ::nn::Result DeactivateSixAxisSensor() NN_NOEXCEPT;

    //!< 6軸センサーの LIFO のハンドルを取得します。
    ::nn::Result GetSixAxisSensorLifoHandle(::nn::os::NativeHandle* outValue
                                           ) NN_NOEXCEPT;
    //!< 入力状態を解決します。
    void Sample() NN_NOEXCEPT;

private:
    //!< 入力状態を JoyXpad の LIFO に追加します。
    void AppendStateToJoyXpadLifo() NN_NOEXCEPT;

    //!< 入力状態を 6軸センサー の LIFO に追加します。
    void AppendStateToSixAxisSensorLifo() NN_NOEXCEPT;

};

}}} // namespace nn::hid::detail
