﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_IHidSystemServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/debug/hid_HomeButton.h>
#include <nn/hid/system/hid_HomeButton.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_HidDebugServer.h"
#include "hid_HidSystemServer.h"
#include "hid_HomeButtonImpl.h"
#include "hid_HomeButtonLifo.h"
#include "hid_SharedMemoryAccessor.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_StaticObject.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< ホームボタンの共有メモリへのアクセスを扱うクラスです。
class HomeButtonSharedMemoryAccessor final : public SharedMemoryAccessor
{
private:
    //!< ホームボタンの LIFO
    ::std::atomic<HomeButtonLifo*> m_pLifo;

public:
    HomeButtonSharedMemoryAccessor() NN_NOEXCEPT
        : m_pLifo(nullptr)
    {
        this->SetResultActivationUpperLimitOver(
            ::nn::hid::ResultHomeButtonActivationUpperLimitOver());
        this->SetResultDeactivationLowerLimitOver(
            ::nn::hid::ResultHomeButtonDeactivationLowerLimitOver());
    }

    virtual ~HomeButtonSharedMemoryAccessor() NN_NOEXCEPT NN_OVERRIDE
    {
        // 何もしない
    }

    //!< ホームボタンの LIFO を返します。
    const HomeButtonLifo* GetLifo() const NN_NOEXCEPT
    {
        return m_pLifo;
    }

protected:
    virtual ::nn::Result Attach(SharedMemoryFormat* ptr) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(ptr);
        ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
        NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));
        NN_RESULT_DO(pHidSystemServer->ActivateHomeButton(
                         ::nn::applet::AppletResourceUserId::GetInvalidId()));
        m_pLifo = &::nn::util::Get(ptr->homeButton.lifo);
        NN_RESULT_SUCCESS;
    }

    virtual ::nn::Result Detach() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
        NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
        NN_RESULT_DO(pHidDebugServer->DeactivateHomeButton());
        m_pLifo = nullptr;
        NN_RESULT_SUCCESS;
    }
};

//!< ホームボタンの共有メモリアクセッサを返します。
HomeButtonSharedMemoryAccessor& GetHomeButtonSharedMemoryAccessor() NN_NOEXCEPT
{
    return StaticObject<HomeButtonSharedMemoryAccessor>::Get();
}

} // namespace

::nn::Result BindHomeButtonEvent(::nn::os::SystemEventType* pEvent,
                                 ::nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));

    ::nn::sf::NativeHandle handle;
    NN_RESULT_DO(
        pHidSystemServer->AcquireHomeButtonEventHandle(
            &handle, ::nn::applet::AppletResourceUserId::GetInvalidId()));

    ::nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                                handle.GetOsHandle(),
                                                handle.IsManaged(),
                                                clearMode);

    // ハンドルの管理権を放棄
    handle.Detach();

    NN_RESULT_SUCCESS;
}

::nn::Result InitializeHomeButton() NN_NOEXCEPT
{
    HomeButtonSharedMemoryAccessor&
        accessor = GetHomeButtonSharedMemoryAccessor();
    NN_RESULT_DO(accessor.SetAppletResourceUserId(
                     ::nn::applet::AppletResourceUserId::GetInvalidId()));
    NN_RESULT_DO(accessor.Activate());
    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeHomeButton() NN_NOEXCEPT
{
    NN_RESULT_DO(GetHomeButtonSharedMemoryAccessor().Deactivate());
    NN_RESULT_SUCCESS;
}

::nn::Result GetHomeButtonStates(int* pOutCount,
                                 ::nn::hid::system::HomeButtonState outStates[],
                                 int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr,
                           ResultHomeButtonNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultHomeButtonInvalidStateCount());
    const HomeButtonLifo* pLifo = GetHomeButtonSharedMemoryAccessor().GetLifo();
    NN_RESULT_THROW_UNLESS(pLifo != nullptr, ResultHomeButtonNotInitialized());
    NN_SDK_REQUIRES(!pLifo->IsEmpty());
    *pOutCount = pLifo->Read(outStates, count);
    NN_RESULT_SUCCESS;
}

::nn::Result SetHomeButtonAutoPilotState(
    const ::nn::hid::debug::HomeButtonAutoPilotState& value) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->SetHomeButtonAutoPilotState(value));
    NN_RESULT_SUCCESS;
}

::nn::Result UnsetHomeButtonAutoPilotState() NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidDebugServer> pHidDebugServer;
    NN_RESULT_DO(CreateHidDebugServerProxy(&pHidDebugServer));
    NN_RESULT_DO(pHidDebugServer->UnsetHomeButtonAutoPilotState());
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
