﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_IHidSystemServer.sfdl.h>
#include <nn/hid/system/hid_Keyboard.h>
#include <nn/hid/system/hid_NpadCommon.h>
#include <nn/hid/system/hid_NpadSystemExt.h>
#include <nn/hid/system/hid_AudioControl.h>
#include <nn/util/util_MathTypes.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

namespace nn { namespace hid { namespace detail {

//!< hid システム機能サービスを扱うクラスです。
class HidSystemServer final
{
    NN_DISALLOW_COPY(HidSystemServer);
    NN_DISALLOW_MOVE(HidSystemServer);

public:
    HidSystemServer() NN_NOEXCEPT;

    //!< Keyboard のロックキーイベントを送信します。
    ::nn::Result SendKeyboardLockKeyEvent(
        ::nn::hid::system::KeyboardLockKeyEventSet value) NN_NOEXCEPT;

    //!< ホームボタンの入力通知のハンドルを確保します。
    ::nn::Result AcquireHomeButtonEventHandle(
        ::nn::sf::Out<nn::sf::NativeHandle> outValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ホームボタンをアクティブ化します。
    ::nn::Result ActivateHomeButton(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< スリープボタンの入力通知のハンドルを確保します。
    ::nn::Result AcquireSleepButtonEventHandle(
        ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< スリープボタンをアクティブ化します。
    ::nn::Result ActivateSleepButton(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 撮影ボタンの入力通知のハンドルを確保します。
    ::nn::Result AcquireCaptureButtonEventHandle(
        ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 撮影ボタンをアクティブ化します。
    ::nn::Result ActivateCaptureButton(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID を設定します。
    ::nn::Result SetAppletResourceUserId(
        const ::nn::applet::AppletResourceUserId& value) NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID を登録します。
    ::nn::Result RegisterAppletResourceUserId(
        const ::nn::applet::AppletResourceUserId& aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID の登録を解除します。
    ::nn::Result UnregisterAppletResourceUserId(
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    //!< HID 入力の提供を行うか否かを設定します。
    ::nn::Result EnableAppletToGetInput(
        const ::nn::applet::AppletResourceUserId& aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< 振動の制御権を持つアプレットリソースユーザ ID を設定します。
    ::nn::Result SetAruidValidForVibration(
        ::nn::applet::AppletResourceUserId aruid, bool enable) NN_NOEXCEPT;

    //!< 6 軸センサーの提供を行うか否かを設定します。
    ::nn::Result EnableAppletToGetSixAxisSensor(
        const ::nn::applet::AppletResourceUserId& aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< 携帯機モードでのみ有効な Hid デバイスを有効にします。
    ::nn::Result EnableHandheldHids() NN_NOEXCEPT;

    //!< 携帯機モードでのみ有効な Hid デバイスを無効にします。
    ::nn::Result DisableHandheldHids() NN_NOEXCEPT;

    //!< 振動のマスターボリュームを設定します。
    ::nn::Result SetVibrationMasterVolume(float masterVolume) NN_NOEXCEPT;

    //!< 振動のマスターボリュームを取得します。
    ::nn::Result GetVibrationMasterVolume(::nn::sf::Out<float> outValue) NN_NOEXCEPT;

    //!< 一時的な振動強制ON期間を開始します。
    ::nn::Result BeginPermitVibrationSession(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 一時的な振動強制ON期間を停止します。
    ::nn::Result EndPermitVibrationSession() NN_NOEXCEPT;

    //!< Nfc デバイスの状態変化通知イベントをバインドします。
    ::nn::Result AcquireNfcDeviceUpdateEventHandle(::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT;

    //!< 有効な Nfc デバイスにを搭載した Npad を取得します。
    ::nn::Result GetNpadsWithNfc(::nn::sf::Out<int64_t> outCount, ::nn::sf::OutArray<NpadIdType> outIds) NN_NOEXCEPT;

    //!< NpadId に対して割り当てられている Nfc のデバイスハンドルを取得します
    ::nn::Result GetXcdHandleForNpadWithNfc(::nn::sf::Out<::nn::xcd::DeviceHandle> outHandle, NpadIdType id, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Nfc デバイスのアクティブ化イベントをバインドします。
    ::nn::Result AcquireNfcActivateEventHandle(::nn::sf::Out<::nn::sf::NativeHandle> outValue, NpadIdType npadId) NN_NOEXCEPT;

    //!< Nfc デバイスのアクティブ状態を制御します。
    ::nn::Result ActivateNfc(NpadIdType npadId, bool activate, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Nfc がアクティブかどうかを返します。
    ::nn::Result IsNfcActivated(::nn::sf::Out<bool> outIsActivated, NpadIdType id) NN_NOEXCEPT;

    //!< IrSensor デバイスの状態変化イベントをバインドします。
    ::nn::Result AcquireIrSensorEventHandle(::nn::sf::Out<::nn::sf::NativeHandle> outValue, NpadIdType npadId) NN_NOEXCEPT;

    //!< IrSensor デバイスのアクティブ状態を制御します。
    ::nn::Result ActivateIrSensor(NpadIdType npadId, bool activate, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< NpadId に対して割り当てられている IrSensor のデバイスハンドルを取得します
    ::nn::Result GetXcdHandleForNpadWithIrSensor(NpadIdType npadId, ::nn::sf::Out<::nn::xcd::DeviceHandle> outHandle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< NpadId に対する IrSensor の状態を取得します
    ::nn::Result GetIrSensorState(NpadIdType id, ::nn::sf::Out<int64_t> outValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Npad のシステム機能をアクティベートします。
    nn::Result ActivateNpadSystem(NpadIdType id) NN_NOEXCEPT;

    //!< 本体機能共通の Npad の割り当てポリシーを設定します。
    ::nn::Result ApplyNpadSystemCommonPolicy(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 最大数の接続を許可する本体機能共通の Npad の割り当てポリシーを設定します。
    ::nn::Result ApplyNpadSystemCommonPolicyFull(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 最後に操作されたコントローラーを取得します。
    ::nn::Result GetLastActiveNpad(::nn::sf::Out<NpadIdType> outValue) NN_NOEXCEPT;

    //!< 本体機能の操作形態の情報を取得します
    ::nn::Result GetNpadSystemExtStyle(::nn::sf::Out<int64_t> outMainStyle,
                                       ::nn::sf::Out<int64_t> outSubStyle,
                                       NpadIdType id) NN_NOEXCEPT;

    //!< SlSr ボタンが押されて接続された際に、1本持ちとして接続することを許可します
    ::nn::Result EnableAssigningSingleOnSlSrPress(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< SlSr ボタンが押されて接続された際に、1本持ちとして接続する機能を無効化します
    ::nn::Result DisableAssigningSingleOnSlSrPress(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< プレイヤーランプを点滅させるデバイスを設定します
    ::nn::Result SetNpadPlayerLedBlinkingDevice(::nn::applet::AppletResourceUserId aruid, NpadIdType npadId, system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT;

    //!< Switch Pro Controller のグリップ色を取得します
    ::nn::Result GetNpadFullKeyGripColor(::nn::sf::Out<util::Color4u8Type> outLeftGrip,
                                         ::nn::sf::Out<util::Color4u8Type> outRightGrip,
                                         NpadIdType id) NN_NOEXCEPT;

    //!< アプリがビルドされた SDK に応じてマスクされた有効な NpadStyleSet を取得します。
    ::nn::Result GetMaskedSupportedNpadStyleSet(::nn::sf::Out<NpadStyleSet> outValue, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< UniquePad をアクティベートします。
    nn::Result ActivateUniquePad(::nn::applet::AppletResourceUserId aruid,
                                 system::UniquePadId id) NN_NOEXCEPT;

    //!< UniquePad の接続状態変化通知イベントをバインドします。
    ::nn::Result AcquireUniquePadConnectionEventHandle(::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT;

    //!< UniquePadId を取得します。
    ::nn::Result GetUniquePadIds(::nn::sf::Out<::std::int64_t> outGotCount,
                                 const ::nn::sf::OutArray<::nn::hid::system::UniquePadId>& outUniquePadIds) NN_NOEXCEPT;

    //!< UniquePad の Bluetooth Address を取得します。
    ::nn::Result GetUniquePadBluetoothAddress(::nn::sf::Out<::nn::bluetooth::Address> outGotAddress,
                                              system::UniquePadId id) NN_NOEXCEPT;

    //!< UniquePad を切断します。
    ::nn::Result DisconnectUniquePad(system::UniquePadId id) NN_NOEXCEPT;

    //!< UniquePad のデバイス種別を取得します。
    ::nn::Result GetUniquePadType(::nn::sf::Out<int64_t> outValue, system::UniquePadId id) NN_NOEXCEPT;

    //!< UniquePad の接続インターフェースを取得します。
    ::nn::Result GetUniquePadInterface(::nn::sf::Out<int64_t> outValue,
                                       system::UniquePadId id) NN_NOEXCEPT;

    //!< UniquePad のシリアル番号を取得します。
    ::nn::Result GetUniquePadSerialNumber(::nn::sf::Out<system::UniquePadSerialNumber> outValue,
                                          system::UniquePadId id) NN_NOEXCEPT;

    //!< UniquePad のコントローラー番号を取得します。
    ::nn::Result GetUniquePadControllerNumber(::nn::sf::Out<int64_t> outValue,
                                              system::UniquePadId id) NN_NOEXCEPT;

    //!< UniqueSixAxisSensorHandle を列挙します。
    ::nn::Result ListSixAxisSensorHandles(::nn::sf::Out<std::int64_t> outGotCount,
                                          const ::nn::sf::OutArray<::nn::hid::system::UniqueSixAxisSensorHandle>& outUniqueSixAxisSensorHandles,
                                          ::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< 6 軸センサーのユーザーキャリブレーション機能のサポート有無を取得します。
    ::nn::Result IsSixAxisSensorUserCalibrationSupported(::nn::sf::Out<bool> outIsSupported,
                                                         ::nn::hid::system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT;

    //!< 工程出荷時の 6 軸センサーのキャリブレーション値を書き込みます。
    ::nn::Result ResetSixAxisSensorCalibrationValues(::nn::hid::system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT;

    //!< 6 軸センサーのユーザーキャリブレーションを開始します。
    ::nn::Result StartSixAxisSensorUserCalibration(::nn::hid::system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT;

    //!< 6 軸センサーのユーザーキャリブレーション処理をキャンセルします。
    ::nn::Result CancelSixAxisSensorUserCalibration(::nn::hid::system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT;

    //!< 6軸センサーのユーザーキャリブレーションの状況を取得します。
    ::nn::Result GetSixAxisSensorUserCalibrationStage(::nn::sf::Out<int64_t> outValue,
                                                      system::UniqueSixAxisSensorHandle handle) NN_NOEXCEPT;

    //!< 6 軸センサーの高精度ユーザーキャリブレーション機能のサポート有無を取得します。
    ::nn::Result IsSixAxisSensorAccurateUserCalibrationSupported(::nn::sf::Out<bool> outIsSupported,
                                                                 ::nn::hid::system::UniqueSixAxisSensorHandle handle,
                                                                 ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 6 軸センサーの高精度ユーザーキャリブレーションを開始します。
    ::nn::Result StartSixAxisSensorAccurateUserCalibration(::nn::hid::system::UniqueSixAxisSensorHandle handle,
                                                           ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 6 軸センサーの高精度ユーザーキャリブレーション処理をキャンセルします。
    ::nn::Result CancelSixAxisSensorAccurateUserCalibration(::nn::hid::system::UniqueSixAxisSensorHandle handle,
                                                            ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 6軸センサーの高精度ユーザーキャリブレーションの状況を取得します。
    ::nn::Result GetSixAxisSensorAccurateUserCalibrationState(::nn::sf::Out<::nn::hid::system::SixAxisSensorAccurateUserCalibrationState> outValue,
                                                              ::nn::hid::system::UniqueSixAxisSensorHandle handle,
                                                              ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アナログスティックの手動キャリブレーション処理を開始します。
    ::nn::Result StartAnalogStickManualCalibration(system::UniquePadId id, int64_t position) NN_NOEXCEPT;

    //!< 現在のアナログスティックのキャリブレーション処理をやり直します。
    ::nn::Result RetryCurrentAnalogStickManualCalibrationStage(system::UniquePadId id, int64_t position) NN_NOEXCEPT;

    //!< 進行中アナログスティックのキャリブレーション処理をキャンセルします。
    ::nn::Result CancelAnalogStickManualCalibration(system::UniquePadId id, int64_t position) NN_NOEXCEPT;

    //!< アナログスティックのキャリブレーション値を初期値に戻します。
    ::nn::Result ResetAnalogStickManualCalibration(system::UniquePadId id, int64_t position) NN_NOEXCEPT;

    //!< アナログスティックの状態を取得します。
    ::nn::Result GetAnalogStickState(::nn::sf::Out<AnalogStickState> outValue,
                                     system::UniquePadId id,
                                     int64_t position) NN_NOEXCEPT;

    //!< アナログスティックのキャリブレーションの状態を取得します。
    ::nn::Result GetAnalogStickManualCalibrationStage(::nn::sf::Out<int64_t> outValue,
                                                      system::UniquePadId id,
                                                      int64_t position) NN_NOEXCEPT;

    //!< アナログスティックの状態を取得します。
    ::nn::Result IsAnalogStickButtonPressed(::nn::sf::Out<bool> outValue,
                                            system::UniquePadId id,
                                            int64_t position) NN_NOEXCEPT;

    //!< キャリブレーション中のアナログスティックがリリースポジションにある場合は true を返します。
    ::nn::Result IsAnalogStickInReleasePosition(::nn::sf::Out<bool> outValue,
                                                system::UniquePadId id,
                                                int64_t position) NN_NOEXCEPT;

    //!< キャリブレーション中のアナログスティックが外周にある場合は true を返します。
    ::nn::Result IsAnalogStickInCircumference(::nn::sf::Out<bool> outValue,
                                              system::UniquePadId id,
                                              int64_t position) NN_NOEXCEPT;

    //!< InputDetector をアクティベートします。
    nn::Result ActivateInputDetector(nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< InputDetector に入力変化を通知します。
    nn::Result NotifyInputDetector(nn::hid::system::InputSourceIdSet id) NN_NOEXCEPT;

    //!< Bluetooth オフ中のジョイコン取り外し通知のハンドルを確保します。
    ::nn::Result AcquireJoyDetachOnBluetoothOffEventHandle (
        ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< プレイレポート用のコントローラーの操作状態の情報に更新があった際に通知をうけるイベントハンドルを取得します
    ::nn::Result AcquirePlayReportControllerUsageUpdateEvent(::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT;

    //!< プレイレポート用のコントローラーの操作状態の情報を取得します。
    ::nn::Result GetPlayReportControllerUsages(::nn::sf::Out<std::int64_t> outGotCount,
                                               const ::nn::sf::OutArray<::nn::hid::system::PlayReportControllerUsage>& outValues) NN_NOEXCEPT;

    //!< プレイレポート用のコントローラーの登録情報に更新があった際に通知をうけるイベントハンドルを取得します
    ::nn::Result AcquirePlayReportRegisteredDeviceUpdateEvent(::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT;

    //!< プレイレポート用のコントローラーの登録情報を取得します。【互換性維持用】
    ::nn::Result GetRegisteredDevicesOld(::nn::sf::Out<std::int64_t> outGotCount,
                                      const ::nn::sf::OutArray<::nn::hid::system::RegisteredDeviceOld>& outValues) NN_NOEXCEPT;

    //!< プレイレポート用のコントローラーの登録情報を取得します。
    ::nn::Result GetRegisteredDevices(::nn::sf::Out<std::int64_t> outGotCount,
                                      const ::nn::sf::OutArray<::nn::hid::system::RegisteredDevice>& outValues) NN_NOEXCEPT;

    //!< 本体からコントローラーに対して接続可能なコントローラーの登録情報を取得します。
    ::nn::Result GetConnectableRegisteredDevices(::nn::sf::Out<std::int64_t> outGotCount,
                                                 const ::nn::sf::OutArray<::nn::hid::system::RegisteredDevice>& outValues) NN_NOEXCEPT;

    //!< コントローラーへの接続処理がタイムアウトした際の通知をうけるイベントハンドルを取得します
    ::nn::Result AcquireConnectionTriggerTimeoutEvent(::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT;

    //!< コントローラーに対して接続を試みます
    ::nn::Result SendConnectionTrigger(nn::bluetooth::Address address) NN_NOEXCEPT;

    //!< Bluetooth 接続可能なコントローラーの数を取得します
    ::nn::Result GetAllowedBluetoothLinksCount(::nn::sf::Out<std::int64_t> outGotCount) NN_NOEXCEPT;

    //!< コントローラーが新たに登録された際に通知をうけるイベントハンドルを取得します
    ::nn::Result AcquireDeviceRegisteredEventForControllerSupport(::nn::sf::Out<::nn::sf::NativeHandle> outValue) NN_NOEXCEPT;

    //!< Pro Controller 有線 USB 通信を有効または無効にして、設定値を本体 NAND に保存します。
    ::nn::Result EnableUsbFullKeyController(bool enabled) NN_NOEXCEPT;

    //!< Pro Controller 有線 USB 通信が有効になっているか取得します。
    ::nn::Result IsUsbFullKeyControllerEnabled(::nn::sf::Out<bool> outIsEnabled) NN_NOEXCEPT;

    //!< コントローラーが USB 接続状態を取得します。
    ::nn::Result IsUsbConnected(::nn::sf::Out<bool> outIsConnected, ::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< 据置モード中の携帯機コントローラーの操作が行われたかどうか
    ::nn::Result IsHandheldButtonPressedOnConsoleMode(::nn::sf::Out<bool> outValue) NN_NOEXCEPT;

    //!< コントローラーファームウェア更新機能を初期化します。
    ::nn::Result InitializeFirmwareUpdate() NN_NOEXCEPT;

    //!< 指定されたコントローラーについてファームウェア更新を開始します。
    ::nn::Result StartFirmwareUpdate(::nn::sf::Out<::nn::hid::system::FirmwareUpdateDeviceHandle> outHandle,
                                     ::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< 実行中のコントローラーファームウェア更新を中断します。
    ::nn::Result AbortFirmwareUpdate() NN_NOEXCEPT;

    //!< 指定されたコントローラーの Firmware Version を取得します
    ::nn::Result GetFirmwareVersion(::nn::sf::Out<::nn::hid::system::FirmwareVersion> outValue,
                                    ::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< 指定されたコントローラーの更新先 Firmware Version を取得します
    ::nn::Result GetAvailableFirmwareVersion(::nn::sf::Out<::nn::hid::system::FirmwareVersion> outValue,
                                             ::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< 指定されたコントローラーのファームウェア更新有無を判定します。
    ::nn::Result IsFirmwareUpdateAvailable(::nn::sf::Out<bool> outIsAvailable,
                                           ::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< 指定されたコントローラーに対してファームウェア更新が要求されているかを判定します。
    ::nn::Result CheckFirmwareUpdateRequired(::nn::sf::Out<int64_t> outReason,
                                             ::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< ファームウェア更新の経過状況を取得します。
    ::nn::Result GetFirmwareUpdateState(::nn::sf::Out<::nn::hid::system::FirmwareUpdateState> outState,
                                        ::nn::hid::system::FirmwareUpdateDeviceHandle handle) NN_NOEXCEPT;

    //!< Hotfix 起因のファームウェア更新をスキップします。
    ::nn::Result SetFirmwareHotfixUpdateSkipEnabled(bool isEnabled) NN_NOEXCEPT;

    //!< Npad に割り当てられた UniquePad を取得します。
    ::nn::Result GetUniquePadsFromNpad(::nn::sf::Out<int64_t> outGotCount, ::nn::sf::OutArray<::nn::hid::system::UniquePadId> outIds, NpadIdType id) NN_NOEXCEPT;

    //!< オーディオコントロールをアクティブ化します。
    ::nn::Result ActivateAudioControl() NN_NOEXCEPT;

    //!< オーディオコントロールの入力通知のハンドルを確保します。
    ::nn::Result AcquireAudioControlEventHandle(
        ::nn::sf::Out<nn::sf::NativeHandle> outValue) NN_NOEXCEPT;

    ::nn::Result GetAudioControlStates(
        ::nn::sf::Out<int64_t> outCount,
        const nn::sf::OutArray<system::AudioControlState>& outStates) NN_NOEXCEPT;

    //!< オーディオコントロールを非アクティブ化します。
    ::nn::Result DeactivateAudioControl() NN_NOEXCEPT;

    //!< Hidbus の system 向けサービスオブジェクトを取得します。
    nn::Result GetHidbusSystemServiceObject(nn::sf::Out<nn::sf::SharedPointer<nn::hidbus::IHidbusSystemServer>> outService) NN_NOEXCEPT;

};

//!< hid システム機能サービスのプロクシを作成します。
::nn::Result CreateHidSystemServerProxy(
    ::nn::sf::SharedPointer<IHidSystemServer>* pOutValue) NN_NOEXCEPT;

}}} // namespace nn::hid::detail
