﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/hid_IHidDebugServer.sfdl.h>
#include <nn/hid/hid_JoyXpad.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/hid_SevenSixAxisSensor.h>
#include <nn/hid/hid_Xpad.h>
#include <nn/hid/debug/hid_CaptureButton.h>
#include <nn/hid/debug/hid_DebugPad.h>
#include <nn/hid/debug/hid_HomeButton.h>
#include <nn/hid/debug/hid_Keyboard.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/hid/debug/hid_SleepButton.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/hid/system/hid_UniquePad.h>
#include <nn/hid/detail/hid_PadTypes.h>
#include <nn/hid/detail/hid_SensorTypes.h>
#include <nn/sf/sf_ISharedObject.h>

namespace nn { namespace hid { namespace detail {

//!< hid デバッグ機能サービスを扱うクラスです。
class HidDebugServer final
{
    NN_DISALLOW_COPY(HidDebugServer);
    NN_DISALLOW_MOVE(HidDebugServer);

public:
    HidDebugServer() NN_NOEXCEPT;

    //!< DebugPad を非アクティブ化します。
    ::nn::Result DeactivateDebugPad() NN_NOEXCEPT;

    //!< DebugPad に自動操作状態を設定します。
    ::nn::Result SetDebugPadAutoPilotState(
        const ::nn::hid::debug::DebugPadAutoPilotState& value) NN_NOEXCEPT;

    //!< DebugPad の自動操作状態を解除します。
    ::nn::Result UnsetDebugPadAutoPilotState() NN_NOEXCEPT;

    //!< Keyboard を非アクティブ化します。
    ::nn::Result DeactivateKeyboard() NN_NOEXCEPT;

    //!< Keyboard に自動操作状態を設定します。
    ::nn::Result SetKeyboardAutoPilotState(
        const ::nn::hid::debug::KeyboardAutoPilotState& value) NN_NOEXCEPT;

    //!< Keyboard の自動操作状態を解除します。
    ::nn::Result UnsetKeyboardAutoPilotState() NN_NOEXCEPT;

    //!< Mouse を非アクティブ化します。
    ::nn::Result DeactivateMouse() NN_NOEXCEPT;

    //!< Mouse に自動操作状態を設定します。
    ::nn::Result SetMouseAutoPilotState(
        const ::nn::hid::debug::MouseAutoPilotState& value) NN_NOEXCEPT;

    //!< Mouse の自動操作状態を解除します。
    ::nn::Result UnsetMouseAutoPilotState() NN_NOEXCEPT;

    //!< TouchScreen を非アクティブ化します。
    ::nn::Result DeactivateTouchScreen() NN_NOEXCEPT;

    //!< TouchScreen に自動操作状態を設定します。
    ::nn::Result SetTouchScreenAutoPilotState(
        const ::nn::sf::InArray<::nn::hid::TouchState>& touches) NN_NOEXCEPT;

    //!< TouchScreen の自動操作状態を解除します。
    ::nn::Result UnsetTouchScreenAutoPilotState() NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad を非アクティブ化します。
    ::nn::Result DeactivateXpad(::nn::hid::BasicXpadId xpadId) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad に自動操作状態を設定します。
    ::nn::Result SetXpadAutoPilotState(
        ::nn::hid::BasicXpadId xpadId,
        const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT;

    //!< 基本的な構成を持つ Xpad の自動操作状態を解除します。
    ::nn::Result UnsetXpadAutoPilotState(::nn::hid::BasicXpadId xpadId
                                         ) NN_NOEXCEPT;

    //!< 基本的な構成を持つ仮想コントローラーに自動操作状態を設定します。
    ::nn::Result SetAutoPilotVirtualPadState(
        const ::nn::hid::debug::AbstractedVirtualPadId& virtualPadId,
        const ::nn::hid::debug::AbstractedPadState& value) NN_NOEXCEPT;

    //!< 基本的な構成を持つ仮想コントローラーの自動操作状態を解除します。
    ::nn::Result UnsetAutoPilotVirtualPadState(
        const ::nn::hid::debug::AbstractedVirtualPadId& virtualPadId) NN_NOEXCEPT;

    //!< 基本的な構成を持つ仮想コントローラーの全ての自動操作状態を解除します。
    ::nn::Result UnsetAllAutoPilotVirtualPadState() NN_NOEXCEPT;

    //!< JoyXpad を非アクティブ化します。
    nn::Result DeactivateJoyXpad(nn::hid::JoyXpadId xpadId) NN_NOEXCEPT;

    //!< Gesture を非アクティブ化します。
    ::nn::Result DeactivateGesture() NN_NOEXCEPT;

    //!< ホームボタンを非アクティブ化します。
    ::nn::Result DeactivateHomeButton() NN_NOEXCEPT;

    //!< ホームボタンに自動操作状態を設定します。
    ::nn::Result SetHomeButtonAutoPilotState(
        ::nn::hid::debug::HomeButtonAutoPilotState value) NN_NOEXCEPT;

    //!< ホームボタンの自動操作状態を解除します。
    ::nn::Result UnsetHomeButtonAutoPilotState() NN_NOEXCEPT;

    //!< スリープボタンを非アクティブ化します。
    ::nn::Result DeactivateSleepButton() NN_NOEXCEPT;

    //!< スリープボタンに自動操作状態を設定します。
    ::nn::Result SetSleepButtonAutoPilotState(
        ::nn::hid::debug::SleepButtonAutoPilotState value) NN_NOEXCEPT;

    //!< スリープボタンの自動操作状態を解除します。
    ::nn::Result UnsetSleepButtonAutoPilotState() NN_NOEXCEPT;

    //!< 撮影ボタンを非アクティブ化します。
    ::nn::Result DeactivateCaptureButton() NN_NOEXCEPT;

    //!< 撮影ボタンに自動操作状態を設定します。
    ::nn::Result SetCaptureButtonAutoPilotState(
        ::nn::hid::debug::CaptureButtonAutoPilotState value) NN_NOEXCEPT;

    //!< 撮影ボタンの自動操作状態を解除します。
    ::nn::Result UnsetCaptureButtonAutoPilotState() NN_NOEXCEPT;

    //!< 加速度センサーのキャリブレーション値のシフト量を設定します。
    ::nn::Result SetShiftAccelerometerCalibrationValue(const SixAxisSensorHandle& handle,
                                                       const ::nn::applet::AppletResourceUserId& aruid,
                                                       float shiftOrigin,
                                                       float shiftSensitivity) NN_NOEXCEPT;

    //!< 加速度センサーのキャリブレーション値のシフト量を取得します。
    ::nn::Result GetShiftAccelerometerCalibrationValue(nn::sf::Out<float> outShiftOrigin,
                                                       nn::sf::Out<float> outShiftSensitivity,
                                                       const SixAxisSensorHandle& handle,
                                                       const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    //!< ジャイロセンサーのキャリブレーション値のシフト量を設定します。
    ::nn::Result SetShiftGyroscopeCalibrationValue(const SixAxisSensorHandle& handle,
                                                   const ::nn::applet::AppletResourceUserId& aruid,
                                                   float shiftOrigin,
                                                   float shiftSensitivity) NN_NOEXCEPT;

    //!< ジャイロセンサーのキャリブレーション値のシフト量を取得します。
    ::nn::Result GetShiftGyroscopeCalibrationValue(nn::sf::Out<float> outShiftOrigin,
                                                   nn::sf::Out<float> outShiftSensitivity,
                                                   const SixAxisSensorHandle& handle,
                                                   const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor を非アクティブ化します。
    ::nn::Result DeactivateConsoleSixAxisSensor() NN_NOEXCEPT;

    //!< サンプリング間隔を返します。
    ::nn::Result GetConsoleSixAxisSensorSamplingFrequency(::nn::sf::Out<::std::int64_t> outValue,
                                                          ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< SevenSixAxisSensor を非アクティブ化します。
    ::nn::Result DeactivateSevenSixAxisSensor() NN_NOEXCEPT;

    //!< ConsoleSixAxisSensor の生値を取得します。
    ::nn::Result GetConsoleSixAxisSensorCountStates(
        nn::sf::Out<std::int32_t> outCount,
        const nn::sf::OutArray<nn::hid::debug::ConsoleSixAxisSensorCountState>& outStates,
        nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ファームウェア更新処理を有効にします。
    ::nn::Result ActivateFirmwareUpdate() NN_NOEXCEPT;

    //!< ファームウェア更新処理を無効にします。
    ::nn::Result DeactivateFirmwareUpdate() NN_NOEXCEPT;

    //!< 指定されたコントローラーについてファームウェア更新を開始します。
    ::nn::Result StartFirmwareUpdate(::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< 指定されたコントローラーについて、デバッグ/巻き戻し用データを使用してファームウェア更新を開始します。
    ::nn::Result StartFirmwareUpdateForRevert(::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< 指定されたコントローラーの Firmware Version を取得します
    ::nn::Result GetFirmwareVersion(::nn::sf::Out<debug::FirmwareVersion> outValue,
                                    ::nn::hid::NpadIdType id,
                                    ::nn::hid::system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT;

    //!< 指定されたコントローラーの、デバッグ/巻き戻し用データの更新先 Firmware Version を取得します
    ::nn::Result GetAvailableFirmwareVersionForRevert(::nn::sf::Out<debug::FirmwareVersion> outValue,
                                                      ::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< 指定されたコントローラーの更新先 Firmware Version を取得します
    ::nn::Result GetDestinationFirmwareVersion(::nn::sf::Out<debug::FirmwareVersion> outValue,
                                               ::nn::hid::NpadIdType id,
                                               ::nn::hid::system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT;

    //!< ファームウェア更新の経過状況を取得します。
    ::nn::Result GetFirmwareUpdateStage(::nn::sf::Out<::std::int64_t> outStage, ::nn::sf::Out<::std::int64_t> outProgress) NN_NOEXCEPT;

    //!< ファームウェア更新中のデバイスかどうか確認します。
    ::nn::Result IsFirmwareUpdatingDevice(
        ::nn::sf::Out<bool> outIsUpdating,
        ::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< 指定されたコントローラーに対して、チップ単位のファームウェア更新を行います。
    ::nn::Result StartFirmwareUpdateIndividual(
        ::nn::sf::Out<::nn::hid::system::FirmwareUpdateDeviceHandle> outHandle,
        ::nn::hid::system::UniquePadId id,
        std::int64_t target,
        ::nn::sf::NativeHandle&& imageHandle,
        std::uint64_t imageSize) NN_NOEXCEPT;

    //!< デバッグ/巻き戻し用ファームウェア情報のキャッシュを破棄します。
    ::nn::Result DiscardFirmwareInfoCacheForRevert() NN_NOEXCEPT;

    //!< 指定されたコントローラーの色を更新します
    ::nn::Result UpdateControllerColor(::nn::util::Color4u8Type mainColor, ::nn::util::Color4u8Type subColor, ::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< コントローラーのデザイン情報を更新する
    ::nn::Result UpdateDesignInfo(::nn::util::Color4u8Type mainColor,
                                  ::nn::util::Color4u8Type subColor,
                                  ::nn::util::Color4u8Type thirdColor,
                                  ::nn::util::Color4u8Type forthColor,
                                  uint32_t variation,
                                  ::nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    //!< UniquePad の入力状態を取得します。
    ::nn::Result GetUniquePadDriverState(::nn::sf::Out<PadDriverState> outState,
                                         const system::UniquePadId& id) NN_NOEXCEPT;

    //!< UniquePad の 6軸センサーの状態を取得します。
    ::nn::Result GetSixAxisSensorDriverStates(::nn::sf::Out<::std::int64_t> outCount,
                                              ::nn::sf::OutArray<SixAxisSensorDriverState> outStates,
                                              const system::UniquePadId& id) NN_NOEXCEPT;

    //!< パケットの受信履歴を取得します。
    ::nn::Result GetRxPacketHistory(::nn::sf::Out<RxPacketHistory> outValue, const system::UniquePadId& id) NN_NOEXCEPT;

    //!<  シリアルフラッシュへの操作が完了した際に通知するイベントをバインドします。
    ::nn::Result AcquireSerialFlashEventHandle(
        ::nn::sf::Out<::nn::sf::NativeHandle> outValue,
        const system::UniquePadId& id) NN_NOEXCEPT;

    //!<  シリアルフラッシュの任意のアドレスから読み込みを行います
    ::nn::Result ReadSerialFlash(
        uint32_t address,
        ::nn::sf::NativeHandle handle,
        uint64_t size,
        const system::UniquePadId& id) NN_NOEXCEPT;

    //!<  シリアルフラッシュの任意のアドレスへの書き込みを行います
    ::nn::Result WriteSerialFlash(
        uint32_t address,
        ::nn::sf::NativeHandle handle,
        uint64_t bufferSize,
        uint64_t writeSize,
        const system::UniquePadId& id) NN_NOEXCEPT;

    //!<  シリアルフラッシュへの操作結果を取得します
    ::nn::Result GetSerialFlashResult(const system::UniquePadId& id) NN_NOEXCEPT;

    //!< InputDetector を非アクティブ化します。
    ::nn::Result DeactivateInputDetector() NN_NOEXCEPT;

    //!< USB コントローラを接続します。
    ::nn::Result ConnectUsbPadsAsync() NN_NOEXCEPT;

    //!< USB コントローラを切断します。
    ::nn::Result DisconnectUsbPadsAsync() NN_NOEXCEPT;

    //!< ペアリング情報を手動で登録します。
    ::nn::Result AddRegisteredDevice(nn::hid::system::RegisteredDevice device) NN_NOEXCEPT;

    //!< 全ての接続済み AbstractedPad のハンドルを取得します。
    ::nn::Result GetAbstractedPadHandles(
        ::nn::sf::Out<int>pOutCount,
        ::nn::sf::OutArray<debug::AbstractedPadHandle> pOutHandles) NN_NOEXCEPT;

    //!< AbstractedPad の状態を取得します。
    ::nn::Result GetAbstractedPadState(
        ::nn::sf::Out<debug::AbstractedPadState> pOutState,
        ::nn::hid::debug::AbstractedPadHandle handle) NN_NOEXCEPT;

    //!< 全ての接続済み AbstractedPad のハンドルと状態を取得します。
    ::nn::Result GetAbstractedPadsState(
        ::nn::sf::Out<std::int32_t> pOutCount,
        const ::nn::sf::OutArray<::nn::hid::debug::AbstractedPadHandle>& pOutHandles,
        const ::nn::sf::OutArray<::nn::hid::debug::AbstractedPadState>& pOutStates) NN_NOEXCEPT;
};

//!< hid デバッグ機能サービスのプロクシを作成します。
::nn::Result CreateHidDebugServerProxy(
    ::nn::sf::SharedPointer<IHidDebugServer>* pOutValue) NN_NOEXCEPT;

}}} // namespace nn::hid::detail
