﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ジャイロセンサーの処理に特化した API の宣言
 */

#pragma once

#include <nn/util/util_MathTypes.h>
#include <nn/xcd/xcd_Input.h>

#include "hid_SixAxisSensorAppletSetting.h"

namespace nn { namespace hid { namespace detail {

/**
 * @brief       3軸をもつセンサーの値を表す構造体です。
 */
struct ZeroDriftParameters
{
    float   radius; //!< ドリフトカウント許容半径 (1 = 360deg/sec)
    float   power;  //!< ドリフト修正パワー
    int16_t count;  //!< ドリフト修正カウント
};

const ZeroDriftParameters ZeroDrift[] =
{ //  radius  power   count
    { 0.01000f, 0.040f, 200 },      // Loose
    { 0.00750f, 0.020f, 200 },      // Standard
    { 0.00250f, 0.010f, 200 },      // Tight
    { 0.00000f, 0.000f, 200 },      // None
};

//!< ジャイロセンサーの処理を担うクラス
class Gyroscope final
{
    NN_DISALLOW_COPY(Gyroscope);
    NN_DISALLOW_MOVE(Gyroscope);

private:
    nn::util::Float3            m_ZeroCount;             //!< ゼロ点カウント[count]
    nn::util::Float3            m_SensitivityCount;      //!< 感度カウント[count]
    nn::util::Float3            m_Scale;                 //!< 角速度値のスケーリングパラメータ。[360dps を 1.0 とする単位系]

#ifdef NERD_USE_DENOISED_ROTATION_SPEED
    float                       m_ZeroCountTotalWeight;
    float                       m_ZeroCountTotalWeightCap;
    nn::util::Float3            m_DenoisedRotationSpeed;
#endif
    nn::util::Float3            m_LastAngularVelocityCount; //!< 前回算出した角速度カウント[G]
    nn::util::Float3            m_StableCountSum;         //!< 連続して静止状態となったカウント値の総和[count]
    nn::xcd::SensorState        m_StableCountTimes;       //!< 連続して静止状態となった回数

    nn::util::Float3            m_AngularVelocityMax;     //!< 角速度カウント[count] の区間最大値
    nn::util::Float3            m_AngularVelocityMin;     //!< 角速度カウント[count] の区間最小値

    bool                        m_IsAtRest;               //!< 静止状態にあるか

    const GyroscopeSetting*     m_pGyroscopeSetting;      //!< 設定値

public:
    Gyroscope()  NN_NOEXCEPT;
    ~Gyroscope() NN_NOEXCEPT {};

    //!< ジャイロセンサーのパラメータを設定します。
    void SetGyroscopeSetting(const GyroscopeSetting* const pSetting) NN_NOEXCEPT;

    //!< 角速度を取得します。
    void GetState(nn::util::Float3* pOutState,
                  const nn::util::Float3& count,
                  const nn::hid::SixAxisSensorAttributeSet& attributes) NN_NOEXCEPT;

    //!< 6軸センサーのキャリブレーション値を設定します。
    void SetCalibrationValue( const nn::xcd::SensorState& gyroscopeOrigin, const nn::xcd::SensorState& gyroscopeSensitivity ) NN_NOEXCEPT;

    //!< 静止状態にあるかを返します。
    bool IsAtRest() const NN_NOEXCEPT;
    void ResetIsAtRest() NN_NOEXCEPT;

    //!< 現在の姿勢を角速度値を用いて更新します。
    void UpdateDirection( nn::hid::DirectionState* pOutDirection, const nn::util::Float3& angularVelocity, const float& samplingInterval ) const NN_NOEXCEPT;

private:

    //!< 角速度のスケーリングパラメータを設定します。
    void SetScaleValue() NN_NOEXCEPT;

    //!< ある軸における静止判定を行いパラメータを更新します。
    void UpdateGyroParameter(int32_t* pStableCountTimes,
                             float* pStableCountSum,
                             float* pAngularVelocityMax,
                             float* pAngularVelocityMin,
                             const float currentCount,
                             const float previousCount,
                             //float* pDenoisedRotationSpeed,
                             const float& scale) NN_NOEXCEPT;

    //!< 角速度カウントを内部パラメータを用いて更新します。
    void ReviseAngularVelocityCountByStability(nn::util::Float3* pOut, const nn::util::Float3& count) const NN_NOEXCEPT;

    //!< ゼロ点を入力角速度カウントに近づけます。
    void UpdateZeroCount(const nn::util::Float3& count) NN_NOEXCEPT;

    //!< 内部的に保持している前回サンプルの角速度カウント値を更新します。
    void UpdateLastAngularVelocityCount(const nn::util::Float3& count) NN_NOEXCEPT;

    //!< 角速度値[1.0=360dps] にカウント値を変換します。
    void ConvertToAngularVelocity(nn::util::Float3* pOut, const nn::util::Float3& count) const NN_NOEXCEPT;

    //!< ゼロ点遊びにより角速度を更新します。
    void ReviseAngularVelocityByZeroPlay( nn::util::Float3* pOut ) const NN_NOEXCEPT;

    //!< 6軸センサーのキャリブレーション値をシフトします。
    void ShiftCalibrationValue(const float& origin, const float& sensitivity) NN_NOEXCEPT;

};

}}} // namespace nn::hid::detail
