﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_Gesture.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_MathTypes.h>

namespace nn { namespace hid { namespace detail {

//!< ジェスチャ認識器のバージョン
const int32_t GestureRecognizerVersion = 1;

//!< ジェスチャを構成するタッチの集合を表す構造体です。
struct GestureTouchSet final
{
    int32_t count;                              //!< タッチの数
    NN_PADDING4;
    TouchState touches[GesturePointCountMax];   //!< タッチ
};

//!< ジェスチャ認識器の制御フラグ定義です。
struct GestureMachineFlag final
{
    typedef ::nn::util::BitFlagSet<32, GestureMachineFlag>::Flag<0>
            IsNewTouch;     //!< 持ち越しを含まない新規のタッチか否か
    typedef ::nn::util::BitFlagSet<32, GestureMachineFlag>::Flag<1>
            IsDoubleTap;    //!< ダブルタップか否か
    typedef ::nn::util::BitFlagSet<32, GestureMachineFlag>::Flag<2>
            AcceptsAction;  //!< タッチの動きを受け付けるか否か
    typedef ::nn::util::BitFlagSet<32, GestureMachineFlag>::Flag<3>
            AcceptsPan;     //!< パン操作を受け付けるか否か
    typedef ::nn::util::BitFlagSet<32, GestureMachineFlag>::Flag<4>
            AcceptsPinch;   //!< ピンチ操作を受け付けるか否か
    typedef ::nn::util::BitFlagSet<32, GestureMachineFlag>::Flag<5>
            AcceptsRotate;  //!< 回転操作を受け付けるか否か
};

//!< ジェスチャ認識器の制御フラグ集合を扱う型です。
typedef ::nn::util::BitFlagSet<32, GestureMachineFlag> GestureMachineFlagSet;

//!< ジェスチャ認識器のステートマシンの状態を表す構造体です。
struct GestureMachineState final
{
    int64_t eventNumber;                //!< イベント番号
    int64_t contextNumber;              //!< コンテキスト番号
    int32_t position;                   //!< オートマトン上の位置
    GestureMachineFlagSet flags;        //!< 制御フラグ
    ::nn::TimeSpanType timeStamp;       //!< タイムスタンプ
    ::nn::TimeSpanType deltaTime;       //!< 時間差分
    ::nn::TimeSpanType touchTimer;      //!< タッチ操作用タイマ
    ::nn::TimeSpanType doubleTapTimer;  //!< ダブルタップ操作用タイマ
    GestureTouchSet touch;              //!< タッチ操作を構成するタッチ
    GestureTouchSet doubleTap;          //!< ダブルタップ操作を構成するタッチ
    float distance;                     //!< 最後に観測されたタッチ間距離
    float angle;                        //!< 最後に観測されたタッチ間角度
    NN_PADDING4;
    int32_t direction;                  //!< ジェスチャの方向
    int32_t x;                          //!< ジェスチャの x 座標
    int32_t y;                          //!< ジェスチャの y 座標
    int32_t deltaX;                     //!< ジェスチャの x 座標の移動差分
    int32_t deltaY;                     //!< ジェスチャの y 座標の移動差分
    ::nn::util::Float2 velocity;        //!< ジェスチャの速度 ([/s])
    float scale;                        //!< ジェスチャのスケール
    float rotationAngle;                //!< ジェスチャの回転の度数
};

//!< ジェスチャ認識を扱うクラスです。
class GestureRecognizer final
{
    NN_DISALLOW_COPY(GestureRecognizer);
    NN_DISALLOW_MOVE(GestureRecognizer);

private:
    //!< ジェスチャの状態
    GestureState m_GestureState;

    //!< ステートマシンの状態
    GestureMachineState m_MachineState;

public:
    GestureRecognizer() NN_NOEXCEPT;

    //!< ジェスチャの状態を返します。
    const GestureState& GetState() const NN_NOEXCEPT;

    //!< 内部状態をリセットします。
    void Reset() NN_NOEXCEPT;

    //!< 内部状態を指定されたタッチで更新します。
    void Update(const TouchState touches[], int32_t count) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
